/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.tree;

import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JOptionPane;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.map.ClassMap;

/**
 * Редакторы для значений различных типов по умолчанию.
 * 
 * <p>
 * Поддерживается редактирование для следующих типов
 * <ul>
 * <li>int
 * <li>short
 * <li>byte
 * <li>long
 * <li>float
 * <li>double
 * <li>boolean
 * <li>String
 * </ul>
 * 
 * Возможно расширение этого списка
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class TreeTableNodeValueEditorDef extends TreeTableNodeValueEditor {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TreeTableNodeValueEditorDef.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(TreeTableNodeValueEditorDef.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(TreeTableNodeValueEditorDef.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(TreeTableNodeValueEditorDef.class.getName(), method, result);
    }
    //</editor-fold>
    
    public static class ConvertTextFieldEditor<T> extends TextFieldEditor {
        //<editor-fold defaultstate="collapsed" desc="parser">
        protected Func1<T,String> parser;
        
        public Func1<T, String> getParser() {
            return parser;
        }
        
        public void setParser(Func1<T, String> parser) {
            this.parser = parser;
        }
        
        @Override
        public Object getCellEditorValue() {
            try{
                Object ostr = super.getCellEditorValue();

                Func1<T,String> prsr = parser;
                if( prsr==null )throw new IllegalStateException("parser not set");

                if( ostr==null ){
                    throw new IllegalStateException("return null string");
                }

                return prsr.apply(ostr.toString());
            }catch( Throwable err ){
                JOptionPane.showMessageDialog(null, err.getMessage(), "parse error", JOptionPane.ERROR_MESSAGE);
                return defValue;
            }
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="serializer">
        protected Func1<String,T> serializer;
        
        public Func1<String, T> getSerializer() {
            return serializer;
        }
        
        public void setSerializer(Func1<String, T> serializer) {
            this.serializer = serializer;
        }
        
        @Override
        public void startEditing(Object value, Object context) {
            try{
                Func1<String,T> srlz = getSerializer();
                if( srlz==null ){
                    throw new IllegalStateException("serializer not set");
                }
                
                String str = srlz.apply((T)value);
                
                super.startEditing(str, context);
            }catch( Throwable err ){
                JOptionPane.showMessageDialog(null, err.getMessage(), "serilize error", JOptionPane.ERROR_MESSAGE);
            }
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="defValue">
        protected T defValue;
        
        public T getDefValue() {
            return defValue;
        }
        
        public void setDefValue(T defValue) {
            this.defValue = defValue;
        }
        //</editor-fold>
        
        public ConvertTextFieldEditor( Func1<T,String> parser, Func1<String,T> serializer, T defValue ){
            this.parser = parser;
            this.serializer = serializer;
            this.defValue = defValue;
        }
    }
    
    @Override
    protected ClassMap<Editor> createTypeEditors() {
        ClassMap<Editor> editors = super.createTypeEditors();
        
        //<editor-fold defaultstate="collapsed" desc="int">
        editors.put(Integer.class, new ConvertTextFieldEditor<Integer>(
            //str -> "null".equals(str) ? null : (Integer)Integer.parseInt(str),
            new Func1<Integer, String>() {
                @Override
                public Integer apply(String str) {
                    return "null".equals(str) ? null : (Integer)Integer.parseInt(str);
                }
            },
            //val -> val!=null ? val.toString() : "null",
            new Func1<String, Integer>() {
                @Override
                public String apply(Integer val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        
        editors.put(int.class, new ConvertTextFieldEditor<Integer>(
            new Func1<Integer, String>() {
                @Override
                public Integer apply(String str) {
                    return Integer.parseInt(str);
            }},
            //val -> val!=null ? val.toString() : "null",
            new Func1<String, Integer>() {
                @Override
                public String apply(Integer val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="short">
        editors.put(short.class, new ConvertTextFieldEditor<Short>(
            //str -> "null".equals(str) ? null : (Integer)Integer.parseInt(str),
            new Func1<Short, String>() {
                @Override
                public Short apply(String str) {
                    return "null".equals(str) ? null : Short.parseShort(str);
                }
            },
            //val -> val!=null ? val.toString() : "null",
            new Func1<String, Short>() {
                @Override
                public String apply(Short val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        
        editors.put(Short.class, new ConvertTextFieldEditor<Short>(
            new Func1<Short, String>() {
                @Override
                public Short apply(String str) {
                    return (Short)Short.parseShort(str);
            }},
            //val -> val!=null ? val.toString() : "null",
            new Func1<String, Short>() {
                @Override
                public String apply(Short val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="byte">
        editors.put(byte.class, new ConvertTextFieldEditor<Byte>(
            new Func1<Byte, String>() {
                @Override
                public Byte apply(String str) {
                    return "null".equals(str) ? null : Byte.parseByte(str);
                }
            },
            new Func1<String, Byte>() {
                @Override
                public String apply(Byte val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        
        editors.put(Byte.class, new ConvertTextFieldEditor<Byte>(
            new Func1<Byte, String>() {
                @Override
                public Byte apply(String str) {
                    return (Byte)Byte.parseByte(str);
            }},
            new Func1<String, Byte>() {
                @Override
                public String apply(Byte val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );        
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="long">
        editors.put(long.class, new ConvertTextFieldEditor<Long>(
            new Func1<Long, String>() {
                @Override
                public Long apply(String str) {
                    return "null".equals(str) ? null : Long.parseLong(str);
                }
            },
            new Func1<String, Long>() {
                @Override
                public String apply(Long val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        
        editors.put(Long.class, new ConvertTextFieldEditor<Long>(
            new Func1<Long, String>() {
                @Override
                public Long apply(String str) {
                    return (Long)Long.parseLong(str);
            }},
            new Func1<String, Long>() {
                @Override
                public String apply(Long val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="float">
        editors.put(float.class, new ConvertTextFieldEditor<Float>(
            new Func1<Float, String>() {
                @Override
                public Float apply(String str) {
                    return "null".equals(str) ? null : Float.parseFloat(str);
                }
            },
            new Func1<String, Float>() {
                @Override
                public String apply(Float val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        
        editors.put(Float.class, new ConvertTextFieldEditor<Float>(
            new Func1<Float, String>() {
                @Override
                public Float apply(String str) {
                    return (Float)Float.parseFloat(str);
            }},
            new Func1<String, Float>() {
                @Override
                public String apply(Float val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="double">
        editors.put(double.class, new ConvertTextFieldEditor<Double>(
            new Func1<Double, String>() {
                @Override
                public Double apply(String str) {
                    return "null".equals(str) ? null : Double.parseDouble(str);
                }
            },
            new Func1<String, Double>() {
                @Override
                public String apply(Double val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        
        editors.put(Double.class, new ConvertTextFieldEditor<Double>(
            new Func1<Double, String>() {
                @Override
                public Double apply(String str) {
                    return (Double)Double.parseDouble(str);
            }},
            new Func1<String, Double>() {
                @Override
                public String apply(Double val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="boolean">
        editors.put(boolean.class, new ConvertTextFieldEditor<Boolean>(
            new Func1<Boolean, String>() {
                @Override
                public Boolean apply(String str) {
                    return "null".equals(str) ? null : (boolean)( str.trim().equalsIgnoreCase("true") );
                }
            },
            new Func1<String, Boolean>() {
                @Override
                public String apply(Boolean val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        
        editors.put(Boolean.class, new ConvertTextFieldEditor<Boolean>(
            new Func1<Boolean, String>() {
                @Override
                public Boolean apply(String str) {
                    return "null".equals(str) ? null : (Boolean)( str.trim().equalsIgnoreCase("true") );
            }},
            new Func1<String, Boolean>() {
                @Override
                public String apply(Boolean val) {
                    return val!=null ? val.toString() : "null";
                }
            },
            null
        ) );
        //</editor-fold>
        
        return editors;
    }
}
