/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.gui.swing.tree.ob;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Icon;
import javax.swing.JMenuItem;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.tree.TreeTableNodeBasic;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormat;

/**
 * Создание контекста для браузера объектов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @param <T> Класс данных узла дерева, для которого создается контекст
 */
public class ContextBuilder<T> implements Context<T>, AutoCloseable {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(ContextBuilder.class.getName());

    private static Level logLevel(){ return logger.getLevel(); }
    
    private static boolean isLogSevere(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null ? true : logLevel.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level logLevel = logger.getLevel(); 
        return logLevel==null  ? true : logLevel.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINER.intValue();
    }
    
    private static boolean isLogFinest(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINEST.intValue();
    }

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(ContextBuilder.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(ContextBuilder.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(ContextBuilder.class.getName(), method, result);
    }
    //</editor-fold>
    
    private boolean applied;
    private Class<T> cls;
    private ObjectBrowser baseObjectBrowser;
    private final ConcurrentLinkedQueue<Runnable> applyQueue;
    private final CloseableSet cset;

    /**
     * Конструктор
     * @param ob браузер объектов
     * @param cls Класс данных узла дерева, для которого создается контекст
     */
    public ContextBuilder( ObjectBrowser ob, Class<T> cls) {
        if( cls==null )throw new IllegalArgumentException("cls == null");
        if( ob==null )throw new IllegalArgumentException("ob == null");
        this.applied = false;
        this.cls = cls;
        this.baseObjectBrowser = ob;
        applyQueue = new ConcurrentLinkedQueue<>();
        cset = new CloseableSet();
    }
    
    //<editor-fold defaultstate="collapsed" desc="contextClass : Class<T>">
    /**
     * Возвращает тип данных для которых создается контекст поведения
     * @return тип данных
     */
    public Class<T> getContextClass(){ return cls; }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="naming()">
    /**
     * Указывает функцию именования объектов
     * @param nameFn функция именования объектов
     * @return self ссылка
     */
    @Override
    public synchronized Context<T> naming(final Func1<String, T> nameFn) {
        if( nameFn==null )throw new IllegalArgumentException("nameFn == null");
        if( applied )throw new IllegalStateException("already applied");
        if( baseObjectBrowser==null )throw new IllegalArgumentException("BaseObjectBrowser(ob field) not set");
        final WeakReference<ObjectBrowser> fob = new WeakReference<>(baseObjectBrowser);
        applyQueue.add(new Runnable() {
            @Override
            public void run() {
                ObjectBrowser bob = fob.get();
                if( bob==null )throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                
                synchronized(bob.getNamingMap()){
                    bob.getNamingMap().put(cls, (Func1)nameFn);
                }
                
                cset.add(new Runnable() {
                    @Override
                    public void run() {
                        ObjectBrowser bob = fob.get();
                        if( bob==null )return;
                        
                        synchronized(bob.getNamingMap()){
                            Object existsVal = bob.getNamingMap().get(cls);
                            if( existsVal==nameFn ){
                                bob.getNamingMap().remove(cls);
                            }
                        }
                    }
                });
            }
        });
        return this;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="follower()">
    /**
     * Указывает функцию "следования" к дочерним узлам
     * @param <A> Тип дочерних узлов
     * @param followerFn функция "следования"
     * @param followable функция проверки наличия дочерних узлов
     * @return self ссылка
     */
    @Override
    public synchronized <A> Context<T> follower(final NodesExtracter<T, A> followerFn, final Followable<T> followable) {
        if( followerFn==null )throw new IllegalArgumentException("followerFn == null");
        if( followable==null )throw new IllegalArgumentException("followable == null");
        if( applied )throw new IllegalStateException("already applied");
        
        final WeakReference<ObjectBrowser> fob = new WeakReference<>(baseObjectBrowser);
        applyQueue.add(new Runnable() {
            @Override
            public void run() {
                ObjectBrowser bob = fob.get();
                if( bob==null )throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                
                FollowableExtracter fe = new FollowableExtracter(followerFn, followable);
                synchronized(bob.getFollowerMap()){
                    bob.getFollowerMap().add(cls, fe);
                }
                
                final WeakReference<FollowableExtracter>
                    ffe = new WeakReference<>( fe );
                
                cset.add(new Runnable() {
                    @Override
                    public void run() {
                        ObjectBrowser bob = fob.get();
                        if( bob==null )return;
                        
                        FollowableExtracter fe = ffe.get();
                        if( fe==null )return;
                        
                        synchronized(bob.getFollowerMap()){
                            bob.getFollowerMap().remove(cls, fe);
                        }
                    }
                });
            }
        });
        
        return this;
    }
    
    /**
     * Указывает функция следования
     * @param <A> тип дочерних
     * @param followerFn функция следования
     * @return self ссылка
     */
    @Override
    public <A> Context<T> follower( final NodesExtracter<T,A> followerFn ){
        if( applied )throw new IllegalStateException("already applied");
        if( followerFn==null )throw new IllegalArgumentException("followerFn == null");
        
        final WeakReference<ObjectBrowser> fob = new WeakReference<>(baseObjectBrowser);
        applyQueue.add(new Runnable() {
            @Override
            public void run() {
                ObjectBrowser bob = fob.get();
                if( bob==null )throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                
                synchronized(bob.getFollowerMap()){
                    bob.getFollowerMap().add(cls, followerFn);
                }
                
                final WeakReference<NodesExtracter> ffe = new WeakReference<>( (NodesExtracter)followerFn );
                
                cset.add(new Runnable() {
                    @Override
                    public void run() {
                        ObjectBrowser bob = fob.get();
                        if( bob==null )return;
                        
                        NodesExtracter fe = ffe.get();
                        if( fe==null )return;
                        
                        synchronized(bob.getFollowerMap()){
                            bob.getFollowerMap().remove(cls, fe);
                        }
                    }
                });
            }
        });
        
        return this;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="followable()">
    /**
     * Указыает функцию проверки наличия дочерних не извелеченных узлов
     * @param followableFn функция проверки
     * @return true - есть дочерние узлы
     */
    public Context<T> followable( final Func1<Boolean,T> followableFn ){
        if( followableFn==null )throw new IllegalArgumentException("followableFn == null");
        if( applied )throw new IllegalStateException("already applied");
        
        final WeakReference<ObjectBrowser> fob = new WeakReference<>(baseObjectBrowser);
        applyQueue.add(new Runnable() {
            @Override
            public void run() {
                ObjectBrowser bob = fob.get();
                if( bob==null )throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                
                synchronized(bob.getFollowableMap()){
                    bob.getFollowableMap().put(cls, (Func1)followableFn);
                }
                
                final WeakReference<Func1> ffe = new WeakReference<>( (Func1)followableFn );
                
                cset.add(new Runnable() {
                    @Override
                    public void run() {
                        ObjectBrowser bob = fob.get();
                        if( bob==null )return;
                        
                        Func1 fe = ffe.get();
                        if( fe==null )return;
                        
                        synchronized(bob.getFollowableMap()){
                            Object v = bob.getFollowableMap().get(cls);
                            if( v==fe )bob.getFollowableMap().remove(cls);
                        }
                    }
                });
            }
        });
        
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="format()">
    /**
     * Указывает функция форматирования
     * @param fmtfn функция форматирования
     * @return self ссылка
     */
    @Override
    public synchronized Context<T> format( final Func1<TreeTableNodeFormat,T> fmtfn ){
        if( fmtfn==null )throw new IllegalArgumentException("fmtfn == null");
        if( applied )throw new IllegalStateException("already applied");
        
        final WeakReference<ObjectBrowser> fob = new WeakReference<>(baseObjectBrowser);
        applyQueue.add(new Runnable() {
            @Override
            public void run() {
                ObjectBrowser bob = fob.get();
                if( bob==null )throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                
                synchronized(bob.getFormatMap()){
                    bob.getFormatMap().put(cls, (Func1)fmtfn);
                }
                
                final WeakReference<Func1> ffe = new WeakReference<>( (Func1)fmtfn );
                
                cset.add(new Runnable() {
                    @Override
                    public void run() {
                        ObjectBrowser bob = fob.get();
                        if( bob==null )return;
                        
                        Func1 fe = ffe.get();
                        if( fe==null )return;
                        
                        synchronized(bob.getFormatMap()){
                            Object v = bob.getFormatMap().get(cls);
                            if( v==fe )bob.getFormatMap().remove(cls);
                        }
                    }
                });
            }
        });
        
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="icon()">
    /**
     * Указывает функцию иконки для отображения данного типа узла
     * @param iconFun функция сопостовления иконки
     * @return self ссылка
     */
    @Override
    public synchronized Context<T> icon( final Func1<Icon,T> iconFun ){
        //if( iconFun==null )throw new IllegalArgumentException("iconFun == null");
        if( applied )throw new IllegalStateException("already applied");
        
        final WeakReference<ObjectBrowser> fob = new WeakReference<>(baseObjectBrowser);
        applyQueue.add(new Runnable() {
            @Override
            public void run() {
                ObjectBrowser bob = fob.get();
                if( bob==null )throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                
                synchronized(bob.getTypeIconMap()){
                    if( iconFun!=null ){
                        bob.getTypeIconMap().put(cls, (Func1)iconFun);
                    }else{
                        bob.getTypeIconMap().remove(cls);
                    }
                }
                
                if( iconFun!=null ){
                    final WeakReference<Func1> ffe = new WeakReference<>( (Func1)iconFun );
                    cset.add(new Runnable() {
                        @Override
                        public void run() {
                            ObjectBrowser bob = fob.get();
                            if( bob==null )return;
                            
                            Func1 fe = ffe.get();
                            if( fe==null )return;
                            
                            synchronized(bob.getTypeIconMap()){
                                Object v = bob.getTypeIconMap().get(cls);
                                if( v==fe )bob.getTypeIconMap().remove(cls);
                            }
                        }
                    });
                }
            }
        });
        
        return this;
    }
    
    /**
     * Указывает иконку для данного типа узла
     * @param ico иконка
     * @return self ссылка
     */
    @Override
    public synchronized Context<T> icon( final Icon ico ){
        if( ico==null ){
            return icon((Func1<Icon, T>)null);
        }
        
        return icon(new Func1<Icon, T>() {
            @Override
            public Icon apply(T arg) {
                return ico;
            }
        });
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="ContextMenuBuilder<T>">
    /**
     * Кнструирование контекстого меню
     * @param <T> тип данных узла дерева
     */
    public static class ContextMenuBuilder<T> implements ContextMenu<T>{
        private ContextBuilder<T> cb;
        private boolean contextMenuBuilder;
        
        //<editor-fold defaultstate="collapsed" desc="ContextMenuBuilder()">
        public ContextMenuBuilder(ContextBuilder<T> cb){
            if( cb==null )throw new IllegalArgumentException("cb == null");
            this.cb = cb;
            callList = new ArrayList<>();
            contextMenuBuilder = false;
        }
        
        public ContextMenuBuilder(ContextBuilder<T> cb, boolean ctxmbld){
            if( cb==null )throw new IllegalArgumentException("cb == null");
            this.cb = cb;
            callList = new ArrayList<>();
            contextMenuBuilder = ctxmbld;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="label : String">
        private String label;
        
        @Override
        public ContextMenu<T> label(String label) {
            this.label = label;
            return this;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="icon : Icon">
        private Icon icon;
        
        @Override
        public ContextMenu<T> icon(Icon icon) {
            this.icon = icon;
            return this;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="call()">
        private List<Func2<Object,TreeTableNodeBasic,T>> callList;
        
        @Override
        public ContextMenu<T> call(final Reciver<T> callfn) {
            if( callfn==null )throw new IllegalArgumentException("callfn == null");
            callList.add(new Func2<Object, TreeTableNodeBasic, T>() {
                @Override
                public Object apply(TreeTableNodeBasic arg1, T arg2) {
                    callfn.recive(arg2);
                    return null;
                }
            });
            return this;
        }
        
        @Override
        public ContextMenu<T> call(final Func2<Object,TreeTableNodeBasic,T> callfn) {
            if( callfn==null )throw new IllegalArgumentException("callfn == null");
            callList.add(callfn);
            return this;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="add()">
        @Override
        public Context<T> add() {
            final WeakReference<ObjectBrowser> fbob = new WeakReference<>( cb.baseObjectBrowser );
            
            final String flabel = label;
            final Icon ico = icon;
            
            final List<Func2<Object,TreeTableNodeBasic,T>> recivers = new ArrayList<>();
            recivers.addAll(callList);
            
            final WeakReference<ContextBuilder<T>> fcb = new WeakReference<>(cb);
            
            cb.applyQueue.add(new Runnable() {
                @Override
                public void run() {
                    ContextBuilder<T> cb = fcb.get();
                    if( cb==null ){
                        throw new IllegalStateException("apply fail : ContextBuilder not set");
                    }
                    
                    ObjectBrowser bob = fbob.get();
                    if( bob == null ){
                        throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                    }
                    
                    //<editor-fold defaultstate="collapsed" desc="context menu">
                    final NodesExtracter ne = new NodesExtracter() {
                        @Override
                        public Iterable extract(Object from) {
                            Func2 fn = new Func2() {
                                @Override
                                public Object apply(final Object tn, final Object tnData) {
                                    if( tn instanceof TreeTableNodeBasic ){
                                        JMenuItem mi = new JMenuItem(flabel);
                                        if( ico!=null )mi.setIcon(icon);
                                        
                                        mi.addActionListener(new ActionListener() {
                                            @Override
                                            public void actionPerformed(ActionEvent e) {
                                                for( Func2 rcv : recivers ){
                                                    if( rcv!=null ){
                                                        //rcv.recive(arg2);
                                                        rcv.apply(tn, tnData);
                                                    }
                                                }
                                            }
                                        });
                                        
                                        return mi;
                                    }
                                    return null;
                                }
                            };
                            return Arrays.asList(fn);
                        }
                    };
                    
                    final Class cbclass = cb.getContextClass();
                    final Func1<List<JMenuItem>,TreeTableNodeBasic>
                        ctxMenuBuilder = new Func1<List<JMenuItem>, TreeTableNodeBasic>() {
                            @Override
                            public List<JMenuItem> apply(final TreeTableNodeBasic ttnb) {
                                if( ttnb==null )return null;
                                
                                Object data = ttnb.getData();
                                if( data==null )return null;
                                
                                Class tdata = data.getClass();
                                if( !cbclass.isAssignableFrom(tdata) ){
                                    return null;
                                }
                                
                                JMenuItem mi = new JMenuItem(flabel);
                                if( ico!=null )mi.setIcon(icon);
                                mi.addActionListener(new ActionListener() {
                                    @Override
                                    public void actionPerformed(ActionEvent e) {
                                        for( Func2 rcv : recivers ){
                                            if( rcv!=null ){
                                                //rcv.recive(arg2);
                                                rcv.apply(ttnb, ttnb.getData());
                                            }
                                        }
                                    }
                                });
                                
                                ArrayList<JMenuItem> list = new ArrayList<>();
                                list.add(mi);
                                
                                return list;
                            }
                        };
                    
                    if( contextMenuBuilder ){
                        bob.getContextMenuBuilders().add(ctxMenuBuilder);
                    }else{
                        bob.getNodeContextMenu().add(cb.getContextClass(), ne);
                    }
                    
                    final Class ctxcls = cb.getContextClass();
                    cb.cset.add(new Runnable() {
                        @Override
                        public void run() {
                            ObjectBrowser bob = fbob.get();
                            if( bob == null )return;
                            
                            bob.getNodeContextMenu().remove(ctxcls, ne);
                            bob.getContextMenuBuilders().remove(ctxMenuBuilder);
                        }
                    });
                    //</editor-fold>
                }
            });
            
            Context ctx = cb;
            cb = null;
            return ctx;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="items()">
        @Override
        public Context<T> items( final Func2<Iterable<JMenuItem>,TreeTableNodeBasic,T> buildMenus ){
            if (buildMenus== null) {
                throw new IllegalArgumentException("buildMenus==null");
            }
            
            final WeakReference<ObjectBrowser> fbob = new WeakReference<>( cb.baseObjectBrowser );
            final WeakReference<ContextBuilder<T>> fcb = new WeakReference<>(cb);
            // final WeakReference<Func2<Iterable<JMenuItem>,TreeTableNodeBasic,T>> fBuildMenus
            //    = new WeakReference<>(buildMenus);
            
            cb.applyQueue.add(new Runnable()
            {
                @Override
                public void run() {
                    //Func2<Iterable<JMenuItem>,TreeTableNodeBasic,T>
                    //    buildMenus = fBuildMenus.get();
                    
                    //if( buildMenus==null )return;
                    
                    ContextBuilder<T> cb = fcb.get();
                    if( cb==null ){
                        throw new IllegalStateException("apply fail : ContextBuilder not set");
                    }
                    
                    ObjectBrowser bob = fbob.get();
                    if( bob == null ){
                        throw new IllegalStateException("apply fail : BaseObjectBrowser(ob field) not set");
                    }
                    
                    final NodesExtracter ne = new NodesExtracter() {
                        @Override
                        public Iterable extract(Object from) {
                            Func2 fn = new Func2() {
                                @Override
                                public Object apply(final Object tn, final Object tnData) {
                                    if( tn instanceof TreeTableNodeBasic ){
                                        Iterable<JMenuItem> miIts = buildMenus.apply((TreeTableNodeBasic)tn, (T)tnData);
                                        
                                        if( miIts!=null ){
                                            ArrayList<JMenuItem> res = new ArrayList<>();
                                            for( JMenuItem mi : miIts ){
                                                if( mi!=null ){
                                                    res.add(mi);
                                                }
                                            }
                                            return res;
                                        }
                                    }
                                    return null;
                                }
                            };
                            return Arrays.asList(fn);
                        }
                    };
                    
                    final Class cbclass = cb.getContextClass();
                    final Func1<List<JMenuItem>,TreeTableNodeBasic>
                        ctxMenuBuilder = new Func1<List<JMenuItem>, TreeTableNodeBasic>() {
                            @Override
                            public List<JMenuItem> apply(TreeTableNodeBasic ttnb) {
                                if( ttnb==null )return null;
                                
                                Object data = ttnb.getData();
                                if( data==null )return null;
                                
                                Class tdata = data.getClass();
                                if( !cbclass.isAssignableFrom(tdata) ){
                                    return null;
                                }
                                
                                Iterable<JMenuItem> itms = buildMenus.apply(ttnb, (T)data);
                                if( itms==null )return null;
                                
                                ArrayList<JMenuItem> list = new ArrayList<>();
                                for( JMenuItem mi : itms ){
                                    if( mi!=null )list.add(mi);
                                }
                                
                                return list;
                            }
                        };
                    
                    if( contextMenuBuilder ){
                        bob.getContextMenuBuilders().add(ctxMenuBuilder);
                    }else{
                        bob.getNodeContextMenu().add(cb.getContextClass(), ne);
                    }
                    
                    final Class ctxcls = cb.getContextClass();
                    cb.cset.add(new Runnable() {
                        @Override
                        public void run() {
                            ObjectBrowser bob = fbob.get();
                            if( bob == null )return;
                            
                            bob.getNodeContextMenu().remove(ctxcls, ne);
                            bob.getContextMenuBuilders().remove(ctxMenuBuilder);
                        }
                    });
                }
            });
            
            Context ctx = cb;
            cb = null;
            return ctx;
        }
        //</editor-fold>
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="menu()">
    /**
     * Конструирование контекстного меню
     * @return конструктор меню
     */
    @Override
    public ContextMenu<T> menu(){
        return new ContextMenuBuilder<>(this);
    }

    /**
     * Конструирование контекстного меню
     * @param matchSoft true - создание контекстного меню производится и для дочерних типов (instanceof)
     * false - строгая проверка совпадения типа
     * @return self ссылка
     */
    @Override
    public ContextMenu<T> menu(boolean matchSoft){
        return new ContextMenuBuilder<>(this,matchSoft);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="leftClick() / leftDblClick()">
    /**
     * Указывает функцию срабатываемую при левом клике мыши
     * @param clickfn функция
     * @return self ссылка
     */
    @Override
    public synchronized Context<T> leftClick( final Func3<Object,T,TreeTableNodeBasic,MouseEvent> clickfn ){
        return leftClick(false, clickfn);
    }
    
    /**
     * Указывает функцию срабатываемую при левом клике мыши
     * @param clickfn функция
     * @return self ссылка
     */
    @Override
    public synchronized Context<T> leftDblClick( final Func3<Object,T,TreeTableNodeBasic,MouseEvent> clickfn ){
        return leftClick(true, clickfn);
    }
    
    /**
     * Указывает функцию срабатываемую при левом клике мыши
     * @param dblClick true - двойной клик
     * @param clickfn функция
     * @return self ссылка
     */
    public synchronized Context<T> leftClick( final boolean dblClick, final Func3<Object,T,TreeTableNodeBasic,MouseEvent> clickfn ){
        if( clickfn==null )return null;
        applyQueue.add(new Runnable() {
            @Override
            public void run() {
                if( baseObjectBrowser==null )return;
                Func3<Object,MouseEvent,TreeTableNodeBasic,Object> clickFn = new Func3<Object, MouseEvent, TreeTableNodeBasic, Object>() {
                    @Override
                    public Object apply(MouseEvent arg1, TreeTableNodeBasic arg2, Object arg3) {
                        clickfn.apply((T)(arg2.getData()), arg2, arg1);
                        return null;
                    }
                };
                
                NamedCollection ncol = new NamedCollection();
                ncol.values(clickFn);
                
                if( dblClick ){
                    baseObjectBrowser.getLeftMouseDblClick().add(cls, ncol);
                }else{
                    baseObjectBrowser.getLeftMouseClick().add(cls, ncol);
                }
                
                final WeakReference<ObjectBrowser> fbob = new WeakReference<>(baseObjectBrowser);
                final WeakReference<Class> fcls = new WeakReference<>((Class)cls);
                final WeakReference<NamedCollection> fncol = new WeakReference<>(ncol);
                //final WeakReference<Boolean> fdbl = new WeakReference<>(dblClick);
                
                if( cset!=null ){
                    cset.add(new Runnable() {
                        @Override
                        public void run() {
                            ObjectBrowser bob = fbob.get();
                            Class cls = fcls.get();
                            NamedCollection nc = fncol.get();
                            if( bob==null || cls==null || nc==null )return;
                            
                            bob.getLeftMouseClick().remove(cls, nc);
                            bob.getLeftMouseDblClick().remove(cls, nc);
                        }
                    });
                }
            }
        });
        return this;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="apply()">
    /**
     * Применение настроек контекста
     * @return отмена настроек
     */
    @Override
    public synchronized AutoCloseable apply() {
        if( applied )throw new IllegalStateException("already applied");
        while( true ){
            Runnable r = applyQueue.poll();
            if( r==null )break;
            r.run();
        }
        cset.add(new Runnable() {
            @Override
            public void run() {
                synchronized(ContextBuilder.this){
                    applied = false;
                }
            }
        });
        baseObjectBrowser = null;
        return this;
    }
    //</editor-fold>

    @Override
    public void close() throws Exception {
        cset.closeAll();
        baseObjectBrowser = null;
    }
}
