/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package xyz.cofe.gui.swing.tree.ob;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.WeakHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.SwingUtilities;
import xyz.cofe.collection.BasicTriple;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.collection.Triple;
import xyz.cofe.collection.map.ClassMap;
import xyz.cofe.collection.tree.ClassNode;
import xyz.cofe.collection.tree.TreeNodeAsyncFollowed;
import xyz.cofe.collection.tree.TreeNodeAsyncFollowing;
import xyz.cofe.gui.swing.tree.TreeNodesExtracter;
import xyz.cofe.gui.swing.tree.TreeTableNode;
import xyz.cofe.gui.swing.tree.TreeTableNodeBasic;

/**
 * "Следование/извлечение" из объектов указанного типа с поддержкой асинхронного извлечения.
 * 
 * <p>
 * Процедура извлечения:
 * 
 * <ol>
 * <li> Получение способов следования следования для указанного класса из текущей карты
 * <ul>
 *   <li> Если не найдны способы и установлено свойство inheritExtracter=true, 
 *        то ищутся способы в выше стоящих узлан (parentNode.getDataFollower())
 *   <li> Если не найдны способы и ввыше стоящих, то завершение работы
 * </ul>
 * <li> Состоявляется методы извлечения: синхронный/асинхронный
 * <ul>
 *   <li> Сначала проверяется объект (treeTableNode.getData()) на (а)синхронность извлечения
 *   <li> Потом (приоритет ниже) проверяется класс объекта 
 *        (treeTableNode.getData().getClass()) на (а)синхронность извлечения
 * </ul>
 * <li>Синхронные объекты возвращаются немедленно
 * <li>Асихронные объекты
 * <ol>
 *   <li> перед началом рассылается node.popup(TreeNodeAsyncFollowing)
 *   <li> если указан, то используется ExecutorService; если нет - то создается поток Thread 
 *        с заданием на извелечение
 *   <li> Полученные объекты пересылаются в sendToAwtConsumer(node,fetched)
 *   <li> sendToAwtConsumer добавляет в очередь и добавляет задание в AWT/SWING 
 *        потоке на обработку очереди (awtFlushQueue())
 *   <li> awtFlushQueue() - читает данные из очереди и устанавливает значения в дереве
 *   <li> по заершеию асинхронной работы рассылается node.popup(TreeNodeAsyncFollowed)
 * </ol>
 * </ol>
 * @author user
 */
public class FollowerMap extends ClassNode implements TreeNodesExtracter {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(FollowerMap.class.getName());
    
    private static Level logLevel(){ return logger.getLevel(); }
    private static boolean isLogSevere(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.SEVERE.intValue();
    }
    private static boolean isLogWarning(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.WARNING.intValue();
    }
    private static boolean isLogInfo(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.INFO.intValue();
    }
    private static boolean isLogFine(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.FINE.intValue();
    }
    private static boolean isLogFiner(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINER.intValue();
    }
    private static boolean isLogFinest(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINEST.intValue();
    }
    
    private static void logEntering(String method,Object ... args){
        logger.entering(FollowerMap.class.getName(), method, args);
    }
    private static void logExiting(String method,Object result){
        logger.exiting(FollowerMap.class.getName(), method, result);
    }
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }
    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }
    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }    
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="async()">
    private Boolean isAsyncExtractFrom( TreeTableNode node, Object data ){
        Boolean asnc = asyncInst.get(node);
        if( asnc!=null)return asnc;
        
        asnc = asyncInst.get(data);
        if( asnc!=null )return asnc;
        
        if( data!=null ){
            asnc = asyncType.fetch(data.getClass());
            if( asnc!=null )return asnc;
        }
        
        return null;
    }
    private Boolean isAsyncExtracter( TreeTableNode node, Object data, NodesExtracter ne ){
        Boolean asnc = asyncInst.get(ne);
        if( asnc!=null )return asnc;
        
        if( ne!=null ){
            asnc = asyncType.fetch(ne.getClass());
            if( asnc!=null )return asnc;
        }

        /*if( data!=null && node!=null ){
            asnc 
        }*/
                
        return null;
    }
    
    private final WeakHashMap<Object,Boolean> asyncInst = new WeakHashMap<>();
    
    /**
     * Указанный объект должен извлекаться (а)синхронно
     * @param obj объект
     * @param async true - асинхронно / false - синхронно
     * @return карта "следования" (self ссылка)
     */
    public FollowerMap async(Object obj, boolean async){
        if( obj==null )return this;
        logFine( async ? "set async {0}" : "set sync {0}", obj);
        asyncInst.put(obj, async);
        return this;
    }
    /**
     * Указанный объект должен извлекаться асинхронно
     * @param obj объект
     * @return карта "следования" (self ссылка)
     */
    public FollowerMap async(Object obj){
        return async(obj, true);
    }
    
    private final ClassMap<Boolean> asyncType = new ClassMap<>();
    /**
     * Указанный класс должен извлекаться (а)синхронно
     * @param cls класс
     * @param async true - асинхронно / false - синхронно
     * @return карта "следования" (self ссылка)
     */
    public FollowerMap async(Class cls, boolean async){
        if( cls!=null ){
            logFine( async ? "set async {0}" : "set sync {0}", cls);
            asyncType.put(cls, async);
        }
        
        return this;
    }
    /**
     * Указанный класс должен извлекаться асинхронно
     * @param cls класс
     * @return карта "следования" (self ссылка)
     */
    public FollowerMap async(Class cls){
        return async(cls, true);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="inheritExtracter : boolean">
    protected volatile boolean inheritExtracter = true;
    /**
     * Указывает наследовать способы извлечения/следования из выше стоящих узлов дерева
     * @return true - наследовать способы извлечения/следования
     */
    public boolean isInheritExtracter() { return inheritExtracter; }
    /**
     * Указывает наследовать способы извлечения/следования из выше стоящих узлов дерева
     * @param inheritExtracter true - наследовать способы извлечения/следования
     */
    public void setInheritExtracter(boolean inheritExtracter) { this.inheritExtracter = inheritExtracter; }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="extract()">
    //private NodesExtracter[] extractersOf1(Class cls)
    @Override
    public Iterable extract(TreeTableNode node) {
        if( node==null )return null;
        logFine("extract node={0} data={1}", node, node.getData());
        
        final Object data = node.getData();
        if( data==null ){
            logFiner("data=null return");
            return null;
        }
        
        NodesExtracter[] neArr = null;
        neArr = extractersOf(data.getClass());
        //<editor-fold defaultstate="collapsed" desc="resolve from parents">
        if( neArr==null || neArr.length<1 && inheritExtracter ){
            Object oprnt = node.getParent();
            if( oprnt instanceof TreeTableNodeBasic ){
                logFiner("found 0 extracters from {0}, try found extracters from parents",data.getClass());
                TreeTableNodeBasic prntNode = (TreeTableNodeBasic)oprnt;
                List<TreeTableNodeBasic> prntNodes = prntNode.getNodePath();
                List<ClassNode> fmaps = new ArrayList<>();
                for( TreeTableNodeBasic pn : prntNodes ){
                    Object df = pn.getDataFollower();
                    if( df instanceof ClassNode ){
                        fmaps.add((ClassNode)df);
                    }
                }
                
                int selfIdx = fmaps.indexOf(this);
                logFiner( "found {0} parent ClassNode's, self index={1}",fmaps.size(), selfIdx );
                if( selfIdx>=0 ){
                    if( selfIdx==0 )return null;
                    for( int fmi=selfIdx-1; fmi>=0; fmi-- ){
                        ClassNode cn = fmaps.get(fmi);
                        if( cn==null )continue;
                        
                        neArr = cn.extractersOf(data.getClass());
                        if( neArr==null || neArr.length<1 )continue;
                        
                        logFiner( "selected [{0}] {1} with {2} extracters",fmi,cn, neArr.length );
                        break;
                    }
                }else{
                    for( int fmi=fmaps.size(); fmi>=0; fmi-- ){
                        ClassNode cn = fmaps.get(fmi);
                        if( cn==null )continue;
                        
                        neArr = cn.extractersOf(data.getClass());
                        if( neArr==null || neArr.length<1 )continue;
                        
                        logFiner( "selected [{0}] {1} with {2} extracters",fmi,cn, neArr.length );
                        break;
                    }
                }
            }else{
                logFiner("return, found 0 extracters from {0}",data.getClass());
                return null;
            }
            
            if( neArr==null || neArr.length<1 ){
                logFiner("return, found 0 extracters from {0}",data.getClass());
                return null;
            }
        }
        //</editor-fold>
        
        if( neArr==null || neArr.length<1 ){
            logFiner("return, found 0 extracters from {0}",data.getClass());
            return null;
        }
        
        Boolean asyncExtrFrm = isAsyncExtractFrom(node, data);
        if( asyncExtrFrm!=null ){            
            if( asyncExtrFrm==false ){
                logFiner("use sync extract");
                ArrayList<NodesExtracter> syncExtracters = new ArrayList<>(Arrays.asList(neArr));
                ArrayList extracted = new ArrayList<>();
                for( NodesExtracter ne : syncExtracters ){
                    Iterable iter = ne.extract(data);
                    if( iter!=null ){
                        for( Object e : iter ){
                            extracted.add(e);
                        }
                    }
                }
                return extracted;
            }else{
                logFiner("use async extract");
                startAsync(node, data, Arrays.asList(neArr));
                return null;
            }
        }else{
            ArrayList<NodesExtracter> syncExtracters = new ArrayList<>();
            ArrayList<NodesExtracter> asyncExtracters = new ArrayList<>();
            
            for( NodesExtracter ne : neArr ){
                Boolean asyncExt = isAsyncExtracter(node, data, ne);
                if( Objects.equals(asyncExt,true) ){
                    asyncExtracters.add(ne);
                }else{
                    syncExtracters.add(ne);
                }
            }
            logFiner("use {0} sync extracters, {1} async extracters", syncExtracters.size(), asyncExtracters.size());
            if(asyncExtracters.size()>0){
                startAsync(node, data, asyncExtracters);
            }
            
            ArrayList extracted = new ArrayList<>();
            for( NodesExtracter ne : syncExtracters ){
                Iterable iter = ne.extract(data);
                if( iter!=null ){
                    for( Object e : iter ){
                        extracted.add(e);
                    }
                }
            }
            return extracted;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="executorService : ExecutorService">
    private ExecutorService executorService;
    /**
     * Указывает сервис для выполнения асинхронных задач
     * @return ExecutorService
     */
    public ExecutorService getExecutorService() { return executorService; }
    /**
     * Указывает сервис для выполнения асинхронных задач
     * @param executorService ExecutorService
     */
    public void setExecutorService(ExecutorService executorService) { this.executorService = executorService; }
    //</editor-fold>    
    //<editor-fold defaultstate="collapsed" desc="startAsync()">
    private void startAsync(final TreeTableNode node, final Object data, final List<NodesExtracter> extracters){
        for( NodesExtracter ne : extracters ){
            startAsync(node, data, ne);
        }
    }
    private void startAsync(final TreeTableNode node, final Object data, final NodesExtracter extracter){
        if( node==null )return;
        
        logFine("startAsync node={0}, data={1}, extracter={2}", node, data, extracter);
        final AtomicReference<TreeNodeAsyncFollowing>
            beginEvent = new AtomicReference<>();
        
        //final List
        Runnable run = new Runnable() {
            @Override
            public void run() {
                try{
                    Iterable iter = extracter.extract(data);
                    if( iter!=null ){
                        final ArrayList fetched = new ArrayList();
                        for( Object it : iter ){
                            TreeTableNodeBasic tn = null;
                            if( it instanceof TreeTableNodeBasic ){
                                tn = (TreeTableNodeBasic)it;
                            }else{
                                tn = new TreeTableNodeBasic(it);
                            }
                            fetched.add(tn);
                        }
                        sendToAwtConsumer(node,fetched,beginEvent.get());
                    }
                }catch( Throwable err ){
                    String msg = err.getMessage();
                    msg = msg==null ? err.toString() : msg;
                    
                    logSevere(
                        "fail extract (follow) from node {2} (data: {3}), extracter={4}, class={0} message={1}", 
                        err.getClass().getName(), msg,
                        node, data, extracter
                    );
                    logException(err);
                }
            }
        };
        
        ExecutorService esrvc = getExecutorService();
        if( esrvc!=null ){
            Future f = esrvc.submit(run);
            
            TreeNodeAsyncFollowing ev = new TreeNodeAsyncFollowing(node);
            ev.setData(data);
            ev.setFuture(f);
            ev.setRunnable(run);
            ev.setExtracter(extracter);
            node.popup(ev);
            
            beginEvent.set(ev);
        }else{
            Thread th = new Thread(run);
            th.setDaemon(true);
            th.setPriority(Thread.MIN_PRIORITY);
            
            TreeNodeAsyncFollowing ev = new TreeNodeAsyncFollowing(node);
            ev.setData(data);
            ev.setThread(th);
            ev.setRunnable(run);
            ev.setExtracter(extracter);
            node.popup(ev);
            
            beginEvent.set(ev);
            
            th.start();
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="sendToAwtConsumer()">
    private void sendToAwtConsumer(
        final TreeTableNode node, 
        final List<TreeTableNodeBasic> fetched,
        final TreeNodeAsyncFollowing parentEvent
    ){
        logFine("sendToAwtConsumer");
        
        TreeNodeAsyncFollowed ev = new TreeNodeAsyncFollowed(node);
        ev.setFetched(fetched);
        ev.setParentEvent(parentEvent);

        awtConsumerQueue.add(new BasicTriple<>(node,fetched,ev));
        
        SwingUtilities.invokeLater( new Runnable() {
            @Override
            public void run() {
                awtFlushQueue();
            }
        });
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="awtFlushQueue()">
    private final LinkedBlockingQueue
        <Triple<TreeTableNode,List<TreeTableNodeBasic>,TreeNodeAsyncFollowed>> 
        awtConsumerQueue = new LinkedBlockingQueue<>();
    
    private void awtFlushQueue(){
        logFine("awtFlushQueue");
        while( true ){
            Triple<TreeTableNode,List<TreeTableNodeBasic>,TreeNodeAsyncFollowed> p = awtConsumerQueue.poll();
            if( p==null )break;
            
            TreeTableNode node = p.A();
            final List<TreeTableNodeBasic> fetched = p.B();
            
            List children = node.getChildrenList();
            /*if( children instanceof BasicEventList ){
                final BasicEventList blist = (BasicEventList)children;
                blist.lockRun(new Func0() {
                    @Override
                    public Object apply() {
                        int pos = blist.size();
                        logFiner("bulk insert {1} nodes into {0} position", pos, fetched.size());
                        blist.bulkInsert(pos, fetched);
                        return null;
                    }
                }, new LockMethod("bulkInsert"));
            }else{*/
                logFiner("append {0} nodes at end", fetched.size());
                //children.addAll(fetched);
                if( node instanceof TreeTableNodeBasic){
                for( TreeTableNodeBasic cn : fetched ){
                    ((TreeTableNodeBasic)node).appendChild(cn);
                }}else{
                    children.addAll(fetched);
                }
            //}
            
            if( node instanceof TreeTableNodeBasic ){
                TreeTableNodeBasic ttnb = (TreeTableNodeBasic)node;
                Date dnow = new Date();
                for( TreeTableNodeBasic cn : fetched ){
                    ttnb.getCachedNodes().put(cn, dnow);
                }
            }
            
            TreeNodeAsyncFollowed ev = p.C();
            if( ev!=null ){
                Object otn = ev.getSource();
                if( otn instanceof TreeTableNodeBasic ){
                    TreeTableNodeBasic tn = (TreeTableNodeBasic)otn;
                    tn.popup(ev);
                }
            }
        }
    }
    //</editor-fold>
}
