/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.tree.ob;


import java.net.URL;
import java.util.List;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.map.ClassMap;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormat;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormatBasic;
import xyz.cofe.gui.swing.tree.TreeTableNodeGetFormatOf;

/**
 * Карта форматирования различных типов данных ула дерева (браузера объектов)
 * @author nt.gocha@gmail.com
 */
public class FormatMap extends ClassMap<Func1<TreeTableNodeFormat,Object>> 
    implements TreeTableNodeGetFormatOf
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(FormatMap.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(FormatMap.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(FormatMap.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(FormatMap.class.getName(), method, result);
    }
    //</editor-fold>
    
    private static volatile Icon componentIcon;
    /**
     * Возвращает иконку для отображения любого объекта дерева - иконка по умолчанию
     * @return иконка по умолчанию
     */
    public static Icon getComponentIcon(){
        if( componentIcon!=null )return componentIcon;
        synchronized(FormatMap.class){
            if( componentIcon!=null )return componentIcon;
            URL url = FormatMap.class.getResource("component-16.png");
            componentIcon = new ImageIcon(url);
            return componentIcon;
        }
    }
    
    protected static volatile Icon refreshIconSmall;
    /**
     * Возвращает иконку обновления (refresh)
     * @return иконка "обновить"
     */
    public static Icon getRefreshIconSmall(){
        if( refreshIconSmall!=null )return refreshIconSmall;
        synchronized(FormatMap.class){
            if( refreshIconSmall!=null )return refreshIconSmall;
            refreshIconSmall = new ImageIcon(FormatMap.class.getResource("refresh-icon-16.png"));
            return refreshIconSmall;
        }
    }
    
    protected volatile TreeTableNodeFormatBasic defaultFormat;
    /**
     * Возвращает форматирование по умолчанию
     * @return форматирование по умолчанию
     */
    public TreeTableNodeFormatBasic getDefaultFormat(){
        synchronized(this){
        if( defaultFormat!=null ){
            return defaultFormat;
        }
        defaultFormat = new TreeTableNodeFormatBasic();
        defaultFormat.getIcons().add(getComponentIcon());
        defaultFormat.setIconWidthMin(16);
        return defaultFormat;
        }
    }
    /**
     * Указывает форматирование по умолчанию
     * @param fmt форматирование по умолчанию
     */
    public synchronized void setDefaultFormat( TreeTableNodeFormatBasic fmt ){
        defaultFormat = fmt;
    }
    
    private final ClassMap<Func1<Icon,Object>> typeIconMap = new ClassMap<>();
    /**
     * Указывает карту функций - иконок, для различных типов данных узлов
     * @return карта иконок
     */
    public ClassMap<Func1<Icon,Object>> getTypeIconMap(){ return typeIconMap; }
    
    private final WeakHashMap<Object,Icon> objectIcon = new WeakHashMap<>();
    /**
     * Возвращает карту иконок для конкретных экземпляров
     * @return карта иконок для экземпляров
     */
    public WeakHashMap<Object,Icon> getObjectIconMap(){ return objectIcon; }
    /**
     * Возвращает иконку для объекта
     * @param obj объект
     * @return иконка или null
     */
    public Icon objectIcon(Object obj){ return objectIcon.get(obj); }
    public void objectIcon(Object obj,Icon ico){ 
        if( obj==null )return;
        if( ico!=null ){
            objectIcon.put(obj, ico);
        }else{
            objectIcon.remove(obj);
        }
    }

    /**
     * Возвращает функцию форматирования узла дерева
     * @param value данные узла дерева
     * @return функция форматирования
     */
    @Override
    public TreeTableNodeFormat getTreeTableNodeFormatOf(Object value) {
        TreeTableNodeFormat deffmt = getDefaultFormat();
            
        if( value!=null ){
            Func1<Icon,Object> fnIco = getTypeIconMap().fetch(value.getClass());
            Icon ico = fnIco!=null ? fnIco.apply(value) : null;
            ico = ico!=null ? ico : objectIcon(value);

            Func1<TreeTableNodeFormat,Object> fn = this.fetch(value.getClass());
            if( fn!=null ){
                TreeTableNodeFormat fmt = fn.apply(value);

                TreeTableNodeFormat rfmt = deffmt.clone();                    
                rfmt.merge(deffmt);
                rfmt.merge(fmt);

                List<Icon> icons = fmt.getIcons();
                if( icons!=null && icons.size()>0 ){
                    rfmt.getIcons().clear();
                    rfmt.getIcons().addAll(icons);
                }else if( ico!=null ){
                    rfmt.getIcons().clear();
                    rfmt.getIcons().add(ico);
                }

                return rfmt;
            }else{
                if( ico!=null ){
                    TreeTableNodeFormat rfmt = deffmt.clone();
                    rfmt.getIcons().clear();
                    rfmt.getIcons().add(ico);
                    return rfmt;
                }
            }
        }

        return deffmt;
    }
    /**
     * Указывает форматирования для определенного типа данных
     * @param <T> тип данных узла дерева
     * @param type тип данных узла дерева
     * @param fmtfn функция форматирования
     */
    public <T> void format( Class<T> type, Func1<TreeTableNodeFormat,T> fmtfn ){
        if (type== null) {
            throw new IllegalArgumentException("type==null");
        }
        if( fmtfn==null ){
            this.remove(type);
        }else{
            this.put(type, (Func1)fmtfn);
        }
    }
    /**
     * Указывает форматирования для определенного типа данных
     * @param <T> тип данных узла дерева
     * @param type тип данных узла дерева
     * @param fmt форматирование
     */
    public <T> void format( Class<T> type, final TreeTableNodeFormat fmt ){
        if (type== null) {
            throw new IllegalArgumentException("type==null");
        }
        if( fmt==null ){
            this.remove(type);
        }else{
            this.put(type, new Func1<TreeTableNodeFormat, Object>() {
                @Override
                public TreeTableNodeFormat apply(Object arg) {
                    return fmt;
                }
            });
        }
    }
}
