/*
 * The MIT License
 *
 * Copyright 2018 georgiy.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.tree.ob;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Icon;
import javax.swing.JMenuItem;
import xyz.cofe.collection.NodesExtracter;

/**
 * Именнованая коллекция узлов
 * @author georgiy
 */
public class NamedCollection implements NodesExtracter<Object, Object>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(NamedCollection.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(NamedCollection.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(NamedCollection.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(NamedCollection.class.getName(), method, result);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="id()">
    private transient static final AtomicLong idseq = new AtomicLong(0);
    private final long id = idseq.incrementAndGet();
    /**
     * Возвращает уникальный идентификатор объекта, уникальный в пределах экземпляра JVM
     * @return идентификатор
     */
    public long id(){ return id; }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="name : String">
    protected transient volatile String name;
    
    /**
     * Указывает имя узла
     * @return имя узла
     */
    public String getName() {
        return name;
    }
    
    /**
     * Указывает имя узла
     * @param name имя узла
     */
    public void setName(String name) {
        this.name = name;
    }
    
    /**
     * Указывает имя узла
     * @param name имя узла
     * @return self - ссылка
     */
    public synchronized NamedCollection name( String name ){
        setName(name);
        return this;
    }
    
    /**
     * Указывает имя узла
     * @param name имя узла
     * @param ico иконка
     * @return self - ссылка
     */
    public synchronized NamedCollection name( String name, Icon ico ){
        setName(name);
        setIcon(ico);
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="icon : Icon">
    protected Icon icon = FormatMap.getComponentIcon();
    
    /**
     * Указывает иконку узла
     * @return иконка
     */
    public synchronized Icon getIcon() {
        return icon;
    }
    
    /**
     * Указывает иконку узла
     * @param icon иконка
     */
    public synchronized void setIcon(Icon icon) {
        this.icon = icon;
    }
    
    /**
     * Указывает иконку узла
     * @param ico иконка
     * @return self - ссылка
     */
    public NamedCollection icon(Icon ico){
        this.icon = ico;
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="bold : Boolean">
    protected Boolean bold;
    
    /**
     * Возвращает что надо отображать жирным шрифтом
     * @return true - отображать жирным шрифтом
     */
    public synchronized Boolean getBold() {
        return bold;
    }
    
    /**
     * Указывает что надо отображать жирным шрифтом
     * @param bold true - отображать жирным шрифтом
     */
    public synchronized void setBold(Boolean bold) {
        this.bold = bold;
    }
    
    /**
     * Указывает что надо отображать жирным шрифтом
     * @param bold true - отображать жирным шрифтом
     * @return self ссылка
     */
    public NamedCollection bold(Boolean bold){
        this.bold = bold;
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="italic : Boolean">
    protected Boolean italic;
    
    /**
     * Указывает что надо отображать наклонным шрифтом
     * @return true - отображать наклонным шрифтом
     */
    public synchronized Boolean getItalic() {
        return italic;
    }
    
    /**
     * Указывает что надо отображать наклонным шрифтом
     * @param italic true - отображать наклонным шрифтом
     */
    public synchronized void setItalic(Boolean italic) {
        this.italic = italic;
    }
    
    /**
     * Указывает что надо отображать наклонным шрифтом
     * @param italic true - отображать наклонным шрифтом
     * @return self ссылка
     */
    public NamedCollection italic(Boolean italic){
        this.italic = italic;
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="foreground : Color">
    protected Color foreground;
    
    /**
     * Указывает цвет текста при отображении узла дерева
     * @return цвет текста
     */
    public synchronized Color getForeground() {
        return foreground;
    }
    
    /**
     * Указывает цвет текста при отображении узла дерева
     * @param foreground цвет текста
     */
    public synchronized void setForeground(Color foreground) {
        this.foreground = foreground;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="node : Object">
    private transient volatile WeakReference nodeRef;
    
    /**
     * Указывает ссылку на данные узла
     * @return ссылка на данные
     */
    public synchronized Object getNode(){
        if( nodeRef!=null )return nodeRef.get();
        return null;
    }
    
    /**
     * Указывает ссылку на данные узла
     * @param node ссылка на данные
     */
    public synchronized void setNode( Object node ){
        if( node==null ){
            if( nodeRef!=null )nodeRef.clear();
            nodeRef = null;
        }else{
            if( nodeRef!=null )nodeRef.clear();
            nodeRef = new WeakReference(node);
        }
    }
    
    /**
     * Указывает ссылку на данные узла
     * @param node ссылка на данные
     * @return self ссылка
     */
    public synchronized NamedCollection node( Object node ){
        setNode(node);
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="nodesExtracter : NodesExtracter">
    private transient volatile NodesExtracter nodesExtracter;
    
    /**
     * Указывает функцию "следования", для получения дочерних узлов
     * @return функция "следования"
     */
    public synchronized NodesExtracter getNodesExtracter() {
        return nodesExtracter;
    }
    
    /**
     * Указывает функцию "следования", для получения дочерних узлов
     * @param nodesExtracter функция "следования"
     */
    public synchronized void setNodesExtracter(NodesExtracter nodesExtracter) {
        this.nodesExtracter = nodesExtracter;
    }
    
    /**
     * Указывает функцию "следования", для получения дочерних узлов
     * @param node узел
     * @param extracter функция
     * @return self ссылка
     */
    public synchronized NamedCollection extracter( Object node, NodesExtracter extracter ){
        setNode(node);
        setNodesExtracter(extracter);
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="extracted">
    private transient volatile Iterable extracted;
    
    /**
     * Указывает уже готовый список дочерних объектов
     * @return готовй список дочерних объектов или null
     */
    public synchronized Iterable getExtracted() {
        return extracted;
    }
    
    /**
     * Указывает уже готовый список дочерних объектов
     * @param extracted готовй список дочерних объектов или null
     */
    public synchronized void setExtracted(Iterable extracted) {
        this.extracted = extracted;
    }
    
    /**
     * Указывает уже готовый список дочерних объектов
     * @param extracted готовй список дочерних объектов или null
     * @return self ссылка
     */
    public synchronized NamedCollection extracted( Iterable extracted ){
        setExtracted(extracted);
        return this;
    }
    
    /**
     * Указывает уже готовый список дочерних объектов
     * @param values готовй список дочерних объектов
     * @return self ссылка
     */
    public synchronized NamedCollection values( Object ... values ){
        if( values==null )throw new IllegalArgumentException("values == null");
        setExtracted(Arrays.asList(values));
        return this;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="contextMenu : List<JMenuItem>">
    protected List<JMenuItem> contextMenu;
    /**
     * Указывает пункты контекстного меню
     * @return пункты контекстного меню
     */
    public synchronized List<JMenuItem> getContextMenu(){
        if( contextMenu==null )contextMenu = new ArrayList<>();
        return contextMenu;
    }
    
    /**
     * Добавляет пункт контекстного меню
     * @param label имя пункта меню
     * @param run функция срабатываемая при нажатии на пункт контекстного меню
     * @return self ссылка
     */
    public NamedCollection menu(String label, final Runnable run){
        if( label==null )throw new IllegalArgumentException("label == null");
        if( run==null )throw new IllegalArgumentException("run == null");
        
        JMenuItem mi = new JMenuItem(label);
        mi.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                run.run();
            }
        });
        
        getContextMenu().add(mi);
        
        return this;
    }
    
    /**
     * Добавляет пункт контекстного меню
     * @param mi контекстное меню
     * @return self ссылка
     */
    public NamedCollection menu(JMenuItem mi){
        if( mi==null )throw new IllegalArgumentException("mi == null");
        getContextMenu().add(mi);
        return this;
    }
    //</editor-fold>
    
    /**
     * Извлекает дочерние элемены
     * @param from извлекат данные из узла (getNode)
     * @return дочение объекты или null
     */
    @Override
    public synchronized Iterable<Object> extract(Object from) {
        if( extracted!=null )return extracted;
        
        Object node = getNode();
        NodesExtracter ne = getNodesExtracter();
        if( node==null )return null;
        if( ne==null )return null;
        return ne.extract(node);
    }
    
    @Override
    public synchronized String toString() {
        return name!=null ? name : this.getClass().getSimpleName()+"#"+id;
    }
}
