/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.tree.ob;

import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.collection.map.ClassMap;
import xyz.cofe.collection.tree.ClassNode;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.SwingListener;
import xyz.cofe.gui.swing.tree.TreeTable;
import xyz.cofe.gui.swing.tree.TreeTable;
import xyz.cofe.gui.swing.tree.TreeTableNodeBasic;
import xyz.cofe.gui.swing.tree.TreeTableNodeBasic;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormat;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormat;

/**
 * Браузер объектов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class ObjectBrowser extends TreeTable 
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(ObjectBrowser.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(ObjectBrowser.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(ObjectBrowser.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(ObjectBrowser.class.getName(), method, result);
    }
    //</editor-fold>
    
    public ObjectBrowser(){
        setRowHeight(23);
        
        setRequestFocusEnabled(true);
        setRootVisible(false);
        
        TreeTableNodeBasic root = new TreeRootNode(this);
        super.setRoot(root);
        root.expand();
        
        root.setDataTextReader(nameFn);
        root.setDataFollowable(followableFn);
        root.setDataFollower(followerMap);
        root.setDataFormatter(formatMap);
        
        SwingListener.onMousePressed(this, new Reciver<MouseEvent>() {
            @Override
            public void recive(MouseEvent obj) {
                if( obj==null )return;
                onMousePressed(obj);
            }
        });
        SwingListener.onMouseClicked(this, new Reciver<MouseEvent>() {
            @Override
            public void recive(MouseEvent obj) {
                if( obj==null )return;
                onMouseClick(obj);
            }
        });
        
        reloadPlugins();
    }
    
    /**
     * Обвноляет дерево объектов
     */
    public void rebuildTree(){
        logFine("rebuildTree");
        logFiner("direct model - fireAllChanged");
        getTreeTableModel().getDirectModel().fireAllChanged();
        logFiner("filter model - applyFilter");
        getTreeTableModel().getFilterModel().applyFilter();
    }
    
    //<editor-fold defaultstate="collapsed" desc="plugins">
    protected final Map<ObjectBrowserPlugin,CloseableSet>
        plugins = new LinkedHashMap<>();
    
    /**
     * Возвращает плагины
     * @return плагины
     */
    public Map<ObjectBrowserPlugin,CloseableSet> getPlugins(){
        return plugins;
    }
    
    /**
     * Перезагрузка плагинов
     */
    public void reloadPlugins(){
        if( plugins!=null ){
            for( Map.Entry<ObjectBrowserPlugin,CloseableSet> en : plugins.entrySet() ){
                CloseableSet cs = en!=null ? en.getValue() : null;
                if( cs!=null ){
                    cs.closeAll();
                }
            }
            plugins.clear();
        }
        
        for( ObjectBrowserPlugin obp : ServiceLoader.load(ObjectBrowserPlugin.class) ){
            if( obp==null )continue;
            
            CloseableSet cs = new CloseableSet();
            obp.register(this, cs);
            
            plugins.put(obp, cs);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="refresh()">
    private boolean fix_storeColumnsWithOnToggle = true;
    private boolean fix_storeColumnsWithOnToggle_pref = true;
    
    /**
     * Обновляет поддерево сфокусировано узла
     */
    public void refreshFocused(){
        Object ofoc = getFocusedNode();
        if( ofoc instanceof TreeTableNodeBasic ){
            TreeTableNodeBasic ttnb = (TreeTableNodeBasic)ofoc;
            refresh( ttnb );
        }
    }
    
    /**
     * Обновляет поддерево узла
     * @param ttnb узел
     */
    public void refresh(TreeTableNodeBasic ttnb){
        //<editor-fold defaultstate="collapsed" desc="fix expand bug">
        boolean fixCWidth = fix_storeColumnsWithOnToggle;
        int[] colWidth = null;
        if( fixCWidth ){
            colWidth = getColumnsWidths();
        }
        //</editor-fold>
        
        boolean exp = ttnb.isExpanded();
        ttnb.collapse();
        ttnb.dropCache();
        if( exp )ttnb.expand();
        
        //<editor-fold defaultstate="collapsed" desc="fix expand bug">
        if( colWidth!=null && colWidth.length>0 ){
            setColumnsWidths(fix_storeColumnsWithOnToggle_pref, colWidth);
        }
        //</editor-fold>
        
        repaint();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="root">    
    /**
     * Возвращает корень дерева объектов
     * @return корень дерева объектов
     */
    @Override
    public synchronized TreeTableNodeBasic getRoot() {
        TreeTableNodeBasic root = super.getRoot();
        if( root==null ){
            root = new TreeTableNodeBasic(this);
            super.setRoot(root);
            root.expand();
            
            root.setDataTextReader(nameFn);
            root.setDataFollowable(followableFn);
            root.setDataFollower(followerMap);
            root.setDataFormatter(formatMap);
        }
        return root;
    }
    
    /**
     * Указывает корень дерева объектов
     * @param newroot корень дерева
     */
    @Override
    public void setRoot(TreeTableNodeBasic newroot) {
        super.setRoot(newroot);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="addTop()">
    /**
     * Добавляет узел в верних уровень дерева
     * @param value данные узла
     * @return узел дерева
     */
    public TreeTableNodeBasic addTop( Object value ){
        TreeTableNodeBasic node =
            value instanceof TreeTableNodeBasic
            ? (TreeTableNodeBasic)value
            : new TreeTableNodeBasic(value);
        
        TreeTableNodeBasic root = getRoot();
        root.appendChild(node);
        return node;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="naming()">
    private final ClassMap<Func1<String,Object>> naming = new ClassMap<>();
    public ClassMap<Func1<String,Object>> getNamingMap(){ return naming; }
    
    private final WeakHashMap<Object,String> objectName = new WeakHashMap<>();
    public WeakHashMap<Object,String> objectName(){ return objectName; }
    public String objectName(Object obj){ return objectName.get(obj); }
    
    /**
     * Указывает имя объекта
     * @param obj объект
     * @param name имя или null
     */
    public void objectName(Object obj,String name){ 
        if( obj==null )return;
        if( name==null ){
            objectName.remove(obj);
        }else{
            objectName.put(obj, name);
        }
    }
    
    private final Func1<String,Object> nameFn = new Func1<String, Object>() {
        @Override
        public String apply(Object arg) {
            if( arg==null ){
                return null;
            }
            
            Class cl = arg.getClass();
            
            Func1<String,Object> namefn = naming.fetch(cl);
            if( namefn!=null ){
                return namefn.apply(arg);
            }
            
            String name = objectName(arg);
            if( name!=null )return name;
            
            return arg.toString();
        }
    };
    /**
     * Вохвращает функцию именования объектов
     * @return функуция
     */
    public Func1<String,Object> getNameFn(){ return nameFn; }
    /**
     * Указывает функцию именования объектов
     * @param <T> тип данных (тип объекта в данных узла дерева)
     * @param type тип данных
     * @param nameFn функция
     */
    public <T> void naming( Class<T> type, Func1<String,T> nameFn ){
        if (type== null) {
            throw new IllegalArgumentException("type==null");
        }
        if( nameFn!=null ){
            naming.put(type, (Func1)nameFn);
        }else{
            naming.remove(type);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="followable">
    private final ClassMap<Func1<Boolean,Object>> followableMap = new ClassMap<>();
    /**
     * Возвращает карту достпности дочерних узлов для извлечения дочерних объектов
     * @return карта функция проверки доступности дочерних узлов
     */
    public ClassMap<Func1<Boolean,Object>> getFollowableMap(){ return followableMap; }
    private final Func1<Boolean,Object> followableFn = new Func1<Boolean, Object>() {
        @Override
        public Boolean apply(Object arg) {
            if( arg==null ){
                return null;
            }
            
            Class cl = arg.getClass();
            
            Func1<Boolean,Object> fn = followableMap.fetch(cl);
            if( fn!=null ){
                return fn.apply(arg);
            }else{
                NodesExtracter[] extracters = followerMap.extractersOf(arg.getClass());
                if( extracters==null || extracters.length<1 )return false;
                
                int notFollowableExtrs =  0;
                for( NodesExtracter extr : extracters ){
                    if( extr instanceof Followable ){
                        if( ((Followable) extr).followable(arg) )return true;
                        notFollowableExtrs++;
                    }else{
                        break;
                    }
                }                
                if( notFollowableExtrs==extracters.length )return false;
                
                Iterable itr = followerMap.fetch(arg);
                return itr!=null;
            }
        }
    };
    /**
     * Возвращает функцию проверки доступности дочерних узлов
     * @return функция проверки доступности
     */
    public Func1<Boolean,Object> getFollowableFn(){ return followableFn; }
    
    /**
     * Добавляет функцию проверки доступности дочерних элементов
     * @param <T> тип данных ула
     * @param type тип данных ула
     * @param followableFn функция проверки доступности
     */
    public <T> void followable( Class<T> type, Func1<Boolean,T> followableFn ){
        if (type== null) {
            throw new IllegalArgumentException("type==null");
        }
        if( followableFn!=null ){
            followableMap.put(type, (Func1)followableFn);
        }else{
            followableMap.remove(type);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="follower()">
    private final FollowerMap followerMap = new FollowerMap();
    
    /**
     * Возвращает карту "следования" дочерних объектов
     * @return карта "следования"
     */
    public FollowerMap getFollowerMap(){ return followerMap; }
    
    /**
     * Добавляет функцию "следования" для определенного типа узла
     * @param <T> тип данных узла дерева
     * @param type тип данных узла дерева
     * @param followerFn функция следования
     */
    public <T> void follower( Class<T> type, NodesExtracter<T,Object> followerFn ){
        if( type==null )throw new IllegalArgumentException("type==null");
        if( followerFn==null )throw new IllegalArgumentException("followerFn==null");
        followerMap.add(type, followerFn);
    }
    
    /**
     * Добавляет функцию "следования" для определенного типа узла
     * @param <T> тип данных узла дерева
     * @param type тип данных узла дерева
     * @param followerFn функция следования
     * @param followable функиця проверки доступности дочерних узлов
     */
    public <T> void follower( Class<T> type, NodesExtracter<T,Object> followerFn, Followable<T> followable ){
        if( type==null )throw new IllegalArgumentException("type==null");
        if( followerFn==null )throw new IllegalArgumentException("followerFn==null");
        if( followable==null )throw new IllegalArgumentException("followable==null");
        follower(type, new FollowableExtracter(followerFn, followable));
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="format()">
    public ClassMap<Func1<Icon,Object>> getTypeIconMap(){ return formatMap.getTypeIconMap(); }
    
    /**
     * Возвращает карту иконок для объектов
     * @return карта иконок
     */
    public WeakHashMap<Object,Icon> getObjectIconMap(){ return formatMap.getObjectIconMap(); }
    
    /**
     * Возвращает иконку для объекта
     * @param obj объект
     * @return иконка или null
     */
    public Icon objectIcon(Object obj){ return formatMap.objectIcon(obj); }
    
    /**
     * Указывает иконку для объекта
     * @param obj объект
     * @param ico иконка
     */
    public void objectIcon(Object obj,Icon ico){  formatMap.objectIcon(obj, ico); }
        
    private final FormatMap formatMap = new FormatMap();    
    
    /**
     * Указывает карту форматирвания объектов
     * @return карта форматирования объектов
     */
    public FormatMap getFormatMap(){ return formatMap; }
    
    /**
     * Указывает форматирование для определенного типа данных в узле дерева
     * @param <T> тип данных
     * @param type тип данных
     * @param fmtfn функция форматирования
     */
    public <T> void format( Class<T> type, Func1<TreeTableNodeFormat,T> fmtfn ){
        formatMap.format(type, fmtfn);
    }
    
    /**
     * Указывает форматирование для определенного типа данных в узле дерева
     * @param <T> тип данных
     * @param type тип данных
     * @param fmt форматирование
     */
    public <T> void format( Class<T> type, final TreeTableNodeFormat fmt ){
        formatMap.format(type, fmt);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="node context menu">
    private volatile ClassNode popupMenu;
    
    /**
     * Колекция пунктов контекстного меню для определенного узла (TreeTableNode.data). <p>
     * 
     * Коллекция представляет из себя карту ключ/значение, 
     * где ключ - тип данных, значение - множествно (set) возможных "действий" с данными этого типа. <p>
     * 
     * варианты значений/"действий":
     * <ul>
     * <li> Action
     * <li> JMenuItem
     * <li> Func2( TreeTableNodeBasic, nodeData ) : Action | JMenuItem | Iterable of &lt;Action | JMenuItem&gt;
     * <li> Iterable of &lt;Action | JMenuItem&gt; 
     * </ul>
     * @return коллекция тип данных / "действия"
     */
    public ClassNode getNodeContextMenu(){
        if( popupMenu!=null )return popupMenu;
        synchronized(this){
            if( popupMenu!=null )return popupMenu;
            popupMenu = new ClassNode();
            return popupMenu;
        }
    }
    
    /*private volatile ClassMap<Set> typePopupMenu;
    public ClassMap<Set> getTypeContextMenu(){
        if( typePopupMenu!=null ) return typePopupMenu;
        synchronized(this){
            if( typePopupMenu!=null ) return typePopupMenu;
            typePopupMenu = new ClassMap<>();
            return typePopupMenu;
        }
    }*/
    
    private volatile Set<Func1<List<JMenuItem>,TreeTableNodeBasic>>
        contextMenuBuilders;
    
    /**
     * Возвращает карту для создания контекстного меню
     * @return карта построения контекстного меню
     */
    public Set<Func1<List<JMenuItem>,TreeTableNodeBasic>> getContextMenuBuilders(){
        if( contextMenuBuilders!=null )return contextMenuBuilders;
        synchronized(this){
            if( contextMenuBuilders!=null )return contextMenuBuilders;
            contextMenuBuilders = new LinkedHashSet<>();
            return contextMenuBuilders;
        }
    }
    
    /**
     * Вызывается при обработке событий мыши
     * @param ev событие мыши
     */
    protected void onMousePressed( MouseEvent ev ){
        boolean shift = ev.isShiftDown();
        boolean alt = ev.isAltDown();
        boolean cntrl = ev.isControlDown();
        boolean meta = ev.isMetaDown();
        boolean popupTrig = ev.isPopupTrigger();
        
        if( ev.getButton() == MouseEvent.BUTTON3 ){
            Object onode = getFocusedNode();
            if( onode instanceof TreeTableNodeBasic ){
                TreeTableNodeBasic node = (TreeTableNodeBasic)onode;
                //Object nodeData = node.getData();
                List<JMenuItem> menuEntries = contextMenuOf(node);
                if( menuEntries!=null && menuEntries.size()>0 ){
                    JPopupMenu popup = new JPopupMenu();
                    for( JMenuItem mi : menuEntries ){
                        popup.add( mi );
                    }
                    popup.show(this, ev.getX(), ev.getY());
                }
            }
        }
    }
    
    /**
     * Получение списка контекстного меню, для заданного узла дерева
     * @param node узел дерева
     * @return пункты меню
     */
    protected List<JMenuItem> contextMenuOf( TreeTableNodeBasic node ){
        ArrayList<JMenuItem> menu = new ArrayList<>();
        if( node==null )return menu;
        
        Object data = node.getData();
        if( data==null )return menu;
        
        //getNodeContextMenu().
        
        for( Object o : getNodeContextMenu().extract(data) ){
            if( o==null )continue;
            processCtxMenuItm(menu, node, o, true);
        }
        
        Set<Func1<List<JMenuItem>,TreeTableNodeBasic>> set = getContextMenuBuilders();
        for( Func1<List<JMenuItem>,TreeTableNodeBasic> fn : set ){
            if( fn!=null ){
                List<JMenuItem> lst = fn.apply(node);
                if( lst!=null ){
                    for( JMenuItem mi : lst ){
                        if( mi!=null )menu.add(mi);
                    }
                }
            }
        }
        
        return menu;
    }
    
    private void processCtxMenuItm(
        List<JMenuItem> consumer,
        TreeTableNodeBasic node,
        Object o,
        boolean acceptFn2
    ){
        if( o instanceof Action ){
            consumer.add(new JMenuItem((Action)o));
        }else if( o instanceof JMenuItem ){
            consumer.add((JMenuItem)o);
        }else if( o instanceof Func2 && acceptFn2 ){
            Func2 fn = (Func2)o;
            Object fnres = fn.apply(node, node.getData());
            if( fnres instanceof Iterable ){
                for( Object it : ((Iterable)fnres) ){
                    processCtxMenuItm(consumer, node, it, false);
                }
            }else{
                processCtxMenuItm(consumer, node, fnres, false);
            }
        }else if( o instanceof Iterable ){
            for( Object it : ((Iterable)o) ){
                processCtxMenuItm(consumer, node, it, false);
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="leftMouseDblClick">
    private volatile ClassNode leftMouseDblClick;
    
    /**
     * getLeftMouseDblClick().fetch( nodeData ) обрабатываются следующие типы результатов:
     * <ul>
     * <li>Func3 - Func3&lt;Object,MouseEvent,TreeTableNodeBasic,<i>NodeData</i>&gt;
     * <li>Func2 - Func2&lt;Object,TreeTableNodeBasic,<i>NodeData</i>&gt;
     * <li>Func1 - Func1&lt;Object,<i>NodeData</i>&gt;
     * <li>Reciver - Reciver&lt;<i>NodeData</i>&gt;
     * <li>Runnable;
     * </ul>
     * @return Обработчики leftMouseDoubleClick
     */
    public ClassNode getLeftMouseDblClick(){
        if( leftMouseDblClick!=null )return leftMouseDblClick;
        synchronized(this){
            if( leftMouseDblClick!=null )return leftMouseDblClick;
            leftMouseDblClick = new ClassNode();
            return leftMouseDblClick;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="leftMouseClick">    
    private volatile ClassNode leftMouseClick;
    
    /**
     * getLeftMouseClick().fetch( nodeData ) обрабатываются следующие типы результатов:
     * <ul>
     * <li>Func3 - Func3&lt;Object,MouseEvent,TreeTableNodeBasic,<i>NodeData</i>&gt;
     * <li>Func2 - Func2&lt;Object,TreeTableNodeBasic,<i>NodeData</i>&gt;
     * <li>Func1 - Func1&lt;Object,<i>NodeData</i>&gt;
     * <li>Reciver - Reciver&lt;<i>NodeData</i>&gt;
     * <li>Runnable;
     * </ul>
     * @return Обработчики leftMouseDoubleClick
     */
    public ClassNode getLeftMouseClick(){
        if( leftMouseClick!=null )return leftMouseClick;
        synchronized(this){
            if( leftMouseClick!=null )return leftMouseClick;
            leftMouseClick = new ClassNode();
            return leftMouseClick;
        }
    }
    //</editor-fold>
    
    protected void onMouseClick(MouseEvent me){
        if( me==null )return;
        TreeTableNodeBasic fnode = (TreeTableNodeBasic)getFocusedNode();
        Object nodeData = fnode!=null ? fnode.getData() : null;
        
        if( me.getClickCount()>1 && me.getButton()==MouseEvent.BUTTON1 && nodeData!=null ){
            processMouseEventForFocusedNode(me, fnode, nodeData, getLeftMouseDblClick().fetch(nodeData));
        }else if( me.getButton()==MouseEvent.BUTTON1 && nodeData!=null ){
            processMouseEventForFocusedNode(me, fnode, nodeData, getLeftMouseClick().fetch(nodeData));
        }
    }
    
    private void processMouseEventForFocusedNode( MouseEvent me, TreeTableNodeBasic fnode, Object nodeData, Iterable iter ){
        if( iter==null )return;
        if( nodeData!=null ){
            for( Object consumer : iter ){
                if( consumer instanceof Func3 ){
                    ((Func3)consumer).apply( me, fnode, nodeData);
                } else if( consumer instanceof Func2 ){
                    ((Func2)consumer).apply(fnode, nodeData);
                }else if( consumer instanceof Func1 ){
                    ((Func2)consumer).apply(fnode, nodeData);
                }else if( consumer instanceof Reciver ){
                    ((Reciver)consumer).recive(nodeData);
                }else if( consumer instanceof Runnable ){
                    ((Runnable)consumer).run();
                }
            }
        }
    }

    /**
     * Конструирование контекста для определнного типа данных узла дерева
     * @param <T> тип данных
     * @param cls тип данных
     * @return конструктор контекста
     */
    public <T> Context<T> context(Class<T> cls) {
        return new ContextBuilder<>(this, cls);
    }
}
