/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.tree.ob;

import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.KeyStroke;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.gui.swing.BasicAction;

/**
 * Плагин для ObjectBrowser - обновление дочерних узлов по запросу
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class RefreshObPlugin implements ObjectBrowserPlugin
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(RefreshObPlugin.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(RefreshObPlugin.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(RefreshObPlugin.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(RefreshObPlugin.class.getName(), method, result);
    }
    //</editor-fold>
    
    protected static String refreshText;
    protected static String rebuildTreeText;
    
    static {
        refreshText = "refresh";
        rebuildTreeText = "rebuild tree";
    }

    @Override
    public void register(final ObjectBrowser bob, CloseableSet cs) {
        if( bob==null )return;
        
        refreshSelected(bob, cs);
        rebuildTree(bob, cs);
    }

    private void refreshSelected(final ObjectBrowser bob, CloseableSet cs) {
        final BasicAction refreshFocusedAction = new BasicAction(
            refreshText, 
            new Runnable() {
                @Override
                public void run() {
                    bob.refreshFocused();
                }
            }
        );
        
        refreshFocusedAction.setSmallIcon(FormatMap.getRefreshIconSmall());
        refreshFocusedAction.setShortDescription("CONTROL+R or F5");
        
        final KeyStroke[] keyStrokes = new KeyStroke[]{
            KeyStroke.getKeyStroke("control pressed R"),
            KeyStroke.getKeyStroke("pressed F5")
        };
        
        final String actId = "refreshFocused";
        
        bob.getActionMap().put(actId, refreshFocusedAction);
        
        for( KeyStroke ks : keyStrokes ){
            bob.getInputMap().put(ks, actId);
        }
        
        if( cs!=null ){
            cs.add(new Runnable() {
                @Override
                public void run() {
                    for( KeyStroke ks : keyStrokes ){
                        bob.getInputMap().remove(ks);
                    }
                    bob.getActionMap().remove(actId);
                }
            });
        }
    }

    private void rebuildTree(final ObjectBrowser bob, CloseableSet cs) {
        final BasicAction rebuildTreeAction = new BasicAction(
            rebuildTreeText, 
            new Runnable() {
                @Override
                public void run() {
                    bob.rebuildTree();
                }
            }
        );
        
        rebuildTreeAction.setSmallIcon(FormatMap.getRefreshIconSmall());
        rebuildTreeAction.setShortDescription("CONTROL+SHIFT+R");
        
        final KeyStroke[] keyStrokes = new KeyStroke[]{
            KeyStroke.getKeyStroke("control shift pressed R"),
        };
        
        final String actId = "rebuildTree";
        
        bob.getActionMap().put(actId, rebuildTreeAction);
        
        for( KeyStroke ks : keyStrokes ){
            bob.getInputMap().put(ks, actId);
        }
        
        if( cs!=null ){
            cs.add(new Runnable() {
                @Override
                public void run() {
                    for( KeyStroke ks : keyStrokes ){
                        bob.getInputMap().remove(ks);
                    }
                    bob.getActionMap().remove(actId);
                }
            });
        }
    }
}
