/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.time;


import java.text.DateFormatSymbols;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.BasicTriple;
import xyz.cofe.collection.Triple;
import static xyz.cofe.time.DateHour.getDaysOfYear;

/**
 * календарный день
 * @author user
 */
public class Day 
    implements Comparable<Day>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(Day.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(Day.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(Day.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(Day.class.getName(), method, result);
    }
    //</editor-fold>

    /**
     * Создает указанный день
     * @param year год
     * @param month месяц (1..12)
     * @param date день (1..31)
     */
    public Day(int year, int month, int date){
        validate(year, month, date, true);

        this.year = year;
        this.month = month;
        this.date = date;
    }
    
    /**
     * Конструктор копирования
     * @param sample образец для копирования
     */
    public Day( Day sample ){
        if( sample==null )throw new IllegalArgumentException("sample == null");
        this.year = sample.year;
        this.month = sample.month;
        this.date = sample.date;
    }
    
    /**
     * Создает календарный день исходя из текущей даты
     */
    public Day(){
        GregorianCalendar gcal = new GregorianCalendar();
        this.year = gcal.get(Calendar.YEAR);
        this.month = gcal.get(Calendar.MONTH)+1;
        this.date = gcal.get(Calendar.DATE);
    }
    
    /**
     * Проверяет на правильность указания даты
     * @param year год
     * @param month месяц (1..12)
     * @param date день (1..31)
     * @param throwException true - негерировать исключение
     * @return правильна или нет дата
     */
    public static String validate( int year, int month, int date, boolean throwException ){
        if( month<1 ){
            if( throwException )throw new IllegalArgumentException("month("+month+")<1");
            return "month("+month+")<1";
        }
        if( month>12 ){
            if( throwException )throw new IllegalArgumentException("month("+month+")>12");
            return "month("+month+")>12";
        }

        if( date<1 ){
            if( throwException )throw new IllegalArgumentException("date("+date+")<1");
            return "date("+date+")<1";
        }
        if( date>31 ){
            if( throwException )throw new IllegalArgumentException("date("+date+")>31");
            return "date("+date+")>31";
        }

        int ml = getMonthLength(year, month);
        if( date>ml ){
            if( throwException )throw new IllegalArgumentException("date("+date+")>"+ml);
            return "date("+date+")>"+ml;
        }
        
        return null;
    }

    /**
     * День месяца 1..31
     */
    protected int date;
    
    /**
     * Возвращает день месяца (1..31)
     * @return день месяца (1..31)
     */
    public int getDate() { return date; }

    /**
     * Номер месяца 1..12
     */
    protected int month;
    
    /**
     * Возвращает номер месяца (1..12)
     * @return номер месяца (1..12)
     */
    public int getMonth() { return month; }

    /**
     * Год
     */
    protected int year;
    
    /**
     * Возвращает год
     * @return год
     */
    public int getYear() { return year; }

    /**
     * Сравнивает на совпадение даты с указанным временем
     * @param date время
     * @return true - есть совпадение
     */
    public boolean match( Date date ){
        if( date==null )throw new IllegalArgumentException("date == null");
        Calendar cal = new GregorianCalendar();
        cal.setTime(date);
        return match( cal );
    }

    /**
     * Сравнивает на совпадение даты с указанным временем
     * @param cal время
     * @return true - есть совпадение
     */
    public boolean match( Calendar cal ){
        if( cal==null )throw new IllegalArgumentException("cal == null");
        int cy = cal.get(Calendar.YEAR);
        if( year!=cy )return false;
        
        int cm = cal.get(Calendar.MONTH)+1;
        if( month!=cm )return false;
        
        int cd = cal.get(Calendar.DATE);
        if( date!=cd )return false;
        
        return true;
    }
    
    /**
	 * Возвращает признак год високосный год.
	 * Согласно Григорианскому (c 1583) / Юлианскому (до 1583) календарю.
	 * @param year Год
	 * @return true - високосный год
	 */
	public static boolean isLeapYear(int year){
		if( year>1582 ){
			int mod4 = year % 4;
			int mod100 = year % 100;
			int mod400 = year % 400;
			return (mod4==0 && mod100!=0) || (mod400==0);
		}else{
			int mod4 = year % 4;
			return mod4==0;
		}
	}
    
    /**
	 * Возвращает кол-во дней в году с учетем високосного года
	 * @param year Год
	 * @return Кол-во дней в году
	 */
	public static int getDaysOfYear(int year){
		if( isLeapYear(year) )return 366;
		return 365;
	}
    
    /**
	 * Возвращает кол-во дней в месяце
	 * @param year Год
	 * @param month Месяц (1 - январь;)
	 * @return Кол-во дней или -1 если не правильно указан месяц
	 */
	public static int getMonthLength(int year,int month){
		boolean ly = isLeapYear(year);
		switch( month ){
		case 1: return 31;
		case 2:
			if( ly )return 29;
			return 28;
		case 3: return 31;
		case 4: return 30;
		case 5: return 31;
		case 6: return 30;
		case 7: return 31;
		case 8: return 31;
		case 9: return 30;
		case 10: return 31;
		case 11: return 30;
		case 12: return 31;
		}
		return -1;
	}
    
    protected transient volatile Integer dayOfWeek;
    
    /**
     * Возвращает день недели 0 - пн; 1 - вт; ... 6 - вс.
     * @return день недели 0 - пн; 1 - вт; ... 6 - вс.
     */
    public int getDayOfWeek(){
        if( dayOfWeek!=null )return dayOfWeek;
        synchronized(this){
            if( dayOfWeek!=null )return dayOfWeek;
            dayOfWeek = getDayOfWeek(year, month, date);
            return dayOfWeek;
        }
    }
    
	/**
	 * День недели 0 - пн; 1 - вт; ... 6 - вс.
     * @param year год
     * @param month месяц (1..12)
     * @param date день месяца (1..31)
	 * @return День недели (0 - пн; 1 - вт; ... 6 - вс.)
	 */
	public static int getDayOfWeek(int year, int month, int date){
        if( year>1582) {
            int a = (14 - month) / 12;
            int y = year - a;
            int m = month + 12 * a - 2;
            int d1 = (7000 + (date + y + y/4 - y/100 + y/400 + (31*m)/12)) % 7;

            int d2 = 0;
            switch(d1){
            case 0: d2=6; break;
            case 1: d2=0; break;
            case 2: d2=1; break;
            case 3: d2=2; break;
            case 4: d2=3; break;
            case 5: d2=4; break;
            case 6: d2=5; break;
            }
            return d2;
        }else{
            int a = (14 - month) / 12;
            int y = year - a;
            int m = month + 12 * a - 2;
            int d1 = ((7000 + (date + y + y/4 + (31*m)/12)) - 2) % 7;

            int d2 = 0;
            switch(d1){
            case 0: d2=6; break;
            case 1: d2=0; break;
            case 2: d2=1; break;
            case 3: d2=2; break;
            case 4: d2=3; break;
            case 5: d2=4; break;
            case 6: d2=5; break;
            }
            return d2;
        }
	}

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 41 * hash + this.date;
        hash = 41 * hash + this.month;
        hash = 41 * hash + this.year;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Day other = (Day) obj;
        if (this.date != other.date) {
            return false;
        }
        if (this.month != other.month) {
            return false;
        }
        if (this.year != other.year) {
            return false;
        }
        return true;
    }

    @Override
    public int compareTo(Day d) {
        if( d==null )return -1;
        
        if( year < d.getYear() )return -1;
        if( year > d.getYear() )return 1;
        
        if( month < d.getMonth() )return -1;
        if( month > d.getMonth() )return 1;
        
        if( date < d.getDate() )return -1;
        if( date > d.getDate() )return 1;
        
        return 0;
    }
    
    /**
     * Возвращает дату на начало дня
     * @return дата
     */
    public Date getBeginDate(){
        GregorianCalendar gcal = new GregorianCalendar(year, month-1, date);
        return gcal.getTime();
    }
    
    /**
     * Возвращает следующий календарный день
     * @param year год
     * @param month месяц 1..12
     * @param date дата 1..31
     * @return календарный день (y-m-d)
     */
    public static Triple<Integer,Integer,Integer> nextDay( int year, int month, int date ){
        validate(year, month, date, true);
        
        int mlen = getMonthLength(year, month);
        if( date<mlen )return new BasicTriple<>(year,month,date+1);        
        if( month<12 )return new BasicTriple<>(year,month+1,1);
        return new BasicTriple<>(year+1,1,1);
    }
    
    /**
     * Возвращает следующий календарный день
     * @param yearMonDate тройка: год, месяц (1..12), день (1..31)
     * @return календарный день (y-m-d)
     */
    public static Triple<Integer,Integer,Integer> nextDay( Triple<Integer,Integer,Integer> yearMonDate ){
        if( yearMonDate==null )throw new IllegalArgumentException("yearMonDate == null");
        if( yearMonDate.A()==null )throw new IllegalArgumentException("yearMonDate.A() == null");
        if( yearMonDate.B()==null )throw new IllegalArgumentException("yearMonDate.B() == null");
        if( yearMonDate.C()==null )throw new IllegalArgumentException("yearMonDate.C() == null");
        return nextDay(yearMonDate.A(), yearMonDate.B(), yearMonDate.C());
    }

    /**
     * Возвращает следующий календарный день
     * @return день
     */
    public Day getNextDay(){
        Triple<Integer,Integer,Integer> t = nextDay(year, month, date);
        return new Day(t.A(), t.B(), t.C());
    }
    
    /**
     * Возвращает предыдущий календарный день
     * @param year год
     * @param month месяц 1..12
     * @param date дата 1..31
     * @return календарный день (y-m-d)
     */    
    public static Triple<Integer,Integer,Integer> prevDay( int year, int month, int date ){
        validate(year, month, date, true);
        if( date>1 )return new BasicTriple<>(year,month,date-1);        
        if( month>1 ){
            return new BasicTriple<>(year,month-1,getMonthLength(year, month-1));
        }
        return new BasicTriple<>(year-1,12,getMonthLength(year-1, 12));
    }
    
    /**
     * Возвращает предыдущий календарный день
     * @param yearMonDate тройка: год, месяц (1..12), день (1..31)
     * @return календарный день (y-m-d)
     */
    public static Triple<Integer,Integer,Integer> prevDay( Triple<Integer,Integer,Integer> yearMonDate ){
        if( yearMonDate==null )throw new IllegalArgumentException("yearMonDate == null");
        if( yearMonDate.A()==null )throw new IllegalArgumentException("yearMonDate.A() == null");
        if( yearMonDate.B()==null )throw new IllegalArgumentException("yearMonDate.B() == null");
        if( yearMonDate.C()==null )throw new IllegalArgumentException("yearMonDate.C() == null");
        return prevDay(yearMonDate.A(), yearMonDate.B(), yearMonDate.C());
    }

    /**
     * Возвращает предыдущий календарный день
     * @return день
     */
    public Day getPrevDay(){
        Triple<Integer,Integer,Integer> t = prevDay(year, month, date);
        return new Day(t.A(), t.B(), t.C());
    }
    
    protected transient volatile GregorianCalendar gregorianCalendar;
    
    /**
     * Возвращает грегорианский календарь
     * @return грегорианский календарь
     */
    protected GregorianCalendar getCalendar(){
        if( gregorianCalendar!=null )return gregorianCalendar;
        synchronized(this){
            if( gregorianCalendar!=null )return gregorianCalendar;
            gregorianCalendar = new GregorianCalendar(year, month-1, date);
            return gregorianCalendar;
        }
    }
    
    /**
     * Возвращает номер недели в году
     * @return номер недели в году
     */
    public int getWeekOfYear(){
        return getCalendar().get( GregorianCalendar.WEEK_OF_YEAR );
    }
    
    /**
     * Возвращает номер недели в месяце
     * @return номер недели в месяце
     */
    public int getWeekOfMonth(){
        return getCalendar().get( GregorianCalendar.WEEK_OF_MONTH );
    }
    
    /**
     * Возвращает первый день месяца
     * @return первый день месяца
     */
    public Day getFirstDayOfMonth(){
        return new Day(year, month, 1);
    }
    
    public boolean isFirstDayOfWeek(){
        int weekDay = getDayOfWeek();
        return weekDay == 0;
    }
    
    public boolean isLastDayOfWeek(){
        int weekDay = getDayOfWeek();
        return weekDay == 6;
    }

    public boolean isFirstDayOfMonth(){
        return date == 1;
    }
    
    public boolean isLastDayOfMonth(){
        return date == getMonthLength();
    }
    
    /**
     * Возвращает первый день недели
     * @return первый день недели
     */
    public Day getFirstDayOfWeek(){
        int weekDay = getDayOfWeek();
        if( weekDay==0 )return this;
        if( weekDay>0 ){
            Day d = this;
            for( int i=0; i<weekDay; i++ ){
                d = d.getPrevDay();
            }
            return d;
        }
        return this;
    }
    
    public static String[] getDayNames(){ return getDayNames(false,null); }
    public static String[] getDayNames(boolean shortNames){ return getDayNames(shortNames,null); }
    public static String[] getDayNames(boolean shortNames, Locale loc){
        DateFormatSymbols dfs = loc!=null ? DateFormatSymbols.getInstance(loc) : DateFormatSymbols.getInstance();
        String[] names = shortNames ? dfs.getShortWeekdays() : dfs.getWeekdays();
        /*int shift = 2;
        return new String[]{ 
            names[shift%7], names[(shift+1)%7], names[(shift+2)%7],
            names[(shift+3)%7], names[(shift+4)%7], names[(shift+5)%7],
            names[(shift+6)%7]
        };*/
        return new String[]{ 
            names[Calendar.MONDAY], names[Calendar.TUESDAY], names[Calendar.WEDNESDAY],
            names[Calendar.THURSDAY], names[Calendar.FRIDAY], names[Calendar.SATURDAY],
            names[Calendar.SUNDAY]
        };
    }
    
    public String getMonthName(){ return getMonthName(false); }
    public String getMonthName(boolean shortName){ return getMonthName(shortName, null); }
    public String getMonthName(boolean shortName, Locale loc){
        DateFormatSymbols dfs = loc!=null ? DateFormatSymbols.getInstance(loc) : DateFormatSymbols.getInstance();
        String[] mons = shortName ? dfs.getShortMonths() : dfs.getMonths();
        return mons[month-1];
    }
    
    public String getDayName(){ return getDayName(false); }
    public String getDayName(boolean shortName){ return getDayName(shortName, null); }
    public String getDayName(boolean shortName, Locale loc){
//        DateFormatSymbols dfs = loc!=null ? DateFormatSymbols.getInstance(loc) : DateFormatSymbols.getInstance();
//        String[] names = shortName ? dfs.getShortWeekdays() : dfs.getWeekdays();
        int dw = getDayOfWeek();
        return getDayNames(shortName, loc)[dw];
    }
    
    @Override
    public String toString(){
        return ""+year+"-"+getMonthName(true, null)+"-"+getDate()+" "+getDayName(true, null)+"";
    }
    
    /**
     * Возвращает кол-во дней в месяце
     * @return кол-во дней
     */
    public int getMonthLength(){
        return getMonthLength(year, month);
    }
}
