/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.common;


import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Predicate;
import xyz.cofe.collection.list.IndexEventList;

// TODO Сделать сохранение / восстановление xml - ok
/**
 * Набор фрагментов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class Fragments
implements Iterable<Fragment>, List<Fragment>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(Fragments.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(Fragments.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(Fragments.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(Fragments.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(Fragments.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(Fragments.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(Fragments.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected final Lock lock;
    protected final Lock listenersLock;

    public Fragments(){
        this.lock = null;
        this.listenersLock = null;
    }

    public Fragments(Lock lock, Lock listenersLock){
        this.lock = lock;
        this.listenersLock = listenersLock;
    }

    public Fragments( Iterable<Fragment> source ){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        this.lock = null;
        this.listenersLock = null;

        for( Fragment f : source ){
            if( f!=null )getFragments().add(f);
        }
    }

    public Fragments( Iterable<Fragment> source,Lock lock, Lock listenersLock ){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        this.lock = lock;
        this.listenersLock = listenersLock;

        for( Fragment f : source ){
            if( f!=null )getFragments().add(f);
        }
    }

    public Fragments( Fragments source, Lock lock, Lock listenersLock ){
        if( source==null )throw new IllegalArgumentException( "source==null" );

        this.lock = lock;
        this.listenersLock = listenersLock;

        for( Fragment f : source ){
            if( f!=null )getFragments().add(f.clone());
        }
    }

    @Override
    public Fragments clone(){
        return new Fragments(this, lock, listenersLock);
    }

    public Lock getLock(){ return lock; }

    //<editor-fold defaultstate="collapsed" desc="listeners">
    public interface Listener {
        public void fragmentEvent( Event event );
    }

    protected ListenersHelper<Listener,Event> listeners = null;
    protected ListenersHelper<Listener,Event> listeners(){
        if( listeners!=null )return listeners;
        listeners = new ListenersHelper<Listener, Event>(new Func2<Object, Listener, Event>()
        {
            @Override
            public Object apply(Listener lst, Event e) {
                lst.fragmentEvent(e);
                return null;
            }
        },
        listenersLock
        );
        return listeners;
    }

    public Set<Listener> getListeners() {
        return listeners().getListeners();
    }

    public Closeable addListener(Listener listener) {
        return listeners().addListener(listener);
    }

    public Closeable addListener(Listener listener, boolean weakLink) {
        return listeners().addListener(listener, weakLink);
    }

    public void removeListener(Listener listener) {
        listeners().removeListener(listener);
    }

    public void fireEvent(Event event) {
        listeners().fireEvent(event);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="eventQu">
    protected Queue<Event> eventQu = new LinkedBlockingQueue<Event>();
    protected void fireEventQu(){
//        if( lock!=null ){
//            lock.lock();
//            try{
//                fireEventQu0();
//            }finally{
//                lock.unlock();
//            }
//        }else{
            fireEventQu0();
//        }
    }
    private void fireEventQu0(){
        while( true ){
            Event e = eventQu.poll();
            if( e==null )break;
            fireEvent(e);
        }
    }

    protected final AtomicBoolean syncFireListeners = new AtomicBoolean(true);

    /**
     * Обновление свойств с учетом блокировок и много поточности.
     * Нужно, чтоб listeners вызывались после снятия блокировки.
     * @param <T> Тип результата
     * @param code Код обновления свойств
     * @return Результат
     */
    protected <T> T syncRun( Func0<T> code ){
        if( code==null )throw new IllegalArgumentException( "code==null" );
        T res = null;
        try{
//            synchronized( this ){
                syncFireListeners.set(false);
                if( lock!=null ){
                    lock.lock();
                    try{
                        res = code.apply();
                    }finally{
                        lock.unlock();
                    }
                }else{
                    res = code.apply();
                }
//            }
        }finally{
            syncFireListeners.set(true);
        }
        fireEventQu();
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="Event">
    public class Event {
        protected Fragments source = null;
        public Event(Fragments source){
            this.source = source;
        }

        public Fragments getSource() {
            return source;
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="fragments">
    private IndexEventList<Fragment> fragments = null;
    private final List<Fragment> getFragments(){
        if( fragments!=null )return fragments;

        if( lock!=null ){
            lock.lock();
            try{
                List<Fragment> lfragments = new ArrayList<Fragment>();
                IndexEventList<Fragment> evList = new IndexEventList<Fragment>(lfragments);
                fragments = evList;
            }
            finally{
                lock.unlock();
            }
        }else{
            List<Fragment> lfragments = new ArrayList<Fragment>();
            IndexEventList<Fragment> evList = new IndexEventList<Fragment>(lfragments);
            fragments = evList;
        }
        return fragments;
    }

    @Override
    public Iterator<Fragment> iterator(){
        ArrayList<Fragment> l = new ArrayList<Fragment>();
        if( lock!=null ){
            lock.lock();
            try{
                l.addAll(getFragments());
            }finally{
                lock.unlock();
            }
        }else{
            l.addAll(getFragments());
        }
        return l.iterator();
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="list functions">

    @Override
    public ListIterator<Fragment> listIterator() {
        return this.<ListIterator<Fragment>>syncRun(new Func0<ListIterator<Fragment>>() {
            @Override
            public ListIterator<Fragment> apply() {
                return getFragments().listIterator();
            }});
    }

    @Override
    public ListIterator<Fragment> listIterator(final int index) {
        return this.<ListIterator<Fragment>>syncRun(new Func0<ListIterator<Fragment>>() {
            @Override
            public ListIterator<Fragment> apply() {
                return getFragments().listIterator(index);
            }});
    }

    @Override
    public int indexOf(final Object o) {
        return this.<Integer>syncRun(new Func0<Integer>() {
            @Override
            public Integer apply() {
                return getFragments().indexOf(o);
            }});
    }

    @Override
    public List<Fragment> subList(final int fromIndex,final int toIndexExclusive) {
        return this.<List<Fragment>>syncRun(new Func0<List<Fragment>>() {
            @Override
            public List<Fragment> apply() {
                return getFragments().subList(fromIndex, toIndexExclusive);
            }});
    }

    @Override
    public boolean add(final Fragment e) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().add(e);
            }});
    }

    @Override
    public boolean remove(final Object o) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().remove(o);
            }});
    }

    @Override
    public void add(final int index, final Fragment element) {
        this.<Object>syncRun(new Func0<Object>() {
            @Override
            public Integer apply() {
                getFragments().add(index, element);
                return null;
            }});
    }

    @Override
    public boolean addAll(final Collection<? extends Fragment> c) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().addAll(c);
            }});
    }

    @Override
    public boolean addAll(final int index, final Collection<? extends Fragment> c) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().addAll(index, c);
            }});
    }

    @Override
    public void clear() {
        this.<Integer>syncRun(new Func0<Integer>() {
            @Override
            public Integer apply() {
                getFragments().clear();
                return -1;
            }});
    }

    @Override
    public Fragment remove(final int index) {
        return this.<Fragment>syncRun(
            new Func0<Fragment>() {
            @Override
            public Fragment apply() {
                return getFragments().remove(index);
            }});
    }

    @Override
    public boolean removeAll(final Collection<?> c) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().removeAll(c);
            }});
    }

    @Override
    public boolean retainAll(final Collection<?> c) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().retainAll(c);
            }});
    }

    @Override
    public Fragment set(final int index, final Fragment element) {
        return this.<Fragment>syncRun(new Func0<Fragment>() {
            @Override
            public Fragment apply() {
                return getFragments().set(index, element);
            }});
    }

    @Override
    public boolean contains(final Object o) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
        return getFragments().contains(o);
            }});
    }

    @Override
    public boolean containsAll(final Collection<?> c) {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().containsAll(c);
            }});
    }

    @Override
    public Fragment get(final int index) {
        return this.<Fragment>syncRun(new Func0<Fragment>() {
            @Override
            public Fragment apply() {
                return getFragments().get(index);
            }});
    }

    @Override
    public boolean isEmpty() {
        return this.<Boolean>syncRun(new Func0<Boolean>() {
            @Override
            public Boolean apply() {
                return getFragments().isEmpty();
            }});
    }

    @Override
    public int lastIndexOf(final Object o) {
        return this.<Integer>syncRun(new Func0<Integer>() {
            @Override
            public Integer apply() {
                return getFragments().lastIndexOf(o);
            }});
    }

    @Override
    public int size() {
        return this.<Integer>syncRun(new Func0<Integer>() {
            @Override
            public Integer apply() {
                return getFragments().size();
            }
        });
    }

    @Override
    public Object[] toArray() {
        return this.<Object[]>syncRun(new Func0<Object[]>() {
            @Override
            public Object[] apply() {
                return getFragments().toArray();
            }
        });
    }

    @Override
    public <T> T[] toArray(T[] a) {
        final T[] b = a;
        T[] res = this.<T[]>syncRun(new Func0<T[]>() {
            @Override
            public T[] apply() {
                return getFragments().toArray(b);
            }
        });
        return res;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="removeSameRange">
    public void removeSameRange( final Fragment fragment )
    {
        syncRun(new Func0(){
            @Override
            public Object apply() {
                if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
                for( Fragment f : Fragments.this ){
                    if( f.equalsRange(fragment) ){
                        remove(f);
                    }
                }
                return null;
            }
        });
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="toString">
    @Override
    public String toString(){
        return (String)syncRun( new Func0(){
            @Override
            public Object apply() {
                StringBuilder s = new StringBuilder();
                int i = -1;
                for( Fragment f : Fragments.this ){
                    i++;
                    s.append(i).append(". ").append(f.toString()).append("\n");
                }
                return s.toString();
            }
        });
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="comparators">
    protected static final Comparator<Fragment> sorterByBegin = new Comparator<Fragment>() {
        @Override
        public int compare(Fragment o1, Fragment o2) {
            long l1 = o1.getBegin();
            long l2 = o2.getBegin();
            return l1==l2 ? 0 : (l1 < l2 ? -1 : 1);
        }
    };

    protected static final Comparator<Fragment> sorterBySize = new Comparator<Fragment>() {
        @Override
        public int compare(Fragment o1, Fragment o2) {
            long s1 = o1.getSize();
            long s2 = o2.getSize();
            return s1==s2 ? 0 : (s1 < s2 ? -1 : 1);
        }
    };

    protected static final Comparator<Fragment> sorterByInstanceID = new Comparator<Fragment>() {
        @Override
        public int compare(Fragment o1, Fragment o2) {
            long s1 = o1.instanceID;
            long s2 = o2.instanceID;
            return s1==s2 ? 0 : (s1 < s2 ? -1 : 1);
        }
    };

    protected static final Comparator<Fragment> defSorter = new Comparator<Fragment>() {
        @Override
        public int compare(Fragment o1, Fragment o2) {
            int cBegin = sorterByBegin.compare(o1, o2);
            int cSize = sorterBySize.compare(o1, o2);
            int cID = sorterByInstanceID.compare(o1, o2);
            int cmp = cBegin * 10 + cSize * 5 + cID;
            return cmp;
        }
    };
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="sizeEqualsPredicate( size )">
    public static Predicate<Fragment> sizeEqualsPredicate( final long size ){
        return new Predicate<Fragment>() {
            @Override
            public boolean validate(Fragment f) {
                return f!=null && f.getSize() == size;
            }
        };
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="rangePredicate( begin, end)">
    public static Predicate<Fragment> rangePredicate( final long begin, final long end ){
        return new Predicate<Fragment>() {
            protected ImmutableFragment r = new ImmutableFragment(begin, end);
            @Override
            public boolean validate(Fragment f) {
                return f!=null && f.equalsRange(r);
            }
        };
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="find(Predicate<Fragment> p)">
    /**
     * Поиск фрагментов по критерию
     * @param p критерий
     * @return найденые фрагменты
     */
    public List<Fragment> find(final Predicate<Fragment> p){
        return (List<Fragment>)syncRun(new Func0() {
            @Override
            public Object apply() {
                List l = new ArrayList();
                for( Fragment f : Fragments.this ){
                    if( p.validate(f) )l.add(f);
                }
                return l;
            }
        });
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="sort()">
    /**
     * Сортировка
     */
    public void sort(){
        syncRun(new Func0() {
            @Override
            public Object apply() {
                Collections.sort(getFragments(), defSorter);
                return null;
            }
        });
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="findZeroSize()">
    /**
     * Поиск фрагментов нулей длинны
     * @return нулевой длины элементы
     */
    public List<Fragment> findZeroSize(){
        return find( sizeEqualsPredicate(0) );
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createFagment( begin, end )">
    protected Fragment createFagment( long begin, long end ){
        return new Fragment(begin, end);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="defCreateFragment">
    protected Func2<Fragment,Long,Long> defCreateFragment = new Func2<Fragment, Long, Long>() {
        @Override
        public Fragment apply(Long begin, Long end) {
            return createFagment(begin, end);
        }
    };
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isEquals( f1, f2 )">
    protected boolean isEquals( Fragment f1, Fragment f2 ){
        if( f1==null )throw new IllegalArgumentException( "f1==null" );
        if( f2==null )throw new IllegalArgumentException( "f2==null" );
        return f1.equals(f2);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="visitEachEachWithoutPosition()">
    /**
     * Обход каждый с каждым,
     * без учета позиции в сочетании пар и самого себя. <br>
     * пример: <br>
     * исходно[a, b, c, d] <br>
     * вызовы <br>
     * visitor( a, b )  <br>
     * visitor( a, с )  <br>
     * visitor( a, d )  <br>
     * visitor( b, c )  <br>
     * visitor( b, d )  <br>
     * visitor( c, d )  <br>
     * @param visitor Функция приемник пар
     * @param equals Функция сравнения на равенство
     */
    public void visitEachEachWithoutPosition(
        final Func2<Object,Fragment,Fragment> visitor,
        final Func2<Boolean,Fragment,Fragment> equals
    ){
        syncRun(new Func0() {
            @Override
            public Object apply() {

                if( visitor==null )throw new IllegalArgumentException( "visitor==null" );
                if( equals==null )throw new IllegalArgumentException( "equals==null" );

                Fragments fs = new Fragments(Fragments.this);
                for( Fragment f1 : Fragments.this ){
                    for( Fragment f2 : fs ){
                        boolean eq = equals.apply(f1, f2);
                        if( eq )continue;
                        visitor.apply(f1, f2);
                    }
                    fs.remove(f1);
                }
                return null;
            }});
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="findIntersections()">
    /**
     * Поиск пересечений
     * @return пересечения
     */
    public Map<Fragment,Set<Fragment>> findIntersections(){
        return (Map<Fragment,Set<Fragment>>)syncRun(new Func0() {
            @Override
            public Object apply() {

                final Map<Fragment,Set<Fragment>> m
                        = new LinkedHashMap<Fragment, Set<Fragment>>();

                visitEachEachWithoutPosition(
                        new Func2<Object, Fragment, Fragment>() {
                            @Override
                            public Object apply(Fragment f1, Fragment f2) {
                                if( f1.hasIntersection(f2) ){
                                    Set<Fragment> s = m.get(f1);
                                    if( s==null ){
                                        s = new LinkedHashSet<Fragment>();
                                        m.put(f1, s);
                                    }
                                    s.add(f2);
                                }
                                return null;
                            }
                        },
                        new Func2<Boolean, Fragment, Fragment>() {
                            @Override
                            public Boolean apply(Fragment arg1, Fragment arg2) {
                                return isEquals(arg1, arg2);
                            }}
                );

                return m;
            }});
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="findHoles">
    /**
     * Поиск дыр в указанном диапазоне
     * @param begin начало диапазона
     * @param end конец диапазона искл.
     * @return Фрагменты соответ дырам
     */
    public List<Fragment> findHoles(long begin, long end){
        return findHoles(begin, end, defCreateFragment);
    }

    /**
     * Поиск дыр в указанном диапазоне
     * @param fbegin начало диапазона
     * @param fend конец диапазона искл.
     * @param fcreateFragment Создание фрагмента
     * @return Фрагменты соответ дырам
     */
    public List<Fragment> findHoles(
            final long fbegin,
            final long fend,
            final Func2<Fragment,Long,Long> fcreateFragment
    ){
        return (List<Fragment>)syncRun(new Func0() {
            @Override
            public Object apply() {

                Func2<Fragment,Long,Long> createFragment =
                        ( fcreateFragment==null ) ? defCreateFragment : fcreateFragment;

                long begin = fbegin;
                long end = fend;

                if( begin>end ){
                    long t = begin; begin = end; end = t;
                }

                List<Fragment> holes = new ArrayList<Fragment>();

                long ptr = begin;

                Fragments fragments = new Fragments(Fragments.this);
                fragments.removeAll(fragments.findZeroSize());
                fragments.sort();

                int i = -1;
                int count = fragments.size();
                for( Fragment f : fragments ){
                    i++;
                    boolean last = i == count-1;

                    long b = f.getBegin();
                    long e = f.getEnd();

                    if( !last ){
                        if( ptr<b ){
                            Fragment fHole =
                                    createFragment.apply(ptr, b);
                            holes.add(fHole);
                            ptr = e;
                        }else{
                            ptr = e;
                        }
                    }else{
                        if( ptr<b ){
                            Fragment fHole =
                                    createFragment.apply(ptr, b);
                            holes.add(fHole);
                            ptr = e;
                        }else{
                            ptr = e;
                        }

                        if( ptr<end ){
                            Fragment fHole =
                                    createFragment.apply(ptr, end);
                            holes.add(fHole);
                        }
                    }
                }

                return holes;
            }});
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="removeIntersections()">
    /**
     * Решение пересечения обьеднением
     */
    public final Func2<
        List<Fragment>,
        Fragment,
        Fragment> unionIntersecResolver
        = new Func2<List<Fragment>, Fragment, Fragment>() {
        @Override
        public List<Fragment> apply(Fragment f1, Fragment f2) {
            if( f1==null )throw new IllegalArgumentException( "f1==null" );
            if( f2==null )throw new IllegalArgumentException( "f2==null" );

            if( !f1.hasIntersection(f2) )
                throw new IllegalArgumentException("no intersection");

            List<Fragment> fs = new ArrayList<Fragment>();
            Fragment fu = f1.union(f2);
            fs.add(fu);
            return fs;
        }
    };

    protected final Func2<
        List<Fragment>,
        Fragment,
        Fragment> defIntersecResolver = unionIntersecResolver;

    /**
     * Удаление пересечений, так же удаляет нулевые фрагменты
     */
    public void removeIntersections(){
        removeIntersections(defIntersecResolver);
    }

    /**
     * Удаление пересечений, так же удаляет нулевые фрагменты
     * @param fintersecResolver способ решения пересечений
     */
    public void removeIntersections(
            final Func2<List<Fragment>, Fragment, Fragment> fintersecResolver
    ){
        syncRun(new Func0() {
            @Override
            public Object apply() {
                Func2<List<Fragment>,Fragment,Fragment> intersecResolver = fintersecResolver;
                if( intersecResolver==null )intersecResolver = defIntersecResolver;

                Fragments fs = Fragments.this;
                fs.removeAll(fs.findZeroSize());

                while( true ){
                    Map<Fragment,Set<Fragment>> intersect = fs.findIntersections();
                    if( intersect.isEmpty() )return null;

                    for( Fragment f1 : intersect.keySet() ){
                        Set<Fragment> f2set = intersect.get(f1);
                        for( Fragment f2 : f2set ){
                            List<Fragment> fres = intersecResolver.apply(f1, f2);
                            fs.removeSameRange(f1);
                            fs.removeSameRange(f2);
                            fs.addAll(fres);
                        }
                    }
                }
            }});
    }
//</editor-fold>
}
