/*
 * The MIT License
 *
 * Copyright 2015 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.common;


import java.io.File;
import java.io.FileInputStream;
import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.cbuffer.ContentBuffer;
import xyz.cofe.text.Text;


/**
 * Работа с хэш функциями
 * @author nt.gocha@gmail.com
 */
public class Hash {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(Hash.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(Hash.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(Hash.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(Hash.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(Hash.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(Hash.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(Hash.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    /**
     * блокировка используемая при расчете md5
     */
    protected final Lock lock;

    public Hash(){
        lock = new ReentrantLock();
    }

    /**
     * Конструктор
     * @param lock блокировка используемая при расчете md5
     */
    public Hash(Lock lock){
        if( lock==null )throw new IllegalArgumentException("lock==null");
        this.lock = lock;
    }

    /**
     * блокировка используемая при расчете md5
     * @return блокировка используемая при расчете md5
     */
    public Lock getLock(){ return lock; }

    /**
     * Текстовое hex представление
     * @param hash хэш
     * @return текстовое представление
     */
    public byte[] toBytes( String hash ){
        if( hash==null )throw new IllegalArgumentException( "hash==null" );
        return Text.decodeHex(hash);
    }

    /**
     * Текстовое hex представление
     * @param hash хэш
     * @return текстовое представление
     */
    public String toString( byte[] hash ){
        if( hash==null )throw new IllegalArgumentException( "hash==null" );
        return Text.encodeHex(hash);
    }

    /**
     * Проверка на равенство хэшей
     * @param hash1 первый хэш
     * @param hash2 второй хэш
     * @return true - эквивалентны
     */
    public boolean isEquals( byte[] hash1, byte[] hash2 ){
        if( hash1==null )throw new IllegalArgumentException( "hash1==null" );
        if( hash2==null )throw new IllegalArgumentException( "hash2==null" );
        if( hash1.length!=hash2.length )return false;
        for( int i=0; i<hash1.length; i++ ){
            if( hash1[i]!=hash2[i] )return false;
        }
        return true;
    }

    /**
     * Подсчет sha-1 указанного набора байт
     * @param data массив байтов
     * @param beginIndex начальный индекс (смещение)
     * @param endIndex конечный индекс исключ.
     * @param buffsize размер блока
     * @return хэш
     * @throws java.io.IOException Ошибка IO
     */
    public byte[] sha1( InputStream data, long beginIndex, long endIndex, int buffsize )
            throws
                IOException
    {
        return hash("SHA-1", data, beginIndex, endIndex, buffsize);
    }

    public byte[] sha1( ContentBuffer buffer, ImmutableFragment fragment, int blockSize )
    {
        return sha1( buffer, fragment.getBegin(), fragment.getEnd(), blockSize );
    }

    /**
     * Подсчет md5 указанного набора байт
     * @param buffer массив байтов
     * @param beginIndex начальный индекс (смещение)
     * @param endIndex конечный индекс исключ.
     * @param buffsize размер блока
     * @return хэш
     */
    public byte[] sha1( ContentBuffer buffer, long beginIndex, long endIndex, int buffsize )
    {
        return hash("SHA-1", buffer, beginIndex, endIndex, buffsize);
    }

    /**
     * Подсчет md5 указанного набора байт
     * @param buffer массив байтов
     * @param buffsize размер блока
     * @return хэш
     */
    public byte[] sha1( File buffer, int buffsize )
    {
        try {
            return hash("SHA-1", buffer, buffsize);
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Подсчет md5 указанного набора байт
     * @param buffer массив байтов
     * @return хэш
     */
    public byte[] sha1( File buffer )
    {
        try {
            return hash("SHA-1", buffer);
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    public byte[] sha1( byte[] data, int beginIndex, int endIndex ){
        return hash("SHA-1", data, beginIndex, endIndex);
    }

    public byte[] sha1( byte[] data ){
        if( data==null )throw new IllegalArgumentException( "data==null" );
        return hash("SHA-1", data, 0, data.length);
    }

    
    public byte[] md5( ContentBuffer buffer, ImmutableFragment fragment, int blockSize )
    {
        if( buffer==null )throw new IllegalArgumentException( "buffer==null" );
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
        if( blockSize <= 0 )throw new IllegalArgumentException( "blockSize <= 0" );
        return md5( buffer, fragment.getBegin(), fragment.getEnd(), blockSize );
    }

    /**
     * Подсчет md5 указанного набора байт
     * @param buffer массив байтов
     * @param beginIndex начальный индекс (смещение)
     * @param endIndex конечный индекс исключ.
     * @param buffsize размер блока
     * @return хэш
     */
    public byte[] md5( ContentBuffer buffer, long beginIndex, long endIndex, int buffsize )
    {
        return hash("MD5", buffer, beginIndex, endIndex, buffsize);
    }

    /**
     * Подсчет md5 указанного набора байт
     * @param data массив байтов
     * @param beginIndex начальный индекс (смещение)
     * @param endIndex конечный индекс исключ.
     * @param buffsize размер блока
     * @return хэш
     * @throws java.io.IOException Ошибка IO
     */
    public byte[] md5( InputStream data, long beginIndex, long endIndex, int buffsize )
            throws
//                NoSuchAlgorithmException,
                IOException 
    {
        return hash("MD5", data, beginIndex, endIndex, buffsize);
    }
    
    public byte[] md5( byte[] data, int beginIndex, int endIndex ){
        return hash("MD5", data, beginIndex, endIndex);
    }

    public byte[] md5( byte[] data ){
        if( data==null )throw new IllegalArgumentException( "data==null" );
        return hash("MD5", data, 0, data.length);
    }

    /**
     * Подсчет md5 указанного набора байт
     * @param data массив байтов
     * @param buffsize размер блока
     * @return хэш
     */
    public byte[] md5( File data, int buffsize )
    {
        try {
            return hash("MD5", data,  buffsize);
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Подсчет md5 указанного набора байт
     * @param data массив байтов
     * @return хэш
     */
    public byte[] md5( File data )
    {
        try {
            return hash("MD5", data);
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Подсчет хэша указанного набора байт
     * @param alg алгоритм (MD5/SHA-1)
     * @param buffer массив байтов
     * @param beginIndex начальный индекс (смещение)
     * @param endIndex конечный индекс исключ.
     * @param buffsize размер блока
     * @return хэш
     */
    public byte[] hash( String alg, ContentBuffer buffer, long beginIndex, long endIndex, int buffsize )
//        throws NoSuchAlgorithmException
    {
        lock.lock();
        try{
            if( alg==null )throw new IllegalArgumentException( "alg==null" );
            if( buffer==null )throw new IllegalArgumentException( "buffer==null" );
            if( beginIndex<0 )throw new IllegalArgumentException( "beginIndex<0" );
            if( endIndex<beginIndex )throw new IllegalArgumentException( "endIndex<beginIndex" );
            if( buffsize<=0 )throw new IllegalArgumentException( "buffsize<=0" );

            MessageDigest md;
            try {
                md = MessageDigest.getInstance(alg);
            } catch (NoSuchAlgorithmException ex) {
//                Logger.getLogger(Hash.class.getName()).log(Level.SEVERE, null, ex);
                throw new Error("NoSuchAlgorithmException "+alg,ex);
            }
            md.reset();

            long count = endIndex - beginIndex;
            long totReaded = 0;
            long ptr = beginIndex;
            byte[] buff = null;
            while( true ){
                if( totReaded>=count )break;
                long diff = count - totReaded;
                int readsize = -1;
                if( diff>=buffsize ){
                    readsize = buffsize;
                }else{
                    readsize = (int)diff;
                }
                buff = buffer.get(ptr, readsize);
                ptr += buff.length;
                totReaded += buff.length;
                md.update(buff);
            }

            return md.digest();
        }
        finally{
            lock.unlock();
        }
    }
    
    public byte[] hash( String alg, byte[] data, int beginIndex, int endIndex ){
        if( alg==null )throw new IllegalArgumentException( "alg==null" );
        if( data==null )throw new IllegalArgumentException( "data==null" );
        if( beginIndex<0 )throw new IllegalArgumentException( "beginIndex<0" );
        if( endIndex<beginIndex )throw new IllegalArgumentException( "endIndex<beginIndex" );
        int count = endIndex - beginIndex;

        MessageDigest md;
        try {
            md = MessageDigest.getInstance(alg);
        } catch (NoSuchAlgorithmException ex) {
            throw new Error("NoSuchAlgorithmException "+alg,ex);
        }
        md.reset();
        
        md.update(data, beginIndex, count);
        return md.digest();
    }

    /**
     * Подсчет хэша указанного набора байт
     * @param alg Алгоритм (MD5 / SHA-1)
     * @param data массив байтов
     * @param beginIndex начальный индекс (смещение)
     * @param endIndex конечный индекс исключ.
     * @param buffsize размер блока
     * @return хэш
     * @throws java.io.IOException Ошибка IO
     */
    public byte[] hash( String alg, InputStream data, long beginIndex, long endIndex, int buffsize )
            throws
//                NoSuchAlgorithmException,
                IOException
    {
        lock.lock();
        try{
            if( alg==null )throw new IllegalArgumentException( "alg==null" );
            if( data==null )throw new IllegalArgumentException( "data==null" );
            if( beginIndex<0 )throw new IllegalArgumentException( "beginIndex<0" );
            if( endIndex<beginIndex )throw new IllegalArgumentException( "endIndex<beginIndex" );

            long count = endIndex - beginIndex;
    //        if( count<0 )throw new IllegalArgumentException( "count<0" );

            if( buffsize<=0 )throw new IllegalArgumentException( "buffsize<=0" );

            MessageDigest md;
            try {
                md = MessageDigest.getInstance(alg);
            } catch (NoSuchAlgorithmException ex) {
                throw new Error("NoSuchAlgorithmException "+alg,ex);
            }
            md.reset();

            byte[] buff = new byte[buffsize];

            // пропуск до указанного смещения
            if( beginIndex>0 ){
                long ptr = 0;
                while( true ){
                    long diff = beginIndex - ptr;
                    if( diff<=0 )break;
                    if( diff>=buffsize ){
                        int readed = data.read(buff);
                        if( readed<0 )throw new IOException("end of file");
                        if( readed>0 ){
                            ptr += readed;
                        }
                    }else{
                        int needRead = buffsize - (int)diff;
                        int readed = data.read(buff, 0, needRead);
                        if( readed<0 )throw new IOException("end of file");
                        if( readed>0 ){
                            ptr += readed;
                        }
                    }
                }
            }

            // читаем интересующие байты
            long totalReaded = 0;
            while( true ){
                long diff = count - totalReaded;
                if( diff<=0 )break;
                if( diff>=buffsize ){
                    int readed = data.read(buff);
                    if( readed<0 )throw new IOException("end of file");
                    if( readed>0 ){
                        totalReaded += readed;
                        md.update(buff, 0, readed);
                    }
                }else{
                    int needRead = buffsize - (int)diff;
                    int readed = data.read(buff, 0, needRead);
                    if( readed<0 )throw new IOException("end of file");
                    if( readed>0 ){
                        totalReaded += readed;
                        md.update(buff, 0, readed);
                    }
                }
            }

            return md.digest();
        }
        finally{
            lock.unlock();
        }
    }

    /**
     * Подсчет хэша указанного файла
     * @param alg Алгоритм (MD5 / SHA-1)
     * @param file файл
     * @param buffSize размер блока
     * @return хэш
     * @throws java.io.IOException Ошибка IO
     */
    public byte[] hash( String alg, File file, int buffSize )
        throws IOException
    {
        if( alg==null )throw new IllegalArgumentException( "alg==null" );
        if( file==null )throw new IllegalArgumentException( "file==null" );
        if( buffSize<1 )throw new IllegalArgumentException( "buffSize<1" );

        FileInputStream fin = null;
        try{
            fin = new FileInputStream(file);
            return hash(alg, fin, 0, file.length(), buffSize);
        }finally{
            if( fin!=null )fin.close();
        }
    }

    /**
     * Подсчет хэша указанного файла
     * @param alg Алгоритм (MD5 / SHA-1)
     * @param file файл
     * @return хэш
     * @throws java.io.IOException Ошибка IO
     */
    public byte[] hash( String alg, File file ) throws IOException {
        if( alg==null )throw new IllegalArgumentException( "alg==null" );
        if( file==null )throw new IllegalArgumentException( "file==null" );
        return hash(alg, file, 1024 * 8);
    }
}
