/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.common;

import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.xml.XmlCoder;

/**
 * Фрагмент данных. <br>
 * Связан с картой String / Object. <br>
 * Ключи карты используемые с этом классе: <br>
 * <b>begin</b> : long <br>
 * <b>end</b> : long
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class ImmutableFragment {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(ImmutableFragment.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(ImmutableFragment.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(ImmutableFragment.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(ImmutableFragment.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(ImmutableFragment.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(ImmutableFragment.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(ImmutableFragment.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    public static final AtomicLong sequneceID = new AtomicLong();
    public final long instanceID = sequneceID.incrementAndGet();

    protected volatile long begin;
    protected volatile long end;

    /**
     * Свойства
     */
//    protected Map<String,Object> properties = null;

    /**
     * Конструктор
     * @param begin Начало фрагмента
     * @param end Конец фрагмент Искл.
     */
    public ImmutableFragment(long begin, long end){
//        if( end<begin ){
//            long t = begin; begin = end; end = t;
//        }
//        properties = createProperties();
//        put( BEGIN, begin );
//        put( END, end );
        init(begin, end);
    }

    protected ImmutableFragment(){
    }

    protected void init(long begin,long end){
        if( end<begin ){
            long t = begin; begin = end; end = t;
        }
//        properties = createProperties();
//        put( BEGIN, begin );
//        put( END, end );
        this.begin = begin;
        this.end = end;
    }

    /**
     * Конструктор копирования
     * @param source образец
     */
    public ImmutableFragment(ImmutableFragment source){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        init(source);
    }

    protected void init(ImmutableFragment source){
        if( source==null )throw new IllegalArgumentException( "source==null" );
//        properties = createProperties();
//        properties.putAll(source.properties);
        this.begin = source.begin;
        this.end = source.end;
    }

//    protected Map<String,Object> createProperties(){
//        return new LinkedHashMap();
//    }

    /**
     * Создание клона
     * @return клон
     */
    @Override
    public ImmutableFragment clone(){
        return new ImmutableFragment(this);
    }

    /**
     * Текстовое представление
     * @return текст
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        long b = getBegin();
        long e = getEnd();
        long s = getSize();

        sb.append("[").
            append(b).
            //append(" \u2192 ").
            append(" -> ").
            append(e).
            append("/").
            append(s);
//        for( String k : getKeys() ){
//            if( k==null )continue;
//            Object v = get(k);
//
//            if( v==null )continue;
//            if( k.equals(BEGIN) )continue;
//            if( k.equals(END) )continue;
//
//            sb.append(", ").append(k).append("=").append(v);
//        }
        sb.append("]");
        return sb.toString();
    }

//    /**
//     * Удаление свойства
//     * @param key имя св
//     */
//    protected void remove( Object key ){
//        properties.remove(key);
//    }
//
//    /**
//     * Удаление свойств
//     */
//    protected void clear(){
//        properties.clear();
//    }
//
//    /**
//     * Установка свойства
//     * @param key ключ
//     * @param value значение
//     */
//    protected void put(String key,Object value){
//        properties.put(key, value);
//    }
//
//    /**
//     * Добавление/установка свойств
//     * @param map карта
//     */
//    protected void putAll( Map<String,Object> map ){
//        if( map==null )throw new IllegalArgumentException( "map==null" );
//        properties.putAll(map);
//    }
//
//    /**
//     * Получение списка ключей
//     * @return ключи
//     */
//    public String[] getKeys(){
//        return properties.keySet().toArray( new String[]{} );
//    }
//
//    /**
//     * Получение свойства
//     * @param key имя свойства
//     * @return знаяение или null
//     */
//    public Object get(String key){
//        return properties.get(key);
//    }
//
//    public static final String BEGIN = "begin";

    /**
     * Начало фрагмента
     * @return начало фрагмента
     */
    public long getBegin(){
//        Object v = properties.get(BEGIN);
//        return v instanceof Long ? (long)((Long)v) : (long)0;
        return begin;
    }

//    public static final String END = "end";

    /**
     * Конец фрагмента от нуля (исключительно)
     * @return Конец фрагмента
     */
    public long getEnd(){
//        Object v = properties.get(END);
//        return v instanceof Long ? (long)((Long)v) : (long)0;
        return end;
    }

    /**
     * Размер фрагмента
     * @return размер фрагмента
     */
    @XmlCoder.Ignore
    public long getSize(){
        long v = getEnd() - getBegin();
        return v < 0 ? -v : v;
    }

    /**
     * Создает новый фрагмент
     * @param begin начало, от нуля
     * @param end конец, от нуля исключительно. end должен быть больше или равен begin.
     * @return фрагмент
     */
    public ImmutableFragment range( long begin, long end ){
        if( end<begin ){
            long t = begin; begin = end; end = t;
        }
        ImmutableFragment f = clone();
//        f.put(BEGIN, begin);
//        f.put(END, end);
        f.begin = begin;
        f.end = end;
        return f;
    }

    /**
     * Указание нового размера
     * @param newSize размер &gt;= 0
     * @return клон с новыми размерами
     */
    public ImmutableFragment size( long newSize ){
        if( newSize<0 )throw new IllegalArgumentException("newsize<0");
        ImmutableFragment f = clone();
        long b = f.getBegin();
        long e = b + newSize;
//        f.put(END, e);
        f.end = e;
        return f;
    }

    /**
     * Получение пересечения
     * @param fragment с кем возможно пересечение
     * @return пересечение или null, если его нет
     */
    public ImmutableFragment intersection( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        if( b2 < b1 ){
            long t=b1; b1=b2; b2=t;
            t=e1; e1=e2; e2=t;
        }

        if( b2 >= b1 && b2 < e1 ){
            if( e2 > e1 ){
                return range(b2, e1);
            }else{
                return range(b2, e2);
            }
        }

        return null;
    }

    /**
     * Проверка пересечения
     * @param fragment с кем пересечение
     * @return true - есть пересечение
     */
    public boolean hasIntersection( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );

//        ImmutableFragment intersec = intersection(fragment);
//        if( intersec==null )return null;

        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        if( b2 < b1 ){
            long t=b1; b1=b2; b2=t;
            t=e1; e1=e2; e2=t;
        }

//        if( b2 >= b1 && b2 < e1 ){
//            return true;
//        }

        return b2 >= b1 && b2 < e1;
    }

//
//    public boolean before( ImmutableFragment fragment ){
//        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
//
//        long b1 = getBegin();
//        long e1 = getEnd();
//
//        long b2 = fragment.getBegin();
//        long e2 = fragment.getEnd();
//
//        return b1 == e2;
//    }

    /**
     * Проверка что данный объект расположен вплотную после указанного, но не пересекается - то есть следует:
     * .... [ fragment ] [ this ] ....
     * @param fragment фрагмент после которого следует объект
     * @return объект следует за указанным
     */
    public boolean follow( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );

        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        return b1 == e2;
    }

    /**
     * Проверка что данный объект расположен вплотную перед указанного, но не пересекается:
     * .... [ this ] [ fragment ] ....
     * @param fragment фрагмент перед которым следует объект
     * @return указанный объект следует после объекта
     */
    public boolean follower( ImmutableFragment fragment ){
        return fragment.follow(this);
    }

    /**
     * Объединение фрагментов.
     * фрагменты должны иметь пересечение или следовать вполтную друг за другом
     * @param fragment с кем объединение
     * @return Объединение или null
     */
    public ImmutableFragment union( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );

        if( follow(fragment) || follower(fragment) ){
            long b1 = getBegin();
            long e1 = getEnd();

            long b2 = fragment.getBegin();
            long e2 = fragment.getEnd();

            long minb = b1 > b2 ? b2 : b1;
            long maxe = e1 < e2 ? e2 : e1;

            return range(minb, maxe);
        }

        if( !hasIntersection(fragment) )return null;

        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        long minb = b1 > b2 ? b2 : b1;
        long maxe = e1 < e2 ? e2 : e1;

        return range(minb, maxe);
    }

    /**
     * Проверка включения под фрагмента
     * @param fragment подфрагмент
     * @return true - есть включение
     */
    public boolean include( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );

        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        return b2>=b1 && e2<=e1;
    }

    /**
     * Проверка совпадения диапазона
     * @param fragment фрагмент
     * @return true - совпадение начала и конца диапазона
     */
    public boolean equalsRange( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );

        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        return b1==b2 && e1==e2;
    }

    /**
     * Вычитание фрагмента
     * @param fragment вычитаемый фрагмент: substract( fragment ) = this - fragment
     * @return результат вычитания
     */
    public ImmutableFragment[] substract( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );

        if( fragment.include(this) ){
            // вырез полностью вырезает все берз остатка
            return new ImmutableFragment[]{ size(0) };
        }

        if( equalsRange(fragment) ){
            // вырез полностью вырезает все берз остатка
            return new ImmutableFragment[]{ size(0) };
        }

        if( !hasIntersection(fragment) ){
            // нет пересечения
            return new ImmutableFragment[]{ clone() };
        }

        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        if( this.include(fragment) ){
            // разделение на два фрагмента с вырезом по середине
//            if( b1 < b2 ){
//            }else{
//            }
            return new ImmutableFragment[]{
                range(b1, b2),
                range(e1, e2)
            };
        }else{
            // один фрагмент со срезаным краем
            if( b1 < b2 ){
                // срез с права
                return new ImmutableFragment[]{ range(b1, b2) };
            }else{
                // срез с лева
                return new ImmutableFragment[]{ range(e2, e1) };
            }
        }
    }

    /**
     * Определяет растояние между фрагментами
     * @param fragment фрагмент
     * @return
     * <b>null</b> - если есть пересечения, <br>
     * 0 - если пересечений нет, но фрагменты следуют друг за другом вплотную <br>
     * &lt;0 - если объект следует раньше указанного - указывает на каком растоянии находятся их границы <br>
     * &gt;0 - если объект следует позже указанного - указывает на каком растоянии находятся их границы <br>
     */
    public Long distance( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
        if( hasIntersection(fragment) )return null;

        long b1 = getBegin();
        long e1 = getEnd();

        long b2 = fragment.getBegin();
        long e2 = fragment.getEnd();

        if( b1 < b2 ){
            // следует раньше fragment
            return e1 - b2;
        }else{
            // следует позже fragment
            return b1 - e2;
        }
    }
}
