/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.common;


import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func2;

/**
 * Набор фрагментов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class ImmutableFragments implements Iterable<ImmutableFragment>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(ImmutableFragments.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(ImmutableFragments.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(ImmutableFragments.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(ImmutableFragments.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(ImmutableFragments.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(ImmutableFragments.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(ImmutableFragments.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    /**
     * Конструктор по умолч
     */
    public ImmutableFragments(){
    }
    
    /**
     * Конструктор из фрагментов
     * @param arr фрагменты
     */
    public ImmutableFragments(ImmutableFragment[] arr){
        if( arr==null )throw new IllegalArgumentException( "arr==null" );
        for( int i=0; i<arr.length; i++ ){
            if( arr[i]==null )throw new IllegalArgumentException( "arr["+i+"]==null" );
        }
        this.fragments = arr;
    }

    /**
     * Конструктор из фрагментов
     * @param arr фрагменты
     */
    public ImmutableFragments(Iterable<ImmutableFragment> arr){
        if( arr==null )throw new IllegalArgumentException( "arr==null" );
        ArrayList<ImmutableFragment> l = new ArrayList<ImmutableFragment>();
        int i = -1;
        for( ImmutableFragment f : arr ){
            i++;
            if( f==null )throw new IllegalArgumentException( "arr["+i+"]==null" );
            l.add(f);
        }
        this.fragments = l.toArray(new ImmutableFragment[]{});
    }
    
    /**
     * Конструктор копирования
     * @param source образец
     */
    public ImmutableFragments(ImmutableFragments source){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        this.fragments = source.fragments;
    }
    
    /**
     * Итератор по фрагментам
     */
    public class FragmentsIterator 
    implements Iterator<ImmutableFragment>
    {
        private int ptr = 0;
        private ImmutableFragment f = null;
        
        public FragmentsIterator(){
        }
        
        @Override
        public boolean hasNext() {
            return ptr < getCount();
        }

        @Override
        public ImmutableFragment next() {
            if( ptr>=getCount() )return null;
            ImmutableFragment f = get(ptr);
            ptr++;
            return f;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Not supported yet.");
        }
    }

    @Override
    public Iterator<ImmutableFragment> iterator() {
        return new FragmentsIterator();
    }
    
//    @Override
    public ImmutableFragments copy(){
        return new ImmutableFragments(this);
    }
    
    public ImmutableFragments copyWith(ImmutableFragment[] arr){
        return new ImmutableFragments(arr);
    }
    
    protected ImmutableFragment[] fragments = new ImmutableFragment[]{};
    public ImmutableFragment[] getFragments(){
        if( fragments==null )fragments = new ImmutableFragment[]{};
        return fragments;
    }
    
    public ImmutableFragments setFragments(ImmutableFragment[] arr){
        if( arr==null )throw new IllegalArgumentException( "arr==null" );
        return copyWith(arr);
    }
    
    public ImmutableFragment get( int index ){
        if( index<0 )return null;
        if( fragments==null )return null;
        if( index>=fragments.length )return null;
        return fragments[index];
    }
    
    public int getCount(){ return fragments==null ? 0 : fragments.length; }
    
    public ImmutableFragments set( int index, ImmutableFragment f ){
        if( f==null )throw new IllegalArgumentException( "f==null" );
        if( index<0 )throw new IllegalArgumentException( "index<0" );
        if( index<getCount() ){
            ImmutableFragment[] sarr = Arrays.copyOf(fragments, fragments.length);
            sarr[index] = f;
            return copyWith(sarr);
        }
        
        ImmutableFragment[] sarr = Arrays.copyOf(fragments, index+1);
        sarr[index] = f;
        return copyWith(sarr);
    }
    
    public ImmutableFragments remove( int index ){
        ArrayList<ImmutableFragment> arr = new ArrayList<ImmutableFragment>();
        int i = -1;
        for( ImmutableFragment f : this ){
            i++;
            if( i!=index ){
                arr.add(f);
            }
        }
        return copyWith(arr.toArray(new ImmutableFragment[]{}));
    }
    
    protected boolean isEquals( ImmutableFragment f1, ImmutableFragment f2 ){
        if( f1==null )throw new IllegalArgumentException( "f1==null" );
        if( f2==null )throw new IllegalArgumentException( "f2==null" );
        return f1==f2;
    }
    
    public ImmutableFragments remove( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
        ArrayList<ImmutableFragment> arr = new ArrayList<ImmutableFragment>();
        for( ImmutableFragment f : this ){
            if( !isEquals(f, fragment) ){
                arr.add(f);
            }
        }
        return copyWith(arr.toArray(new ImmutableFragment[]{}));
    }
    
    public ImmutableFragments removeSameRange( ImmutableFragment fragment ){
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
        ArrayList<ImmutableFragment> arr = new ArrayList<ImmutableFragment>();
        for( ImmutableFragment f : this ){
            if( !f.equalsRange(fragment) ){
                arr.add(f);
            }
        }
        return copyWith(arr.toArray(new ImmutableFragment[]{}));
    }
    
    public int indexOf( ImmutableFragment f ){
        if( f==null )throw new IllegalArgumentException( "f==null" );
        int i = -1;
        for( ImmutableFragment fs : this ){
            i++;
            if( isEquals(f, fs) )return i;
        }
        return -1;
    }
    
    public ImmutableFragments setCount( int count ){
        if( count<0 )throw new IllegalArgumentException("count<0");
        return copyWith(Arrays.copyOf(fragments, count));
    }
    
    public ImmutableFragments add( ImmutableFragment f ){
        if( f==null )throw new IllegalArgumentException( "f==null" );
        return set(getCount(), f);
    }
    
    public ImmutableFragments addAll( Iterable<ImmutableFragment> fitr ){
        if( fitr==null )throw new IllegalArgumentException( "fitr==null" );
        ImmutableFragments r = this;
        for( ImmutableFragment f : fitr ){
            r = r.add(f);
        }
        return r;
    }
    
    private String tostr = null;
    
    @Override
    public String toString(){
        synchronized( this ){
            if( tostr!=null )return tostr;
            StringBuilder s = new StringBuilder();
            int i = -1;
            for( ImmutableFragment f : this ){
                i++;
                s.append(i).append(". ").append(f.toString()).append("\n");
            }
            tostr = s.toString();
            return tostr;
        }
    }
    
    /**
     * Обход каждый с каждым, 
     * без учета позиции в сочетании пар и самого себя. <br>
     * пример: <br>
     * исходно[a, b, c, d] <br>
     * вызовы <br>
     * visitor( a, b )  <br>
     * visitor( a, с )  <br>
     * visitor( a, d )  <br>
     * visitor( b, c )  <br>
     * visitor( b, d )  <br>
     * visitor( c, d )  <br>
     * @param visitor Функция приемник пар 
     * @param equals Функция сравнения на равенство
     */
    public void visitEachEachWithoutPosition( 
        Func2<Object,ImmutableFragment,ImmutableFragment> visitor,
        Func2<Boolean,ImmutableFragment,ImmutableFragment> equals
    ){
        if( visitor==null )throw new IllegalArgumentException( "visitor==null" );
        if( equals==null )throw new IllegalArgumentException( "equals==null" );
        
        ImmutableFragments fs = copy();
        for( ImmutableFragment f1 : this ){
            for( ImmutableFragment f2 : fs ){
                boolean eq = equals.apply(f1, f2);
                if( eq )continue;
                visitor.apply(f1, f2);
            }
            fs = fs.remove(f1);
        }
    }
    
    protected Map<ImmutableFragment,Set<ImmutableFragment>> intersections = null;
    
    /**
     * Поиск пересечений
     * @return пересечения
     */
    public Map<ImmutableFragment,Set<ImmutableFragment>> findIntersections(){
        synchronized(this){
            if( intersections!=null )return intersections;
            
            final Map<ImmutableFragment,Set<ImmutableFragment>> m = new LinkedHashMap<ImmutableFragment, Set<ImmutableFragment>>();
            
            visitEachEachWithoutPosition(
                new Func2<Object, ImmutableFragment, ImmutableFragment>() {
                @Override
                public Object apply(ImmutableFragment f1, ImmutableFragment f2) {
                    if( f1.hasIntersection(f2) ){
                        Set<ImmutableFragment> s = m.get(f1);
                        if( s==null ){
                            s = new LinkedHashSet<ImmutableFragment>();
                            m.put(f1, s);
                        }
                        s.add(f2);
                    }
                    return null;
                }
            },
                new Func2<Boolean, ImmutableFragment, ImmutableFragment>() {
                @Override
                public Boolean apply(ImmutableFragment arg1, ImmutableFragment arg2) {
                    return isEquals(arg1, arg2);
                }
            });

//            ImmutableFragments fs = clone();
//            Map<Fragment,Set<Fragment>> m = new LinkedHashMap<Fragment, Set<Fragment>>();
//
//            for( ImmutableFragment f1 : this ){
//                for( ImmutableFragment f2 : fs ){
//                    if( isEquals(f1, f2) )continue;
//                    if( f1.hasIntersection(f2) ){
//                        Set<Fragment> s = m.get(f1);
//                        if( s==null ){
//                            s = new LinkedHashSet<Fragment>();
//                            m.put(f1, s);
//                        }
//                        s.add(f2);
//                    }
//                }
//                fs = fs.remove(f1);
//            }

            intersections = Collections.unmodifiableMap(m);
            return m;
        }
    }
    
    /**
     * Удаление фрагментов
     * @param fragments фрагменты
     * @return фрагменты
     */
    public ImmutableFragments remove( Iterable<ImmutableFragment> fragments ){
        if( fragments==null )throw new IllegalArgumentException( "fragments==null" );
        ImmutableFragments r = this;
        for( ImmutableFragment f : fragments ){
            r = r.remove(f);
        }
        return r;
    }
    
    /**
     * Удаление фрагментов
     * @param fragments фрагменты
     * @return фрагменты
     */
    public ImmutableFragments removeSameRange( Iterable<ImmutableFragment> fragments ){
        if( fragments==null )throw new IllegalArgumentException( "fragments==null" );
        ImmutableFragments r = this;
        for( ImmutableFragment f : fragments ){
            r = r.removeSameRange(f);
        }
        return r;
    }
    
    protected ImmutableFragments zeroSize = null;

    /**
     * Поиск фрагментов нулей длинны
     * @return нулевой длины элементы
     */
    public ImmutableFragments findZeroSize(){
        synchronized(this){
            if( zeroSize!=null )return zeroSize;

            zeroSize = new ImmutableFragments();
            for( ImmutableFragment f1 : this ){
                if( f1.getSize()==0 )zeroSize = zeroSize.add(f1);
            }

//            zeroSize = Collections.unmodifiableSet(zeroSize);
            return zeroSize;
        }
    }
    
    public TreeSet<ImmutableFragment> sort( Comparator<ImmutableFragment> comparator ){
        if( comparator==null )throw new IllegalArgumentException( "comparator==null" );
        TreeSet<ImmutableFragment> ts = new TreeSet<ImmutableFragment>(comparator);
        for( ImmutableFragment f : this )ts.add(f);
        ImmutableFragment[] fs = ts.toArray(new ImmutableFragment[]{});
        return ts;
    }
    
    protected static final Comparator<ImmutableFragment> sorterByBegin = new Comparator<ImmutableFragment>() {
        @Override
        public int compare(ImmutableFragment o1, ImmutableFragment o2) {
            long l1 = o1.getBegin();
            long l2 = o2.getBegin();
            return l1==l2 ? 0 : (l1 < l2 ? -1 : 1);
        }
    };

    protected static final Comparator<ImmutableFragment> sorterBySize = new Comparator<ImmutableFragment>() {
        @Override
        public int compare(ImmutableFragment o1, ImmutableFragment o2) {
            long s1 = o1.getSize();
            long s2 = o2.getSize();
            return s1==s2 ? 0 : (s1 < s2 ? -1 : 1);
        }
    };
    
    protected static final Comparator<ImmutableFragment> sorterByInstanceID = new Comparator<ImmutableFragment>() {
        @Override
        public int compare(ImmutableFragment o1, ImmutableFragment o2) {
            long s1 = o1.instanceID;
            long s2 = o2.instanceID;
            return s1==s2 ? 0 : (s1 < s2 ? -1 : 1);
        }
    };
    
    protected static final Comparator<ImmutableFragment> defSorter = new Comparator<ImmutableFragment>() {
        @Override
        public int compare(ImmutableFragment o1, ImmutableFragment o2) {
            int cBegin = sorterByBegin.compare(o1, o2);
            int cSize = sorterBySize.compare(o1, o2);
            int cID = sorterByInstanceID.compare(o1, o2);
            int cmp = cBegin * 10 + cSize * 5 + cID;
            return cmp;
        }
    };
    
    protected ImmutableFragments sorted = null;
    
    /**
     * Сортирует фрагменты в порядке возрастания
     * @return Отсортированные фрагменты
     */
    public ImmutableFragments sort(){
        synchronized(this){
            if( sorted!=null )return sorted;
            sorted = copyWith(
                sort(defSorter).toArray(new ImmutableFragment[]{})
            );
            return sorted;
        }
    }
    
    protected static final Func2<ImmutableFragment,Long,Long> defCreateFragment = new Func2<ImmutableFragment, Long, Long>() {
        @Override
        public ImmutableFragment apply(Long begin, Long end) {
            return new ImmutableFragment(begin, end);
        }
    };
    
    public ImmutableFragments findHoles(long begin, long end){
        return findHoles(begin, end, defCreateFragment);
    }
    
    /**
     * Поиск дыр в указанном диапазоне
     * @param begin Начало диапазона
     * @param end Конец диапазона искл
     * @param createFragment Создание фрагмента
     * @return Фрагменты соответ дырам
     */
    public ImmutableFragments findHoles(long begin, long end, Func2<ImmutableFragment,Long,Long> createFragment){
        if( createFragment==null )createFragment = defCreateFragment;
        
        if( begin>end ){
            long t = begin; begin = end; end = t;
        }
        
        synchronized(this){
//            TreeSet<Fragment> holes = new TreeSet<Fragment>(defSorter);
            ImmutableFragments holes = new ImmutableFragments();
            
            long ptr = begin;
            
            ImmutableFragments fragments = this;
            fragments = fragments.remove(fragments.findZeroSize());
            fragments = fragments.sort();
            
            int i = -1;
            int count = fragments.getCount();
            for( ImmutableFragment f : fragments ){
                i++;
                boolean last = i == count-1;
                
                long b = f.getBegin();
                long e = f.getEnd();
                
                if( !last ){
                    if( ptr<b ){
                        ImmutableFragment fHole =
                            createFragment.apply(ptr, b);
                            //new ImmutableFragment(ptr, b);
                        holes.add(fHole);
                        ptr = e;
                    }else{
                        ptr = e;
                    }
                }else{
                    if( ptr<b ){
                        ImmutableFragment fHole = 
                            createFragment.apply(ptr, b);
//                            new ImmutableFragment(ptr, b);
                        holes.add(fHole);
                        ptr = e;
                    }else{
                        ptr = e;
                    }
                    
                    if( ptr<end ){
                        ImmutableFragment fHole = 
                            createFragment.apply(ptr, end);
//                            new ImmutableFragment(ptr, end);
                        holes.add(fHole);
                    }
                }
            }
            
            return holes;
        }
    }
    
    /**
     * Решение пересечения обьеднением 
     */
    public final Func2<ImmutableFragments,ImmutableFragment,ImmutableFragment> unionIntersecResolver = new Func2<ImmutableFragments, ImmutableFragment, ImmutableFragment>() {
        @Override
        public ImmutableFragments apply(ImmutableFragment f1, ImmutableFragment f2) {
            if( f1==null )throw new IllegalArgumentException( "f1==null" );
            if( f2==null )throw new IllegalArgumentException( "f2==null" );
            
            if( !f1.hasIntersection(f2) )
                throw new IllegalArgumentException("no intersection");
            
            ImmutableFragments fs = new ImmutableFragments();
            ImmutableFragment fu = f1.union(f2);
            fs = fs.add(fu);
            return fs;
        }
    };
    
    protected final Func2<ImmutableFragments,ImmutableFragment,ImmutableFragment> defIntersecResolver = unionIntersecResolver;
    
    /**
     * Удаление пересечений, так же удаляет нулевые фрагменты
     * @return Фрагменты без пересечений
     */
    public ImmutableFragments removeIntersections(){
        return removeIntersections(defIntersecResolver);
    }

    /**
     * Удаление пересечений, так же удаляет нулевые фрагменты
     * @param intersecResolver способ решения пересечений
     * @return Фрагменты без пересечений
     */
    public ImmutableFragments removeIntersections(Func2<ImmutableFragments,ImmutableFragment,ImmutableFragment> intersecResolver){
        if( intersecResolver==null )intersecResolver = defIntersecResolver;
        
        ImmutableFragments fs = this;
        fs = fs.removeSameRange(fs.findZeroSize());
        
        while( true ){
            Map<ImmutableFragment,Set<ImmutableFragment>> intersect = fs.findIntersections();
            if( intersect.isEmpty() )return fs;
            
            for( ImmutableFragment f1 : intersect.keySet() ){
                Set<ImmutableFragment> f2set = intersect.get(f1);
                for( ImmutableFragment f2 : f2set ){
                    ImmutableFragments fres = intersecResolver.apply(f1, f2);
                    fs = fs.removeSameRange(f1);
                    fs = fs.removeSameRange(f2);
                    fs = fs.addAll(fres);
                }
            }
        }
    }
    
//    public ImmutableFragments compact(
//        Func2<Fragments,ImmutableFragment,ImmutableFragment> joinResolver,
//        Func2<Fragments,ImmutableFragment,ImmutableFragment> intersecResolver
//    ){
//        ImmutableFragments fs = removeIntersections(intersecResolver);
//        fs = fs.sort();
//        return fs;
//    }
}
