/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;


import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import xyz.cofe.text.template.ctx.DateVar;

/**
 * Формат прдеставления времени в протоколе HTTP согласно RFC 2616.
 * 
 * <pre style="white-space:pre; font-family: monospace; font-size:8px">
 * HTTP-date    = rfc1123-date | rfc850-date | asctime-date
 * rfc1123-date = wkday "," SP date1 SP time SP "GMT"
 * rfc850-date  = weekday "," SP date2 SP time SP "GMT"
 * asctime-date = wkday SP date3 SP time SP 4DIGIT
 * date1        = 2DIGIT SP month SP 4DIGIT
 *                ; day month year (e.g., 02 Jun 1982)
 * date2        = 2DIGIT "-" month "-" 2DIGIT
 *                ; day-month-year (e.g., 02-Jun-82)
 * date3        = month SP ( 2DIGIT | ( SP 1DIGIT ))
 *                ; month day (e.g., Jun  2)
 * time         = 2DIGIT ":" 2DIGIT ":" 2DIGIT
 *                ; 00:00:00 - 23:59:59
 * wkday        = "Mon" | "Tue" | "Wed"
 *              | "Thu" | "Fri" | "Sat" | "Sun"
 * weekday      = "Monday" | "Tuesday" | "Wednesday"
 *              | "Thursday" | "Friday" | "Saturday" | "Sunday"
 * month        = "Jan" | "Feb" | "Mar" | "Apr"
 *              | "May" | "Jun" | "Jul" | "Aug"
 *              | "Sep" | "Oct" | "Nov" | "Dec"
 * </pre>
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class DateTime {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(DateTime.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    /**
     * Парсинг даты согласно RFC 2616
     * @param s строка с датой
     * @return Локальное дата/время или null, если строка не соответ. RFC
     */
    public Date parse( String s ){
        if( s==null )throw new IllegalArgumentException( "s==null" );
        return parse(s, false);
    }
    
    /**
     * Парсинг даты согласно RFC 2616
     * @param s строка с датой
     * @param returnGMT true - возвращать время по гринвичу, false - возвращать локальное время
     * @return Дата/время или null, если строка не соответ. RFC
     */
    public Date parse( String s, boolean returnGMT ){
//        if( s==null )return null;
        if( s==null )throw new IllegalArgumentException( "s==null" );

        Date d = null;
        Date d1123 = parse_RFC_1123(s);
        if( d1123==null ){
            Date d850 = parse_RFC_850(s);
            if( d850==null ){
                Date dasctime = parse_asctime(s);
                d = dasctime;
            }else{
                d = d850;
            }
        }else{
            d = d1123;
        }
        
        if( d==null )return null;
        
        if( returnGMT )return d;
        
        int off = TimeZone.getDefault().getRawOffset();
        d = new Date( d.getTime() + off );
        return d;
    }
    
    /**
     * Текстовое представление времени
     * @param date время
     * @param isGMT true - время по гринвичу (GMT), false - локальное время
     * @return Текстовое представление
     */
    public String toDate( Date date, boolean isGMT ){
        if( date==null )throw new IllegalArgumentException( "date==null" );
        return toDateRFC1123(date, isGMT);
    }
    
    /**
     * Текстовое представление времени
     * @param date локальное время
     * @return Текстовое представление
     */
    public String toDate( Date date ){
        if( date==null )throw new IllegalArgumentException( "date==null" );
        return toDateRFC1123(date, false);
    }
    
    /**
     * Текстовое представление времени
     * @param date локальное время
     * @return Текстовое представление
     */
    public String toDateRFC1123( Date date ){
        if( date==null )throw new IllegalArgumentException( "date==null" );
        return toDateRFC1123(date, false);
    }
    
    /**
     * Текстовое представление времени
     * @param date время
     * @param isGMT true - время по гринвичу (GMT), false - локальное время
     * @return Текстовое представление
     */
    public String toDateRFC1123( Date date, boolean isGMT ){
        if( date==null )throw new IllegalArgumentException( "date==null" );
        
        if( !isGMT ){
            int off = TimeZone.getDefault().getRawOffset();
            date = new Date( date.getTime() - off );
        }
        
        GregorianCalendar gcal = new GregorianCalendar(Locale.ENGLISH);
        gcal.setTime(date);
        
        int wd_0 = gcal.get(GregorianCalendar.DAY_OF_WEEK);
        int wd = -1;
        if( wd_0==GregorianCalendar.MONDAY ) wd = 0;
        else if( wd_0==GregorianCalendar.TUESDAY ) wd = 1;
        else if( wd_0==GregorianCalendar.WEDNESDAY ) wd = 2;
        else if( wd_0==GregorianCalendar.THURSDAY ) wd = 3;
        else if( wd_0==GregorianCalendar.FRIDAY ) wd = 4;
        else if( wd_0==GregorianCalendar.SATURDAY ) wd = 5;
        else if( wd_0==GregorianCalendar.SUNDAY ) wd = 6;
        else throw new Error( "!! GregorianCalendar is wrong: day of week="+wd_0+" for "+new DateVar(date) );
        
        String wdString = wkday[wd];
        
        int day_of_month = gcal.get(GregorianCalendar.DAY_OF_MONTH);
        String day_of_month_String = 
            day_of_month < 10 
            ? "0"+day_of_month 
            : Integer.toString(day_of_month);
        
        int mon = gcal.get(GregorianCalendar.MONTH);
        String monString = month[mon];
        
        int year = gcal.get(GregorianCalendar.YEAR);
        String yearString = 
            year < 10
            ? "000" + year
            : ( year < 100
                  ? "00" + year
                  : (year < 1000 
                     ? "0" + year
                     : Integer.toString(year)
                  )
              );

        String date1String = day_of_month_String+" "+monString+" "+yearString;
        
        int hour = gcal.get( GregorianCalendar.HOUR_OF_DAY );
        String hourString = hour < 10 ? "0"+hour : Integer.toString(hour);
        
        int minute = gcal.get( GregorianCalendar.MINUTE );
        String minuteString = minute < 10 ? "0"+minute : Integer.toString(minute);
        
        int sec = gcal.get( GregorianCalendar.SECOND );
        String secString = sec < 10 ? "0"+sec : Integer.toString(sec);
        
        String time = hourString+":"+minuteString+":"+secString;
        
        return wdString+", "+date1String+" "+time+" "+"GMT";
    }
    
    //<editor-fold defaultstate="collapsed" desc="rfc1123Pattern">
    private static final Pattern rfc1123Pattern =
        Pattern.compile(
            "(?is)^"
                + "(Mon|Tue|Wed|Thu|Fri|Sat|Sun)" // gr 1 - day of week
                + ",\\s+"
                + "(\\d{2})" // gr 2 - date of month
                + "\\s+"
                + "(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)" // gr 3 - month name
                + "\\s+"
                + "(\\d{4})" // gr 4 - year
                + "\\s+"
                + "(\\d{2})" // gr 5 - hour 00 - 23
                + ":"
                + "(\\d{2})" // gr 6 - minute 00 - 59
                + ":"
                + "(\\d{2})" // gr 7 - second 00 - 59
                + "\\s+"
                + "GMT"
                + ".*"
        );
    
    private Date parse_RFC_1123( String str ){
        if( str==null )return null;
        Matcher m = rfc1123Pattern.matcher(str.trim());
        if( !m.matches() )return null;
        
        String wdString = m.group(1);
        
        String dateString = m.group(2);
        String monString = m.group(3);
        String yearString = m.group(4);
        String hourString = m.group(5);
        String minString = m.group(6);
        String secString = m.group(7);
        
        GregorianCalendar gcal = new GregorianCalendar(Locale.ENGLISH);
        
        gcal.set(GregorianCalendar.YEAR, Integer.parseInt(yearString));
        gcal.set(GregorianCalendar.MONTH, indexOfMonth(monString));
        gcal.set(GregorianCalendar.DATE, Integer.parseInt(dateString));
        gcal.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(hourString));
        gcal.set(GregorianCalendar.MINUTE, Integer.parseInt(minString));
        gcal.set(GregorianCalendar.SECOND, Integer.parseInt(secString));
        
        return gcal.getTime();
    }    
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="rfc850Pattern">
    private static final Pattern rfc850Pattern =
        Pattern.compile(
            "(?is)^"
                + "(Monday|Tuesday|Wednesday|Thursday|Friday|Saturday|Sunday)" // gr 1 - day of week
                + ",\\s+"
                + "(\\d{2})" // gr 2 - date of month
                + "-"
                + "(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)" // gr 3 - month name
                + "-"
                + "(\\d{2})" // gr 4 - year
                + "\\s+"
                + "(\\d{2})" // gr 5 - hour 00 - 23
                + ":"
                + "(\\d{2})" // gr 6 - minute 00 - 59
                + ":"
                + "(\\d{2})" // gr 7 - second 00 - 59
                + "\\s+"
                + "GMT"
                + ".*"
        );

    private Date parse_RFC_850( String str ){
        if( str==null )return null;
        Matcher m = rfc850Pattern.matcher(str.trim());
        if( !m.matches() )return null;
        
        String wdString = m.group(1);
        
        String dateString = m.group(2);
        String monString = m.group(3);
        String yearString = m.group(4);
        String hourString = m.group(5);
        String minString = m.group(6);
        String secString = m.group(7);
        
        GregorianCalendar gcal = new GregorianCalendar(Locale.ENGLISH);
        
        int year = Integer.parseInt(yearString);
        if( yearString.length()==2 ){
            if( year>=75 ){
                gcal.set(GregorianCalendar.YEAR, year+1900);
            }else{
                gcal.set(GregorianCalendar.YEAR, year+2000);
            }
        }else{
            gcal.set(GregorianCalendar.YEAR, year);
        }
        
        gcal.set(GregorianCalendar.MONTH, indexOfMonth(monString));
        gcal.set(GregorianCalendar.DATE, Integer.parseInt(dateString));
        gcal.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(hourString));
        gcal.set(GregorianCalendar.MINUTE, Integer.parseInt(minString));
        gcal.set(GregorianCalendar.SECOND, Integer.parseInt(secString));
        
        return gcal.getTime();
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="asctimePattern">
    private static final Pattern asctimePattern =
        Pattern.compile(
            "(?is)^"
                + "(Mon|Tue|Wed|Thu|Fri|Sat|Sun)" // gr 1 - day of week
                + "\\s+"
                + "(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)" // - gr 2 - month name
                + "\\s+"
                + "(\\d{1,2})" // gr 3 - date of month
                + "\\s+"
                + "(\\d{2})" // gr 4 - hour 00 - 23
                + ":"
                + "(\\d{2})" // gr 5 - minute 00 - 59
                + ":"
                + "(\\d{2})" // gr 6 - second 00 - 59
                + "\\s+"
                + "(\\d{4})" // gr 7 - year
                + ".*"
        );
    
    private Date parse_asctime( String str ){
        if( str==null )return null;
        Matcher m = rfc1123Pattern.matcher(str.trim());
        if( !m.matches() )return null;
        
        String wdString = m.group(1);
        
        String dateString = m.group(3);
        String monString = m.group(2);
        String yearString = m.group(7);
        String hourString = m.group(4);
        String minString = m.group(5);
        String secString = m.group(6);
        
        GregorianCalendar gcal = new GregorianCalendar(Locale.ENGLISH);
        
        gcal.set(GregorianCalendar.YEAR, Integer.parseInt(yearString));
        gcal.set(GregorianCalendar.MONTH, indexOfMonth(monString));
        gcal.set(GregorianCalendar.DATE, Integer.parseInt(dateString));
        gcal.set(GregorianCalendar.HOUR_OF_DAY, Integer.parseInt(hourString));
        gcal.set(GregorianCalendar.MINUTE, Integer.parseInt(minString));
        gcal.set(GregorianCalendar.SECOND, Integer.parseInt(secString));
        
        return gcal.getTime();
    }    
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="wkday">
    private static final String[] wkday  = new String[]{
        "Mon" , "Tue" , "Wed"
        , "Thu" , "Fri" , "Sat" , "Sun"
    };

    private static int indexOfWkday( String s ){
        if( s==null )return -1;
        for( int i=0; i<wkday.length; i++ ){
            String wd = wkday[i];
            if( s.equalsIgnoreCase(wd) )return i;
        }
        return -1;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="weekday">
    private static final String[] weekday = new String[]{
        "Monday" , "Tuesday" , "Wednesday"
        , "Thursday" , "Friday" , "Saturday" , "Sunday"
    };

    private static int indexOfWeekday( String s ){
        if( s==null )return -1;
        for( int i=0; i<weekday.length; i++ ){
            String wd = weekday[i];
            if( s.equalsIgnoreCase(wd) )return i;
        }
        return -1;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="month">
    private static final String[] month = new String[]{
        "Jan" , "Feb" , "Mar" , "Apr"
        , "May" , "Jun" , "Jul" , "Aug"
        , "Sep" , "Oct" , "Nov" , "Dec"
    };
    
    private static int indexOfMonth( String s ){
        if( s==null )return -1;
        for( int i=0; i<month.length; i++ ){
            String m = month[i];
            if( s.equalsIgnoreCase(m) )return i;
        }
        return -1;
    }
//</editor-fold>
}
