/*
 * The MIT License
 *
 * Copyright 2016 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.http;


import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.codec.binary.Base64OutputStream;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.map.BasicEventMap;
import xyz.cofe.collection.map.LockEventMap;
import xyz.cofe.common.LazyValue;
import xyz.cofe.io.IOFun;
import xyz.cofe.typeconv.ExtendedCastGraph;
import xyz.cofe.typeconv.TypeCastGraph;

/**
 *
 * @author nt.gocha@gmail.com
 */
public class FormMultipartData
implements Map<String, Object>, Func1<Object,URLConnection>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(FormMultipartData.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected final Lock lock;

    private final Map<String,Object> map;

    public FormMultipartData(){
        lock = new ReentrantLock();

        LockEventMap evmap =
            new LockEventMap(
                new LinkedHashMap(), lock);

        map = evmap;
    }

    public FormMultipartData(Map<String,Object> sourcemap){
        lock = new ReentrantLock();

        LockEventMap evmap =
            sourcemap==null
            ? new LockEventMap( new LinkedHashMap(), lock)
            : new LockEventMap( sourcemap, lock);

        map = evmap;
    }

    public FormMultipartData(FormMultipartData source){
        if( source==null )throw new IllegalArgumentException( "source==null" );

        try{
            source.lock.lock();

            lock = new ReentrantLock();

            LockEventMap evmap =
                new LockEventMap<String, String>(
                    new LinkedHashMap<String, String>(), lock);

            for( Map.Entry<String,Object> en : source.map.entrySet() ){
                if( en==null )continue;
                String k = en.getKey();
                Object v = en.getValue();

                if( k==null )continue;
                evmap.put(k, v);
            }

            map = evmap;

            boundary = source.boundary;
            typeCast = source.typeCast;
            charset = source.charset;
        }finally{
            source.lock.unlock();
        }
    }

    public final static AtomicLong partIdSequence = new AtomicLong(0);

    public interface Part {
        void write( OutputStream out ) throws IOException;
    }

    public interface SetCharset {
        void setCharset( Charset cs );
    }

    public static final String CRLF = "\r\n";
    public static final String generateBoundary(){
        return "------------------------"+Long.toHexString(System.currentTimeMillis()+partIdSequence.incrementAndGet());
    }

    //<editor-fold defaultstate="collapsed" desc="write parts">
    public void write( OutputStream out, Charset cs, String bound, Iterable<Part> parts ) throws IOException
    {
        if( out==null )throw new IllegalArgumentException( "out==null" );
        if( parts==null )throw new IllegalArgumentException( "parts==null" );
        if( cs==null )cs = Charset.forName("utf-8");
        
        if( bound==null )bound = generateBoundary();

        OutputStreamWriter writer = new OutputStreamWriter(out, cs);

        for( Part part : parts ){
            if( part==null )continue;

            writer.append("--").append(bound).append(CRLF).flush();

            if( part instanceof SetCharset ){
                ((SetCharset)part).setCharset(cs);
            }

            part.write(out);
            out.flush();
        }

        writer.append("--").append(bound).append("--").append(CRLF).flush();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="TextPart">
    /**
     * Текстовые данные. <br>
     * value - может быть как строкой, так и LazyValue&lt;String&gt;
     */
    public static class TextPart
    implements SetCharset, Part
    {
        public final long id = partIdSequence.incrementAndGet();

        private Charset charset;

        public Charset getCharset() {
            if( charset==null )charset = Charset.forName("utf-8");
            return charset;
        }

        @Override
        public void setCharset( Charset charset ) {
            this.charset = charset;
        }

        private String name;

        public String getName() {
            return name;
        }

        public void setName( String name ) {
            this.name = name;
        }

        private String contentType;

        public String getContentType() {
            if( contentType==null )contentType = "text/plain";
            return contentType;
        }

        public void setContentType( String contentType ) {
            this.contentType = contentType;
        }

        private Object value;

        public Object getValue() {
            return value;
        }

        public void setValue( Object value ) {
            this.value = value;
        }

        private TypeCastGraph typeCast;

        public TypeCastGraph getTypeCast() {
            if( typeCast==null ){
                typeCast = new ExtendedCastGraph();
            }
            return typeCast;
        }

        public void setTypeCast( TypeCastGraph typeCast ) {
            this.typeCast = typeCast;
        }

        @Override
        public void write( OutputStream out ) throws IOException {
            String name = getName();
            if( name==null )name = "part-id-"+id;

            Charset cs = getCharset();

            OutputStreamWriter writer = new OutputStreamWriter(out, cs);
            writer
                .append("Content-Disposition: form-data;")
                .append(" name=\"")
                .append(name) // TODO: read RFC 2047
                .append("\"")
                .append(CRLF);

            writer
                .append("Content-Type: ")
                .append(getContentType())
                .append("; charset=")
                .append(cs.name())
                .append(CRLF);

            writer.append(CRLF);

            Object val = value;
            if( val instanceof LazyValue ){
                val = ((LazyValue)val).evaluate();
            }

            if( val==null ){
                writer.append("").append(CRLF).flush();
            }else{
                if( val instanceof String ){
                    writer.append((String)val).append(CRLF).flush();
                }else{
                    TypeCastGraph tcast = getTypeCast();
                    String txt = tcast.cast(value, String.class);
                    if( txt!=null ){
                        writer.append(txt).append(CRLF).flush();
                    }else{
                        writer.append("").append(CRLF).flush();
                    }
                }
            }
        }
    }
    //</editor-fold>
    
    public static enum TransferEncoding {
        Binary,
        Base64
    }

    //<editor-fold defaultstate="collapsed" desc="BinaryPart">
    /**
     * Бинарные данные (файл). <br>
     * <b>fileName</b> -
     * это данные типа String, <br>
     * либо LazyValue&lt;String&gt; <br>
     * либо LazyValue&lt;java.io.File&gt; <br>
     * либо LazyValue&lt;xyz.io.File&gt; <br>
     * <br>
     *
     * <b>value</b> - LazyValue | InputStream | byte[] | java.io.File | xyz.cofe.fs.File <br>
     * <br>
     *
     * <b>contentType</b> - если не указано, то определяется из URLConnection.guessContentTypeFrom...
     * <br><br>
     *
     * <b>charset</b> - влияет только на заголовок
     */
    public static class BinaryPart
    implements SetCharset, Part
    {
        public final long id = partIdSequence.incrementAndGet();

        //<editor-fold defaultstate="collapsed" desc="name">
        private String name;

        public String getName() {
            return name;
        }

        public void setName( String name ) {
            this.name = name;
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="contentType">
        private String contentType;

        public String getContentType() {
//            if( contentType==null )contentType = "application/octet-stream";
            return contentType;
        }

        public void setContentType( String contentType ) {
            this.contentType = contentType;
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="fileName">
        private Object fileName;

        public Object getFileName() {
            return fileName;
        }

        public void setFileName( Object fileName ) {
            this.fileName = fileName;
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="charset">
        private Charset charset;

        public Charset getCharset() {
            if( charset==null )charset = Charset.forName("utf-8");
            return charset;
        }

        @Override
        public void setCharset( Charset charset ) {
            this.charset = charset;
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="value">
        private Object value;

        public Object getValue() {
            return value;
        }

        public void setValue( Object value ) {
            this.value = value;
        }
//</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="transferEncoding">
        private TransferEncoding transferEncoding = TransferEncoding.Binary;
        
        public TransferEncoding getTransferEncoding() {
            return transferEncoding;
        }
        
        public void setTransferEncoding(TransferEncoding transferEncoding) {
            this.transferEncoding = transferEncoding;
        }
//</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="write()">
        @Override
        public void write( OutputStream out ) throws IOException {
            String name = getName();
            if( name==null )name = "part-id-"+id;

            Object fnameObj = getFileName();
            String fname = null;
            if( fnameObj instanceof String ){
                fname = (String)fnameObj;
                if( fname.length()<1 )fname = name;
            }else if( fnameObj instanceof LazyValue ){
                Object oname = ((LazyValue)fnameObj).evaluate();
                if( oname instanceof String ){
                    fname = (String)oname;
                    if( fname.length()<1 )fname = name;
                }else if( oname instanceof java.io.File ){
                    fname = ((java.io.File)oname).getName();
                    if( fname.length()<1 )fname = name;
                }else if( oname instanceof xyz.cofe.fs.File ){
                    fname = ((xyz.cofe.fs.File)oname).getName();
                    if( fname.length()<1 )fname = name;
                }else{
                    fname = name;
                }
            }else{
                fname = name;
            }

            Charset cs = getCharset();

            Object val = value;
            if( val instanceof LazyValue ){
                val = ((LazyValue)val).evaluate();
            }

            String fileContType = null;

            if( val instanceof java.io.File ){
                java.io.File f = (java.io.File)val;

                fileContType = URLConnection.guessContentTypeFromName(f.getName());
                if( fileContType==null ){
                    java.io.FileInputStream fin = new java.io.FileInputStream(f);
                    fileContType = URLConnection.guessContentTypeFromStream(fin);
                    fin.close();
                }
            }else if( val instanceof xyz.cofe.fs.File ){
                xyz.cofe.fs.File f = (xyz.cofe.fs.File)val;

                fileContType = URLConnection.guessContentTypeFromName(f.getName());
                if( fileContType==null ){
                    java.io.InputStream fin = f.openRead();
                    fileContType = URLConnection.guessContentTypeFromStream(fin);
                    fin.close();
                }
            }

            String contType = contentType==null
                              ? (fileContType!=null ? fileContType : "application/octet-stream")
                              : contentType;

            OutputStreamWriter writer = new OutputStreamWriter(out, cs);
            writer
                .append("Content-Disposition: form-data")
                .append("; name=\"").append(name).append("\"") // TODO: read RFC 2047
                .append("; filename=\"").append(fname).append("\"") // TODO: read RFC 2047
                .append(CRLF);

            writer
                .append("Content-Type: ").append(contType)
                .append(CRLF);
            
            if( transferEncoding!=null ){
                writer.append("Content-Transfer-Encoding: ");
                switch( transferEncoding ){
                    case Binary:
                        writer.append("binary");
                        break;
                    case Base64:
                        writer.append("base64");
                        break;
                }
                writer.append(CRLF);
            }

            writer.append(CRLF);
            writer.flush();
            
            OutputStream dataStream = out;
            boolean closeDataStream = false;
            byte[] suffixData = new byte[]{};
            String suffixDataString = "";
            
            if( transferEncoding!=null ){
                switch( transferEncoding ){
                    case Base64:
                        final OutputStream fout = out;
                        OutputStream nonCloseOut = new OutputStream() {
                            @Override
                            public void write(int b) throws IOException {
                                fout.write(b);
                            }

                            @Override
                            public void write(byte[] b, int off, int len) throws IOException {
                                fout.write(b, off, len);
                            }

                            @Override
                            public void write(byte[] b) throws IOException {
                                fout.write(b);
                            }

                            @Override
                            public void close() throws IOException {
                            }
                        };
                        
                        Base64OutputStream base64out = new Base64OutputStream(nonCloseOut);
                        dataStream = base64out;
                        closeDataStream = false;
                        suffixDataString = "==";
                        break;
                }
            }

            if( val instanceof InputStream ){
                IOFun.copy((InputStream)val, dataStream);
                
                dataStream.flush();
                if( closeDataStream )dataStream.close();
                
            }else if( val instanceof java.io.File ){
                java.io.File f = (java.io.File)val;
                java.io.FileInputStream fin = new java.io.FileInputStream(f);

                IOFun.copy(fin, dataStream);
                fin.close();

                dataStream.flush();
                if( closeDataStream )dataStream.close();
            }else if( val instanceof byte[] ){
                byte[] data = (byte[])val;

                dataStream.write(data);
                
                dataStream.flush();
                if( closeDataStream )dataStream.close();
            }else if( val instanceof xyz.cofe.fs.File ){
                xyz.cofe.fs.File f = (xyz.cofe.fs.File)val;
                java.io.InputStream fin = f.openRead();

                IOFun.copy(fin, dataStream);
                fin.close();

                dataStream.flush();
                if( closeDataStream )dataStream.close();
            }else{
            }
            
            if( suffixData!=null && suffixData.length>0 ){
                out.write(suffixData);
            }
            
            if( suffixDataString!=null && suffixDataString.length()>0 ){
                writer.append(suffixDataString);
            }

            writer.append(CRLF).flush();
        }
//</editor-fold>
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="map methods">
    @Override
    public int size() {
        return map.size();
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public boolean containsKey( Object key ) {
        return map.containsKey(key);
    }

    @Override
    public boolean containsValue( Object value ) {
        return map.containsValue(value);
    }

    @Override
    public Object get( Object key ) {
        return map.get(key);
    }

    @Override
    public Object put( String key, Object value ) {
        return map.put(key, value);
    }

    @Override
    public Object remove( Object key ) {
        return map.remove(key);
    }

    @Override
    public void putAll(
        Map<? extends String, ? extends Object> m ) {
        map.putAll(m);
    }

    @Override
    public void clear() {
        map.clear();
    }

    @Override
    public Set<String> keySet() {
        return map.keySet();
    }

    @Override
    public Collection<Object> values() {
        return map.values();
    }

    @Override
    public Set<Map.Entry<String, Object>> entrySet() {
        return map.entrySet();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="boundary">
    private String boundary;

    public String getBoundary() {
        try{
            lock.lock();
            if( boundary==null ){
                boundary = generateBoundary();
            }
            return boundary;
        }finally{
            lock.unlock();
        }
    }

    public void setBoundary( String boundary ) {
        try{
            lock.lock();
            this.boundary = boundary;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="typeCast">
    private TypeCastGraph typeCast;

    public TypeCastGraph getTypeCast() {
        try{
            lock.lock();
            if( typeCast==null ){
                typeCast = new ExtendedCastGraph();
            }
            return typeCast;
        }finally{
            lock.unlock();
        }
    }

    public void setTypeCast( TypeCastGraph typeCast ) {
        try{
            lock.lock();
            this.typeCast = typeCast;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="write()">
    public void write( OutputStream out ){
        if( out==null )throw new IllegalArgumentException( "out==null" );
        write(out, null);
    }

    public void write( OutputStream out, Charset cs ){
        if( out==null )throw new IllegalArgumentException( "out==null" );
        if( cs==null )cs = Charset.forName("utf-8");

        try{
            lock.lock();

            List<Part> parts = new ArrayList<Part>();

            for( Map.Entry<String,Object> en : entrySet() ){
                String k = en.getKey();
                Object v = en.getValue();

                if( k==null || k.length()<1 )continue;
                if( v==null )continue;

                if( v instanceof LazyValue ){
                    v = ((LazyValue)v).evaluate();
                }
                if( v==null )continue;

                if( v instanceof Part ){
                    parts.add( (Part)v );
                }else if( v instanceof java.io.File ){
                    java.io.File f = (java.io.File)v;

                    BinaryPart bpart = new BinaryPart();
                    bpart.setName(k);
                    bpart.setFileName(f.getName());
                    bpart.setValue(f);
                    bpart.setCharset(cs);
                    parts.add(bpart);
                }else if( v instanceof xyz.cofe.fs.File ){
                    xyz.cofe.fs.File f = (xyz.cofe.fs.File)v;

                    BinaryPart bpart = new BinaryPart();
                    bpart.setName(k);
                    bpart.setFileName(f.getName());
                    bpart.setValue(f);
                    bpart.setCharset(cs);
                    parts.add(bpart);
                }else if( v instanceof byte[] ){
                    BinaryPart bpart = new BinaryPart();
                    bpart.setName(k);
                    bpart.setValue(v);
                    bpart.setCharset(cs);
                    parts.add(bpart);
                }else if( v instanceof InputStream ){
                    BinaryPart bpart = new BinaryPart();
                    bpart.setName(k);
                    bpart.setValue(v);
                    bpart.setCharset(cs);
                    parts.add(bpart);
                }else{
                    TextPart tpart = new TextPart();
                    tpart.setName(k);
                    tpart.setValue(v);
                    tpart.setCharset(cs);
                    tpart.setTypeCast(typeCast);
                    parts.add(tpart);
                }
            }

            String bound = getBoundary();
            bound = bound==null ? generateBoundary() : bound;

            try {
                write(out, cs, bound, parts);
            } catch( IOException ex ) {
                throw new IOError(ex);
            }
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="charset">
    private Charset charset;

    public Charset getCharset() {
        try{
            lock.lock();
            return charset;
        }finally{
            lock.unlock();
        }
    }

    public void setCharset( Charset charset ) {
        try{
            lock.lock();
            this.charset = charset;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    @Override
    public Object apply( URLConnection conn ) {
        if( conn==null )throw new IllegalArgumentException( "conn==null" );

        try{
            lock.lock();

            Charset cs = getCharset();
            cs = cs==null ? Charset.forName("utf-8") : cs;

            conn.setDoOutput(true);

            String bound = getBoundary();
            if( bound==null ){
                bound = generateBoundary();
                setBoundary(bound);
            }

            conn.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + bound);

            try {
                OutputStream out = conn.getOutputStream();
                write(out, cs);
            } catch( IOException ex ) {
                Logger.getLogger(FormUrlencodedMap.class.getName()).log(Level.SEVERE, null, ex);
                throw new IOError(ex);
            }
        }finally{
            lock.unlock();
        }

        return null;
    }
}
