/*
 * The MIT License
 *
 * Copyright 2016 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.http;


import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.map.BasicEventMap;
import xyz.cofe.collection.map.LockEventMap;
import xyz.cofe.common.LazyValue;
import xyz.cofe.typeconv.ExtendedCastGraph;
import xyz.cofe.typeconv.TypeCastGraph;

/**
 *
 * @author nt.gocha@gmail.com
 */
public class FormUrlencodedMap
implements Map<String,Object>, Func1<Object,URLConnection>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(FormUrlencodedMap.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    private final Map<String,Object> map;
    protected final Lock lock;

    public FormUrlencodedMap(){
        lock = new ReentrantLock();

        LockEventMap evmap =
            new LockEventMap(
                new LinkedHashMap(), lock);

        map = evmap;
    }

    public FormUrlencodedMap(Map<String,Object> sourcemap){
        lock = new ReentrantLock();

        LockEventMap evmap =
            sourcemap==null
            ? new LockEventMap( new LinkedHashMap(), lock)
            : new LockEventMap( sourcemap, lock);

        map = evmap;
    }

    public FormUrlencodedMap(FormUrlencodedMap source){
        if( source==null )throw new IllegalArgumentException( "source==null" );

        try{
            source.lock.lock();

            lock = new ReentrantLock();

            LockEventMap evmap =
                new LockEventMap<String, String>(
                    new LinkedHashMap<String, String>(), lock);

            for( Map.Entry<String,Object> en : source.map.entrySet() ){
                if( en==null )continue;
                String k = en.getKey();
                Object v = en.getValue();

                if( k==null )continue;
                evmap.put(k, v);
            }

            map = evmap;

            typeCast = source.typeCast;
            writeNullValue = source.writeNullValue;
            charset = source.charset;
        }finally{
            source.lock.unlock();
        }
    }

    @Override
    public FormUrlencodedMap clone(){
        return new FormUrlencodedMap(this);
    }

    //<editor-fold defaultstate="collapsed" desc="map methods">
    @Override
    public int size() {
        return map.size();
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public boolean containsKey( Object key ) {
        return map.containsKey(key);
    }

    @Override
    public boolean containsValue( Object value ) {
        return map.containsValue(value);
    }

    @Override
    public Object get( Object key ) {
        return map.get(key);
    }

    @Override
    public Object put( String key, Object value ) {
        return map.put(key, value);
    }

    @Override
    public Object remove( Object key ) {
        return map.remove(key);
    }

    @Override
    public void putAll(
        Map<? extends String, ? extends Object> m ) {
        map.putAll(m);
    }

    @Override
    public void clear() {
        map.clear();
    }

    @Override
    public Set<String> keySet() {
        return map.keySet();
    }

    @Override
    public Collection<Object> values() {
        return map.values();
    }

    @Override
    public Set<Map.Entry<String, Object>> entrySet() {
        return map.entrySet();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="writeNullValue">
    private boolean writeNullValue = true;

    public boolean isWriteNullValue() {
        try{
            lock.lock();
            return writeNullValue;
        }finally{
            lock.unlock();
        }
    }

    public void setWriteNullValue( boolean writeNullValue ) {
        try{
            lock.lock();
            this.writeNullValue = writeNullValue;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="typeCast">
    private TypeCastGraph typeCast;

    public TypeCastGraph getTypeCast() {
        try{
            lock.lock();
            if( typeCast==null ){
                typeCast = new ExtendedCastGraph();
            }
            return typeCast;
        }finally{
            lock.unlock();
        }
    }

    public void setTypeCast( TypeCastGraph typeCast ) {
        try{
            lock.lock();
            this.typeCast = typeCast;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="write()">
    public void write( OutputStream out ){
        if( out==null )throw new IllegalArgumentException( "out==null" );
        write(out, null);
    }

    public void write( OutputStream out,Charset cs ){
        if( out==null )throw new IllegalArgumentException( "out==null" );
        if( cs==null )cs = Charset.forName("utf-8");

        OutputStreamWriter outw = new OutputStreamWriter(out, cs);
        write(outw, cs);
        try {
            outw.flush();
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    public void write( Writer out,Charset cs ){
        if( out==null )throw new IllegalArgumentException( "out==null" );
        if( cs==null )cs = Charset.forName("utf-8");
        try{
            lock.lock();
            int writedPairCount = 0;
            for( Map.Entry<String,Object> en : map.entrySet() ){
                String k = en.getKey();
                Object v = en.getValue();
                if( v==null ){
                    if( !writeNullValue )continue;
                    if( k==null )continue;
                    if( k.length()<1 )continue;
                    if( writedPairCount>0 )writePairSplitter(out, cs);
                    if( writeNullValue(out, k, cs) ){
                        writedPairCount++;
                    }
                }else{
                    if( k==null )continue;
                    if( k.length()<1 )continue;
                    if( writedPairCount>0 )writePairSplitter(out,cs);
                    if( writeKeyValue(out, k, v, cs) ){
                        writedPairCount++;
                    }
                }
            }
        }finally{
            lock.unlock();
        }
    }

    protected void writePairSplitter( Writer out, Charset cs ){
        try {
            out.write("&");
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    protected boolean writeNullValue( Writer out, String k, Charset cs ){
        if( k==null )return false;
        if( k.length()<1 )return false;
        if( !writeNullValue )return false;

        try {
            String enck = URLEncoder.encode(k, cs.name());
            out.write(enck);
        } catch( UnsupportedEncodingException ex ) {
            throw new Error(ex.getMessage(), ex);
        } catch( IOException ex ) {
            throw new IOError(ex);
        }

        return true;
    }

    protected boolean writeKeyValue( Writer out, String key, Object value, Charset cs ){
        if( value==null )return writeNullValue( out, key, cs);
        if( value instanceof LazyValue ){
            value = ((LazyValue)value).evaluate();
        }
        if( value==null )return writeNullValue( out, key, cs);

        if( key==null )return false;
        if( key.length()<1 )return false;

        if( value instanceof String ){
            return writeKeyText(out, key, (String)value, cs);
        }else if( value instanceof java.io.File ){
            return writeKeyText(out, key, ((java.io.File)value).getName(), cs);
        }else if( value instanceof xyz.cofe.fs.File ){
            return writeKeyText(out, key, ((xyz.cofe.fs.File)value).getName(), cs);
        }else if( value instanceof byte[] ){
            return writeNullValue(out, key, cs);
        }else if( value instanceof InputStream ){
            return writeNullValue(out, key, cs);
        }

        TypeCastGraph tcast = getTypeCast();
        try{
            String text = tcast.cast(value, String.class);
            if( text==null || text.length()<1 )return writeNullValue(out, key, cs);

            return writeKeyText(out, key, text, cs);
        }catch( Throwable err ){
            logWarning("failed cast value to string", err.getMessage());
            return false;
        }
    }

    protected boolean writeKeyText( Writer out, String key, String text, Charset cs ){
        if( key==null )return false;
        if( key.length()<1 )return false;

        if( text==null )return writeNullValue(out, key, cs);
        if( text.length()<1 )return writeNullValue(out, key, cs);

        try {
            String enckey = URLEncoder.encode(key, cs.name());
            out.write(enckey);

            out.write("=");

            String enctxt = URLEncoder.encode(text, cs.name());
            out.write(enctxt);
        } catch( UnsupportedEncodingException ex ) {
            throw new Error(ex.getMessage(), ex);
        } catch( IOException ex ) {
            throw new IOError(ex);
        }

        return true;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="charset">
    private Charset charset;

    public Charset getCharset() {
        try{
            lock.lock();
            return charset;
        }finally{
            lock.unlock();
        }
    }

    public void setCharset( Charset charset ) {
        try{
            lock.lock();
            this.charset = charset;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    @Override
    public Object apply( URLConnection conn ) {
        if( conn==null )throw new IllegalArgumentException( "conn==null" );

        try{
            lock.lock();

            Charset cs = getCharset();
            cs = cs==null ? Charset.forName("utf-8") : cs;

            conn.setDoOutput(true);

            conn.setRequestProperty("Accept-Charset", cs.name());
            conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded;charset=" + cs.name());

            try {
                OutputStream out = conn.getOutputStream();
                write(out, cs);
            } catch( IOException ex ) {
                Logger.getLogger(FormUrlencodedMap.class.getName()).log(Level.SEVERE, null, ex);
                throw new IOError(ex);
            }
        }finally{
            lock.unlock();
        }

        return null;
    }
}
