/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;

import java.io.Closeable;
import java.net.CookieHandler;
import java.net.CookieManager;
import java.net.CookiePolicy;
import java.net.MalformedURLException;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import xyz.cofe.collection.Func2;
import xyz.cofe.http.download.Mirrors;
//import xyz.cofe.uplaunch.BuildInfo;

/**
 * Http client
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class HttpClient {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(HttpClient.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(HttpClient.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(HttpClient.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(HttpClient.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(HttpClient.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(HttpClient.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(HttpClient.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected static final AtomicLong sequenceID = new AtomicLong();

    /**
     * Идентификатор объекта в пределах приложения
     */
    public final long id = sequenceID.incrementAndGet();

    protected final Lock lock;

    /**
     * Инициализция
     */
    static {
        logFine("init cookie");

        // Прием всех кук
        CookieHandler.setDefault(new CookieManager(null, CookiePolicy.ACCEPT_ALL));

        // доверять всем сертификатам
        TrustManager[] trustAllCertificates = new TrustManager[] {
            new X509TrustManager() {
                @Override
                public X509Certificate[] getAcceptedIssuers() {
                    return null; // Not relevant.
                }
                @Override
                public void checkClientTrusted(X509Certificate[] certs, String authType) {
                    // Do nothing. Just allow them all.
                }
                @Override
                public void checkServerTrusted(X509Certificate[] certs, String authType) {
                    // Do nothing. Just allow them all.
                }
            }
        };

        // доверять всем хостам
        HostnameVerifier trustAllHostnames = new HostnameVerifier() {
            @Override
            public boolean verify(String hostname, SSLSession session) {
                return true; // Just allow them all.
            }
        };

        try {
            System.setProperty("jsse.enableSNIExtension", "false");
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCertificates, new SecureRandom());

            logFine("init ssl socket factory");
            HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());

            logFine("init hostname verifier");
            HttpsURLConnection.setDefaultHostnameVerifier(trustAllHostnames);
        }
        catch (GeneralSecurityException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    //<editor-fold defaultstate="collapsed" desc="create / clone">
    public HttpClient() {
        lock = new ReentrantLock();
    }

    public HttpClient(HttpClient source) {
        if( source==null ){
            throw new IllegalArgumentException( "source==null" );
        }
        lock = new ReentrantLock();
        try{
            lock.lock();
            this.userAgent = source.userAgent;
            this.defCharset = source.defCharset;
            this.responseThreadPriority = source.responseThreadPriority;
            this.connectTimeout = source.connectTimeout;
            this.readTimeout = source.readTimeout;
        }finally{
            lock.unlock();
        }
    }

    @Override
    public HttpClient clone() {
        return new HttpClient(this);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="listeners">
    protected final HttpListenersHelper httpListenerHelper = new HttpListenersHelper();

    public Set<HttpListener> getListeners() {
        return httpListenerHelper.getListeners();
    }

    public Closeable addListener(HttpListener listener) {
        return httpListenerHelper.addListener(listener);
    }

    public Closeable addListener(HttpListener listener, boolean weakLink) {
        return httpListenerHelper.addListener(listener, weakLink);
    }

    public void removeListener(HttpListener listener) {
        httpListenerHelper.removeListener(listener);
    }

    protected void fireEvent(HttpEvent event) {
        httpListenerHelper.fireEvent(event);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="default charset">
    private static final String defaultCharset = "ISO-8859-1";

    protected Charset defCharset = null;

    public Charset getDefaultCharset() {
        try{
            lock.lock();
            if( defCharset==null )defCharset = Charset.forName(defaultCharset);
            return defCharset;
        }finally{
            lock.unlock();
        }
    }
    public void setDefaultCharset(Charset cs) {
        try{
            lock.lock();
            this.defCharset = cs;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="userAgent">
    protected String userAgent = null;

    public String getUserAgent() {
        try{
            lock.lock();
            if( userAgent==null ){
                String ver=BuildInfo.getVersion();
                if( ver!=null ){
                    userAgent = HttpClient.class.getName()+"/"+ver;
                }else{
                    userAgent = HttpClient.class.getName()+"/"+"0.1-SNAPSHOT";
                }
            }
            return userAgent;
        }finally{
            lock.unlock();
        }
    }

//    public static String

    public void setUserAgent(String userAgent) {
        try{
            lock.lock();
            this.userAgent = userAgent;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="connectTimeout">
    protected Integer connectTimeout = null;

    public int getConnectTimeout() {
        try{
            lock.lock();
            if( connectTimeout==null ){
                connectTimeout = 30000;
            }
            return connectTimeout;
        }finally{
            lock.unlock();
        }
    }

    public void setConnectTimeout(int connectTimeoutMS) {
        if( connectTimeoutMS<0 )throw new IllegalArgumentException("connectTimeoutMS<0");
        try{
            lock.lock();
            this.connectTimeout = connectTimeoutMS;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="readTimeout">
    protected Integer readTimeout = null;

    /**
     * Указывает таймаут чтения (миллисек.)
     * @return таймаут в миллисек
     */
    public int getReadTimeout() {
        try{
            lock.lock();
            if( readTimeout==null ){
                readTimeout = 30000;
            }
            return readTimeout;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает таймаут чтения (миллисек.)
     * @param readTimeoutMS таймаут в миллисек
     */
    public void setReadTimeout(int readTimeoutMS) {
        if( readTimeoutMS<0 )throw new IllegalArgumentException("readTimeoutMS<0");
        try{
            lock.lock();
            this.readTimeout = readTimeoutMS;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="proxySelector">
    protected ProxySelector proxySelector = null;

    /**
     * Настройки прокси
     * @return прокси
     */
    public ProxySelector getProxySelector() {
        try{
            lock.lock();
            if( proxySelector==null ){
//                proxySelector = ProxySelector.getDefault();
                proxySelector = new SystemProxySelector();
            }
            return proxySelector;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Настройки прокси
     * @param proxySelector прокси
     */
    public void setProxySelector(ProxySelector proxySelector) {
        try{
            lock.lock();
            this.proxySelector = proxySelector;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="requestThreadPriority">
    protected Integer responseThreadPriority = null;

    /**
     * Приоритет потока в котором исполняется запрос
     * @return Приритет
     */
    public int getResponseThreadPriority() {
        try{
            lock.lock();
            if( responseThreadPriority==null ){
                responseThreadPriority = Thread.MIN_PRIORITY;
            }
            return responseThreadPriority;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Приоритет потока в котором исполняется запрос
     * @param requestThreadPriority Приритет
     */
    public void setResponseThreadPriority(int requestThreadPriority) {
        try{
            lock.lock();
            this.responseThreadPriority = requestThreadPriority;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createResponseThread">
    protected Func2<Thread,HttpResponse,Runnable> defCreateResponseThread = new Func2<Thread, HttpResponse,Runnable>() {
        @Override
        public Thread apply(final HttpResponse it, Runnable run) {
            long responseId = it.id;

            Thread thread = new Thread(run);
            thread.setDaemon(true);
//            thread.setName(Text.template("httpResponse(id:{0})", responseId));
            thread.setName("httpResponse(id:"+responseId+")");
            thread.setPriority(getResponseThreadPriority());
            thread.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
                @Override
                public void uncaughtException( Thread t, Throwable e ) {
                    logException(e);

                    if( it!=null ){
                        try{
//                            it.getErrors().add(e);
                            it.addError(e);
                            it.setState(HttpResponse.State.Finished);
                        }catch( Throwable err ){
                            logException(err);
//                            err.printStackTrace();
                        }
                    }

                    long tbegin = System.currentTimeMillis();
                    long tmax = 1000 * 15; // 15 seconds for stop
                    long sleep = 50; // 50 milli second sleep
                    while( t.isAlive() ){
                        t.interrupt();
                        long tnow = System.currentTimeMillis();
                        long diff = tnow - tbegin;
                        if( diff > tmax ){
                            t.stop();
                        }else{
                            try {
                                Thread.sleep(sleep);
                            } catch( InterruptedException ex ) {
                                t.stop();
                                break;
                            }
                        }
                    }
                }
            });

            return thread;
        }
    };

    public Func2<Thread,HttpResponse,Runnable> createResponseThread(){
        synchronized(this){return defCreateResponseThread;}
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createDownloaderThread">
    protected Func2<Thread,HttpDownloader,Runnable> defCreateDownloaderThread = new Func2<Thread, HttpDownloader,Runnable>() {
        @Override
        public Thread apply(final HttpDownloader it, Runnable run) {
            long responseId = it.id;

            Thread thread = new Thread(run);
            thread.setDaemon(true);
//            thread.setName(Text.template("httpDownloader(id:{0})", responseId));
            thread.setName("httpDownloader(id:"+responseId+")");
            thread.setPriority(getResponseThreadPriority());
            thread.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
                @Override
                public void uncaughtException( Thread t, Throwable e ) {
//                    e.printStackTrace();
                    logException(e);

                    if( it!=null ){
                        try{
                            it.setState(HttpDownloader.State.Finished);
                        }catch( Throwable err ){
                            logException(err);
                        }
                    }

                    long tbegin = System.currentTimeMillis();
                    long tmax = 1000 * 15; // 15 seconds for stop
                    long sleep = 50; // 50 milli second sleep
                    while( t.isAlive() ){
                        t.interrupt();
                        long tnow = System.currentTimeMillis();
                        long diff = tnow - tbegin;
                        if( diff > tmax ){
                            t.stop();
                        }else{
                            try {
                                Thread.sleep(sleep);
                            } catch( InterruptedException ex ) {
                                t.stop();
                                break;
                            }
                        }
                    }
                }
            });

            return thread;
        }
    };

    public Func2<Thread,HttpDownloader,Runnable> createDownloaderThread(){
        synchronized(this){return defCreateDownloaderThread;}
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createRequest">
    /**
     * Создает запрос
     * @param uri URL
     * @return запрос или null, если url не правильно сформирован
     */
    public HttpRequest createRequest( URI uri ){
        if( uri==null )throw new IllegalArgumentException( "uri==null" );
        try {
            return new HttpRequest(uri.toURL(), this);
        } catch (MalformedURLException ex) {
            Logger.getLogger(HttpClient.class.getName()).log(Level.SEVERE, null, ex);
            return null;
        }
    }

    /**
     * Создает запрос
     * @param url URL
     * @return запрос
     */
    public HttpRequest createRequest( URL url ){
        if( url==null )throw new IllegalArgumentException( "url==null" );
        return new HttpRequest(url, this);
    }

    /**
     * Создает запрос
     * @param url URL
     * @return запрос или null, если url не правильно сформирован
     */
    public HttpRequest createRequest( String url ){
        try {
            if( url==null )throw new IllegalArgumentException( "url==null" );
            return new HttpRequest(new URL(url), this);
        } catch (MalformedURLException ex) {
            Logger.getLogger(HttpClient.class.getName()).log(Level.SEVERE, null, ex);
        }
        return null;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createDownloader()">
    /**
     * Создание докачки запроса
     * @param requests зеркала
     * @return дочкачка
     */
    public HttpDownloader createDownloader( Iterable<HttpRequest> requests ){
        if( requests==null )throw new IllegalArgumentException( "requests==null" );
        HttpRequest[] reqs = new HttpRequest[]{};
        if( requests!=null ){
            for( HttpRequest hr : requests ){
                if( hr==null )continue;
                reqs = Arrays.copyOf(reqs, reqs.length+1);
                reqs[reqs.length-1] = hr;
            }
        }
        return new HttpDownloader(new Mirrors(reqs));
    }
    
    /**
     * Создание докачки запроса
     * @param request запрос
     * @param mirrors зеркала
     * @return дочкачка
     */
    public HttpDownloader createDownloader( HttpRequest request, HttpRequest ... mirrors ){
        if( request==null )throw new IllegalArgumentException( "request==null" );
        HttpRequest[] reqs = new HttpRequest[]{};
        reqs = Arrays.copyOf(reqs, reqs.length+1);
        reqs[reqs.length-1] = request;
        if( mirrors!=null ){
            for( HttpRequest hr : mirrors ){
                if( hr==null )continue;
                reqs = Arrays.copyOf(reqs, reqs.length+1);
                reqs[reqs.length-1] = hr;
            }
        }
        return new HttpDownloader(new Mirrors(reqs));
    }
    
    /**
     * Создание докачки запроса
     * @param mirrors зеркала
     * @return дочкачка
     */
    public HttpDownloader createDownloader( Mirrors mirrors ){
        if( mirrors==null )throw new IllegalArgumentException( "request==null" );
        return new HttpDownloader(mirrors);
    }
//</editor-fold>
}