/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;

import java.io.Closeable;
import java.io.IOException;
import java.net.BindException;
import java.net.ConnectException;
import java.net.HttpRetryException;
import java.net.MalformedURLException;
import java.net.NoRouteToHostException;
import java.net.PortUnreachableException;
import java.net.ProtocolException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.UnknownServiceException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.cbuffer.ContentBuffer;
import xyz.cofe.cbuffer.MemContentBuffer;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.set.BasicEventSet;
import xyz.cofe.collection.set.EventSet;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.common.Fragment;
import xyz.cofe.common.ImmutableFragment;
import xyz.cofe.http.download.BasicFragmentValidator;
import xyz.cofe.http.download.ContentValidation;
import xyz.cofe.http.download.ContentValidator;
import xyz.cofe.http.download.ContentValidatorEvent;
import xyz.cofe.http.download.ContentValidatorListener;
import xyz.cofe.http.download.ContentValidatorProgressEvent;
import xyz.cofe.http.download.ContentValidatorSender;
import xyz.cofe.http.download.Counter;
import xyz.cofe.http.download.MaxCounter;
import xyz.cofe.http.download.CreatedNewParts;
import xyz.cofe.http.download.EqualsFragments;
import xyz.cofe.http.download.FragmentValidator;
import xyz.cofe.http.download.GetCounters;
import xyz.cofe.http.download.GetPart;
import xyz.cofe.http.download.GetPartBuilder;
import xyz.cofe.http.download.GetPartList;
import xyz.cofe.http.download.InitFragments;
import xyz.cofe.http.download.IntCounter;
import xyz.cofe.http.download.LongCounter;
import xyz.cofe.http.download.Mirrors;
import xyz.cofe.http.download.OverflowCounter;
import xyz.cofe.http.download.ProgressCounters;
import xyz.cofe.http.download.ResetCounters;
import xyz.cofe.text.Text;
import xyz.cofe.http.download.ResetCounter;
import xyz.cofe.http.download.BindHttpDownloader;

/**
 * Докачивание данных. <br>
 * 
 * <b>Простой пример</b>
 <pre>
<span style='color:#666666; font-style: italic'>// Создание клиента</span>
HttpClient client = new HttpClient()
 
<span style='color:#666666; font-style: italic'>// Создание запроса откуда качать</span>
HttpRequest req = 
  client.createRequest( "http://site/file.zip" )

<span style='color:#666666; font-style: italic'>// Создание докачки</span>
HttpDownloader dlr = 
  client.createDownloader( req )

<span style='color:#666666; font-style: italic'>// Восстановление предыдущего состояния</span>
ContentFragments savedState = ... 

<span style='color:#666666; font-style: italic'>// Выполнять асинхронно</span>
dlr.setAsync( true )
 
<span style='color:#666666; font-style: italic'>// Если поддерживается, то использовать докачку по возможности</span>
dlr.setAllowPartialContent( true )

<span style='color:#666666; font-style: italic'>// Если поддерживается, то принудительно использовать докачку</span>
dlr.setForcePartialDownload( true )

<span style='color:#666666; font-style: italic'>// файл в который производиться закачка</span>
ContentBuffer fileBuf = 
  new CFileBuffer(
    new java.io.File( 'filename' ) );

dlr.setContentBuffer( fileBuf )

<span style='color:#666666; font-style: italic'>// По завершению вызвать функцию</span>
dlr.onFinished( 
new Func1&lt;Object,HttpDownloader&gt;(){
  public Object apply(HttpDownloader dl){
    <span style='color:#666666; font-style: italic'>// закачка закончилась</span>
  }});

<span style='color:#666666; font-style: italic'>// Просмотр прогресса скачивания</span>
dlr.addListener( new HttpListenerAdapter(){
  protected void downloaderStateChanged(
     HttpDownloader.StateChangedEvent event,
     HttpDownloader downloader,
     HttpDownloader.State oldState,
     HttpDownloader.State newState
  ){
    <span style='color:#666666; font-style: italic'>// смена состояния закачки</span>
  }   

  protected void downloaderProgress(
    HttpDownloader.ProgressEvent event,
    HttpDownloader dl
  ){
    <span style='color:#666666; font-style: italic'>// прогресс закачки</span>
    long totalSize = dl.getTotalSize();
    long downloaded = dl.getDownloadedSize();
    
    <span style='color:#666666; font-style: italic'>// процент закачки</span>
    int pct = downloaded * 100 / totalSize;
  }

});

<span style='color:#666666; font-style: italic'>// Запуск скачивания</span>
if( savedState!=null ) {
  dlr.start(savedState);
} else { 
  dlr.start();
}

<span style='color:#666666; font-style: italic'>// Ожидание завершения</span>
dlr.waitForFinished()
 
<span style='color:#666666; font-style: italic'>// Сохранение состояния</span>
savedState = dlr.getFragments() 
 </pre>
 * 
 * <h2>State machine</h2>
 * <img src='http-downloader-states.png' style="border: none;" alt="Диаграмма состояний">
 * 
 * <h2>Счётчики и обработка ошибок</h2>
 * <p>
 * В процессе работы ведется подсчет ошибок, скорости передачи и т.д.
 * Все эти подсчеты хранятся в "счётчиках" (getCounters()).
 * </p>
 * 
 * <p>
 * Счетчики могут быть ограничивающие (MaxCounter, OverflowCounter) или не ограничивающие (IntCounter, LongCounter, ...).
 * И первые и вторые, при наступлении какого-то события, увеличивается соответствующее значение счетчика.
 * </p>
 * 
 * <p>
 * Ограничивающие счетчики - они определяют дальнейшее поведение при превышении порогового значения.
 * Так при превышении максимального значения (при max &gt;= 0) счетчика tryCounter - завершиться работа скачивания данных.
 * </p>
 * 
 * <p>
 OverflowCounter.
 А при превышении счетчика ошибок скачивания фрагментов (tryFagmentCounter) возможно два поведения:
 1 - Пропуск фрагмента, 2 - Остановка с ошибкой;. Поведение задается значением свойства overflowAction.
 </p>
 * 
 * <table summary="Описание методов">
 * 
 * <tr valign="bottom" style='font-weight:bold'>
 * <td>Имя / Метод</td> <td>Тип счетчика</td> <td>Описание</td>
 * </tr>
 * 
 * <tr valign="top">
 * <td>try / getTryCounter()</td> <td>MaxCounter</td> <td>Общий счетчик попыток получения данных</td>
 * </tr>
 * 
 * </table>
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class HttpDownloader
implements IsFinished, ResetCounters, GetCounters
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(HttpDownloader.class.getName());
    private static final Level logLevel = logger.getLevel();

    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected static final AtomicLong sequenceID = new AtomicLong();

    protected final ReentrantLock lock = new ReentrantLock();

    /**
     * Идентификатор объекта в пределах приложения
     */
    public final long id = sequenceID.incrementAndGet();

    /**
     * Конструктор
     * @param request тсочник (URL) с которого качается даные
     */
    @SuppressWarnings("ResultOfObjectAllocationIgnored")
    public HttpDownloader( HttpRequest request ){
        if( request==null )throw new IllegalArgumentException( "request==null" );
        this.mirrors = new Mirrors(lock, request);
        try{
            request.lock.lock();
            this.contentBuffer = request.contentBuffer;
            this.async = request.async;
            this.followRedirect = request.followRedirect;
            
            addListener(new HttpListenerAdapter(){
                @Override
                protected void downloaderResponse(ResponseEvent event, HttpDownloader downloader, HttpRequest request, HttpResponse response) {
                    getRequestsCounter().increment();
                }
            });
            
            getProgressCounters();
        }finally{
            request.lock.unlock();
        }
    }

    /**
     * Конструктор
     * @param mirrors источники с которого производится скачивание
     */
    public HttpDownloader( Mirrors mirrors ){
        if( mirrors==null )throw new IllegalArgumentException( "mirrors==null" );
        if( mirrors.getMirrors().length==0 )throw new IllegalArgumentException( "mirrors is empty" );
        
        this.mirrors = mirrors.clone(lock);
        
        addListener(new HttpListenerAdapter(){
            @Override
            protected void downloaderResponse(ResponseEvent event, HttpDownloader downloader, HttpRequest request, HttpResponse response) {
                getRequestsCounter().increment();
            }
        });
        
        getProgressCounters();
    }
    
    //<editor-fold defaultstate="collapsed" desc="events">
    //<editor-fold defaultstate="collapsed" desc="event class StateChangedEvent">
    /**
     * Уведомление о смене состояния
     */
    public static class StateChangedEvent extends HttpEvent {
        public StateChangedEvent( HttpDownloader downloader, State oldState, State newState ){
            this.httpDownloader = downloader;
            this.oldState = oldState;
            this.newState = newState;
        }
        
        protected HttpDownloader httpDownloader = null;
        protected State oldState = null;
        protected State newState = null;
        
        public HttpDownloader getHttpDownloader() {
            return httpDownloader;
        }
        
        public State getOldState() {
            return oldState;
        }
        
        public State getNewState() {
            return newState;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class RedirectEvent">
    /**
     * Уведомление о переходе
     */
    public static class RedirectEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected URL from = null;
        protected URL to = null;
        
        public RedirectEvent( HttpDownloader downloader, URL from, URL to ){
            this.downloader = downloader;
            this.from = from;
            this.to = to;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public URL getFrom() {
            return from;
        }
        
        public URL getTo() {
            return to;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class ProgressEvent">
    /**
     * Уведомление о прогрессе
     */
    public static class ProgressEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        
        public ProgressEvent( HttpDownloader downloader ){
            this.downloader = downloader;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class ResponseEvent">
    /**
     * Уведомление о создании запроса
     */
    public static class ResponseEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected HttpRequest request = null;
        protected HttpResponse response = null;
        
        public ResponseEvent( HttpDownloader downloader, HttpRequest request, HttpResponse response ){
            if( downloader==null )throw new IllegalArgumentException( "downloader==null" );
            if( response==null )throw new IllegalArgumentException( "response==null" );
            if( request==null )throw new IllegalArgumentException( "request==null" );
            this.downloader = downloader;
            this.request = request;
            this.response = response;
        }
        
        public ResponseEvent( HttpDownloader downloader, HttpResponse response ){
            if( downloader==null )throw new IllegalArgumentException( "downloader==null" );
            if( response==null )throw new IllegalArgumentException( "response==null" );
            this.downloader = downloader;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public HttpRequest getRequest() {
            if( request==null ){
                request = response.getRequest();
            }
            return request;
        }
        
        public HttpResponse getResponse() {
            return response;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class GetPartEvent">
    /**
     * Уведомление о начале получения фрагмента
     */
    public static class GetPartEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected GetPart getPart = null;
        protected ContentFragment fragment = null;
        
        public GetPartEvent( HttpDownloader downloader, GetPart getPart, ContentFragment fragment ){
            if( downloader==null )throw new IllegalArgumentException( "downloader==null" );
            this.downloader = downloader;
            
//            if( getPart==null )throw new IllegalArgumentException( "getPart==null" );
            this.getPart = getPart;
            
//            if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
            this.fragment = fragment;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public GetPart getGetPart() {
            return getPart;
        }
        
        public ContentFragment getFragment() {
            return fragment;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class GetPartStartedEvent">
    /**
     * Уведомление о начале получения фрагмента
     */
    public static class GetPartStartedEvent extends GetPartEvent {
        public GetPartStartedEvent( HttpDownloader downloader, GetPart getPart, ContentFragment fragment ) {
            super(downloader, getPart, fragment);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class GetPartFinishedEvent">
    /**
     * Уведомление о завершении получения фрагмента
     */
    public static class GetPartFinishedEvent extends GetPartEvent {
//        protected ResolveFailed resolveFailed = null;
        
        public GetPartFinishedEvent(
            HttpDownloader downloader,
            GetPart getPart,
            ContentFragment fragment
        ) {
            super(downloader, getPart, fragment);
        }
        
//        public GetPartFinishedEvent(
//            HttpDownloader downloader,
//            GetPart getPart,
//            ContentFragment fragment,
//            ResolveFailed resolveFailed
//        ) {
//            super(downloader, getPart, fragment);
//            this.resolveFailed = resolveFailed;
//        }
        
//        /**
//         * Поведение при провальном запросе
//         * @return поведение или null, если запрос считается успешным
//         */
//        public ResolveFailed getResolveFailed() {
//            return resolveFailed;
//        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class CountersResetedEvent">
    /**
     * Уведомление о сбросе счетчиков (ошибок)
     */
    public static class CountersResetedEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        
        public CountersResetedEvent(
            HttpDownloader downloader
        ) {
            this.downloader = downloader;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class RepeatOverflowEvent">
    /**
     * Уведомление о переполнении повторов
     */
    public static class RepeatOverflowEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected int currentRepeatCount = -1;
        protected int maxRepeatCount = -1;
        
        public RepeatOverflowEvent(
            HttpDownloader downloader,
            int currentRepeatCount,
            int maxRepeatCount
        ) {
            this.downloader = downloader;
            this.currentRepeatCount = currentRepeatCount;
            this.maxRepeatCount = maxRepeatCount;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public int getCurrentRepeatCount() {
            return currentRepeatCount;
        }
        
        public int getMaxRepeatCount() {
            return maxRepeatCount;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class RepeatFragmentEvent">
    /**
     * Уведомление о повторной закачке фрагмента
     */
    public static class RepeatFragmentEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected ContentFragment cf = null;
        
        public RepeatFragmentEvent(
            HttpDownloader downloader,
            ContentFragment cf
        ) {
            this.downloader = downloader;
            this.cf = cf;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public ContentFragment getContentFragment() {
            return cf;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class DownloadedFragmentEvent">
    /**
     * Уведомление о получении фрагмента
     */
    public static class DownloadedFragmentEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected ContentFragment cf = null;
        
        public DownloadedFragmentEvent(
            HttpDownloader downloader,
            ContentFragment cf
        ) {
            this.downloader = downloader;
            this.cf = cf;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public ContentFragment getContentFragment() {
            return cf;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class SkippedFragmentEvent">
    /**
     * Уведомление о пропуске фрагмента
     */
    public static class SkippedFragmentEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected ContentFragment cf = null;
        
        public SkippedFragmentEvent(
            HttpDownloader downloader,
            ContentFragment cf
        ) {
            this.downloader = downloader;
            this.cf = cf;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public ContentFragment getContentFragment() {
            return cf;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class FragmentErrorEvent">
    /**
     * Уведомление о ошибки проверки фрагмента
     */
    public static class FragmentErrorEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected ContentFragment cf = null;
        protected GetPart gp = null;
        protected Throwable err = null;
        protected FragmentValidator fv = null;
        
        public FragmentErrorEvent(
            HttpDownloader downloader,
            FragmentValidator fv,
            ContentFragment cf,
            GetPart gp,
            Throwable err
        ) {
            this.downloader = downloader;
            this.cf = cf;
            this.fv = fv;
            this.gp = gp;
            this.err = err;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public ContentFragment getContentFragment() {
            return cf;
        }
        
        public GetPart getGetPart() {
            return gp;
        }
        
        public FragmentValidator getFragmentValidator(){
            return fv;
        }
        
        public Throwable getError(){
            return err;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class ContentValidateEvent">
    /**
     * Уведомление о проверке контента
     */
    public static class ContentValidateEvent extends HttpEvent {
        protected HttpDownloader downloader = null;
        protected ContentFragments fragments = null;
        protected EventSet<ContentFragment> valid = null;
        
        public ContentValidateEvent(
            HttpDownloader downloader,
            ContentFragments fragments,
            EventSet<ContentFragment> valid
        )
        {
            this.downloader = downloader;
            this.fragments = fragments;
            this.valid = valid;
        }
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        public ContentFragments getFragments(){ return fragments; }
        public EventSet<ContentFragment> getValidated(){ return valid; }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class AddedValidFragmentEvent">
    /**
     * Уведомление о добавлении валидного фрагмента
     */
    public static class AddedValidFragmentEvent extends ContentValidateEvent {
        protected ContentFragment cf;
        
        public AddedValidFragmentEvent(
            HttpDownloader downloader,
            ContentFragments fragments,
            EventSet<ContentFragment> valid,
            ContentFragment cf
        )
        {
            super(downloader, fragments, valid);
            this.cf = cf;
        }
        
        public ContentFragment getContentFragment() {
            return cf;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class RemovedInvalidFragmentEvent">
    /**
     * Уведомление о удалии битого фрагмента
     */
    public static class RemovedInvalidFragmentEvent extends ContentValidateEvent {
        protected ContentFragment cf;
        
        public RemovedInvalidFragmentEvent(
            HttpDownloader downloader,
            ContentFragments fragments,
            EventSet<ContentFragment> valid,
            ContentFragment cf
        )
        {
            super(downloader, fragments, valid);
            this.cf = cf;
        }
        
        public ContentFragment getContentFragment() {
            return cf;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class RemovedInvalidFragmentEvent">
    /**
     * Уведомление о удалии битого фрагмента
     */
    public static class ContentValidateProgressEvent extends HttpEvent {
        public ContentValidateProgressEvent(
            HttpDownloader downloader,
            ContentValidatorProgressEvent pe
        ) {
            this.downloader = downloader;
            cf = pe.getSample();
            sampleNum = pe.getSampleNum();
            sampleTotal = pe.getSampleTotal();
            hashMatched = pe.isHashMatched();
            sampleFragments = pe.getSampleFragments();
            contentBuffer = pe.getContentBuffer();
            contentValidator = pe.getSource();
        }
        
        protected ContentValidator contentValidator;
        
        public ContentValidator getContentValidator() {
            return contentValidator;
        }
        
        protected ContentBuffer contentBuffer;
        
        public ContentBuffer getContentBuffer() {
            return contentBuffer;
        }
        
        protected ContentFragments sampleFragments;
        
        public ContentFragments getSampleFragments() {
            return sampleFragments;
        }
        
        protected ContentFragment cf;
        
        public ContentFragment getSample() {
            return cf;
        }
        
        protected HttpDownloader downloader;
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        protected boolean hashMatched = false;
        
        public boolean isHashMatched() {
            return hashMatched;
        }
        
        protected int sampleNum = -1;
        
        public int getSampleNum() {
            return sampleNum;
        }
        
        protected int sampleTotal = -1;
        
        public int getSampleTotal() {
            return sampleTotal;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class FinishWithErrorEvent">
    /**
     * Уведомление о завершении с ошибкой
     */
    public static class FinishWithErrorEvent extends HttpEvent {
        public FinishWithErrorEvent(
            HttpDownloader downloader,
            String message
        ) {
            this.downloader = downloader;
            this.message = message;
        }
        
        protected HttpDownloader downloader;
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        protected String message;
        
        public String getMessage() {
            return message;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="event class RetryEvent">
    /**
     * Уведомление о повторе попытки
     */
    public static class RetryEvent extends HttpEvent {
        public RetryEvent(
            HttpDownloader downloader,
            String message
        ) {
            this.downloader = downloader;
            this.message = message==null ? this.message : message;
        }
        
        protected HttpDownloader downloader;
        
        public HttpDownloader getHttpDownloader() {
            return downloader;
        }
        
        protected String message="no message";
        
        public String getMessage() {
            return message;
        }
    }
    //</editor-fold>
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="enum State">
    /**
     * Состояние Downloader
     */
    public static enum State {
        /**
         * Подготовка - начальное состояние
         */
        Prepare,

        /**
         * Запуск
         */
        Starting,

        /**
         * Пауза
         */
        Pause,

        /**
         * Скачивание в потоковом режиме
         */
        DownloadingStream,

        /**
         * Переключение на фрагментарный режим
         */
        SwitchPartialDownloading,

        /**
         * Фрагментарный режим
         */
        DownloadingPartial,

        /**
         * Проверка скаченных фрагментов
         */
        Validate,

        /**
         * Конечное состояние
         */
        Finished
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getLock()">
    /**
     * Вовзаращает блокировку
     * @return блокировка
     */
    public Lock getLock(){
        return lock;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentBuffer - Буфер данных">
    /**
     * Указывает буффер для content-а
     * @param contentBuffer буффер
     */
    public void setContentBuffer(ContentBuffer contentBuffer) {
        try{
            lock.lock();
            this.contentBuffer = contentBuffer;
            if( isLogFinest ){
                logFinest("setContentBuffer( {0} )",contentBuffer);
            }
        }finally{
            lock.unlock();
        }
    }

    /**
     * Буфер данных
     */
    protected ContentBuffer contentBuffer = null;

    /**
     * Указывает буффер для content-а
     * @return contentBuffer буффер
     */
    public ContentBuffer getContentBuffer(){
        try{
            lock.lock();
            if( contentBuffer==null ){
                HttpRequest[] reqsts = mirrors.getMirrors();
                if( reqsts!=null ){
                    for( HttpRequest req : reqsts ){
                        if( req==null )continue;
                        ContentBuffer cb = req.getContentBuffer();
                        if( cb!=null ){
                            contentBuffer = cb;
                            if( isLogFiner )logFiner("create content buffer from request");
                        }
                    }
                }
                if( contentBuffer==null ){
                    contentBuffer = new MemContentBuffer();
                    if( isLogFiner )logFiner("create content buffer in memory");
                }
            }
            return contentBuffer;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="mirrors - источники/зекала">
    /**
     * Зекрала/источники с которых происходит скачивание контента
     */
    private final Mirrors mirrors;
    
    /**
     * Зекрала/источники с которых происходит скачивание контента
     * @return зекрала
     */
    public Mirrors getMirrors(){
        return mirrors;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="listeners">
    protected final HttpListenersHelper httpListenerHelper = new HttpListenersHelper();

    public Set<HttpListener> getListeners() {
        if( isLogFinest )logFinest("getListeners()");
        return httpListenerHelper.getListeners();
    }

    public Closeable addListener(HttpListener listener) {
        logFiner("addListener( {0} )",listener);
        Closeable c = httpListenerHelper.addListener(listener);
        return c;
    }

    public Closeable addListener(HttpListener listener, boolean weakLink) {
        logFiner("addListener( {0}, {1} )",listener, weakLink);
        Closeable c= httpListenerHelper.addListener(listener, weakLink);
        return c;
    }

    public void removeListener(HttpListener listener) {
        logFiner("removeListener( {0}, {1} )",listener);
        httpListenerHelper.removeListener(listener);
    }

    public void fireEvent(HttpEvent event) {
        if(isLogFinest)logFinest("fireEvent( {0}, {1} )",event);
        httpListenerHelper.fireEvent(event);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="followRedirect : boolean - переходить по redirect">
    protected boolean followRedirect = true;

    /**
     * Переходить по redirect
     * @return true - переход по redirect
     */
    public boolean isFollowRedirect() {
        try{
            lock.lock();
            return followRedirect;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Переходить по redirect
     * @param followRedirect true - переход по redirect
     */
    public void setFollowRedirect(boolean followRedirect) {
        try{
            lock.lock();
            this.followRedirect = followRedirect;
            logFiner("setFollowRedirect( {0} )",followRedirect);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="downloadedSize">
    protected long downloadedSize = 0;
    
    /**
     * Указывает объем скаченного контента, который содержиться в фрагментах. <br>
     * Код: getFragments().getTotalDownloadedSize()
     * @return кол-во байт
     */
    public long getPartialDownloadedSize(){
        try{
            lock.lock();
            return getFragments().getTotalDownloadedSize();
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает объем скаченного контента. <br>
     * Для режима <b>partial</b> - getPartialDownloadedSize() <br>
     * Для режима <b>stream</b>  - getStreamDownloadedSize() <br>
     * @return кол-во байт
     */
    public long getDownloadedSize() {
        try{
            lock.lock();
            /*
            State s = getState();
            switch( s ){
                case Finished:
                    if( fragments==null ){
                        return streamDownloadedSize;
                    }else{
                        return fragments.getTotalDownloadedSize();
                    }
//                    break;
                case DownloadingStream:
                    return streamDownloadedSize;
//                    break;
                case DownloadingPartial:
                    if( fragments==null ){
                        return 0;
                    }else{
                        return fragments.getTotalDownloadedSize();
                    }
                default:
                    return 0;
            }*/
            if( isPartialMode() ){
                long s = getFragments().getTotalDownloadedSize();
//                long s = getPartialDownloadedSize();
                if( isLogFinest )logFinest("getDownloadedSize() = {0} partial mode", s);
                return s;
            }else{
                long s = streamDownloadedSize;
                if( isLogFinest )logFinest("getDownloadedSize() = {0} stream mode", s);
                return s;
            }
        }finally{
            lock.unlock();
        }
    }

    /**
     * Кол-во байт загруженных в режиме stream
     */
    protected long streamDownloadedSize = 0;

    /**
     * Указывает кол-во байт загруженных в режиме stream
     * @param size  кол-во байт загруженных в режиме stream
     */
    protected void setStreamDownloadedSize(long size){
        try{
            lock.lock();
            this.streamDownloadedSize = size;
            logFiner("setStreamDownloadedSize( {0} )",size);
        }finally{
            lock.unlock();
        }
        fireEvent(new ProgressEvent(HttpDownloader.this));
    }
    
    /**
     * Кол-во байт загруженных в режиме stream
     * @return кол-во байт в режиме stream
     */
    public long getStreamDownloadedSize(){
        try{
            lock.lock();
            return this.streamDownloadedSize;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentLength : long - заголовок content-length">
    protected long contentLength = -1;

    /**
     * Указывает конечный объем контента (Значение береться из заголовка CONTENT-LENGTH). <br>
     * @return кол-во байт или -1, если не известно
     */
    public long getContentLength() {
        return contentLength;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="totalSize : long - конечный объем конета">
    /**
     * Возвращает конечный объем конета. <br>
     * Для режима stream значение берется из contentLength <br>
     * Для режима partial значение берется из downloader.getFragments().getTotalSize()
     * @return Конечный объем
     * @see #isPartialMode()
     * @see #getFragments()
     * @see #getContentLength()
     */
    public long getTotalSize(){
        try{
            lock.lock();
            if( isPartialMode() ){
                long s = getFragments().getTotalSize();
                if( isLogFinest )logFinest("getTotalSize() = {0} partial mode()", s);
                return s;
            }else{
                long s = getContentLength();
                if( isLogFinest )logFinest("getTotalSize() = {0} stream mode", s);
                return s;
            }
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="async : boolean - Работать асинхронно">
    protected Boolean async = null;
    
    /**
     * Работать асинхронно (start)
     * @return true - работать асинхронно
     */
    public boolean isAsync(){
        try{
            lock.lock();
            if( async==null ){
                HttpRequest[] reqsts = mirrors.getMirrors();
                if( reqsts!=null ){
                    for( HttpRequest req : reqsts ){
                        if( req==null )continue;
                        async = req.isAsync();
                    }
                }
                if( async==null ){
                    async = false;
                }
            }
            if( isLogFinest )logFinest("set async = {0}", async);
            return async;
        }finally{
            lock.unlock();
        }
    }
    
    /**
     * Работать асинхронно (start)
     * @param async true - работать асинхронно
     */
    public void setAsync(boolean async){
        try{
            lock.lock();
            this.async = async;
            logFiner("setAsync( {0} )", async);
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="thread">
    protected Thread thread = null;
    /**
     * Возвращает thread в котром происходит исполнение
     * @return thread
     */
    public Thread getThread(){ return thread; }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="allowParialContent : boolean - Поддерживать частичную докачку">
    protected boolean allowParialContent = true;

    /**
     * Поддерживать частичную докачку фрагментов
     * @return true (по умолчанию)
     */
    public boolean isAllowParialContent() {
        try{
            lock.lock();
            return allowParialContent;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Поддерживать частичную докачку фрагментов
     * @param allowParialContent true (по умолчанию)
     */
    public void setAllowParialContent(boolean allowParialContent) {
        try{
            lock.lock();
            this.allowParialContent = allowParialContent;
            // TODO log it
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="forcePartialDownload : boolean - Форсировать частичную докачку">
    protected boolean forcePartialDownload = false;

    /**
     * Форсировать частичную докачку, даже если это не требуется
     * @return false (по умолчанию)
     */
    public boolean isForcePartialDownload() {
        try{
            lock.lock();
            return forcePartialDownload;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Форсировать частичную докачку, даже если это не требуется
     * @param forcePartialDownload false (по умолчанию)
     */
    public void setForcePartialDownload(boolean forcePartialDownload) {
        try{
            lock.lock();
            this.forcePartialDownload = forcePartialDownload;
            // TODO log it
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="serverContinueSupported : Boolean - Указывает, сервер поддерживает докачку">
    protected Boolean serverContinueSupported = null;

    /**
     * Указывает, сервер поддерживает докачку
     * @return null - если не известно, true/false - когда известно
     */
    public Boolean isServerContinueSupported() {
        try{
            lock.lock();
            return serverContinueSupported;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает, сервер поддерживает докачку
     * @param serverContinueSupported null - если не известно, true/false - когда известно
     */
    protected void setServerContinueSupported(Boolean serverContinueSupported) {
        try{
            lock.lock();
            this.serverContinueSupported = serverContinueSupported;
            // TODO log it
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="maxParallesGetParts : int - максимальное возможное кол-во паралельных потоков">
    /**
     * максимальное возможное кол-во паралельных потоков
     */
    protected int maxParallesGetParts = 1;

    /**
     * Вовзаращает максимальное возможное кол-во паралельных потоков
     * @return максимальное возможное кол-во паралельных потоков, по умолчанию = 1
     */
    public int getMaxParallesGetParts() {
        try{
            lock.lock();
            return maxParallesGetParts;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает максимальное возможное кол-во паралельных потоков
     * @param maxParallesGetParts максимальное возможное кол-во паралельных потоков, 1 и больше
     */
    public void setMaxParallesGetParts(int maxParallesGetParts) {
        if( maxParallesGetParts<1 )throw new IllegalArgumentException("maxParallesGetParts<1");
        try{
            lock.lock();
            this.maxParallesGetParts = maxParallesGetParts;
            // TODO log it
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentValidator">
    /**
     * Проверка скаченного контента
     */
    protected volatile ContentValidator contentValidator;
    private final CloseableSet contentValidatorCS = new CloseableSet();

    /**
     * Проверка скачанного контента
     * @return проверяющий объект
     */
    public ContentValidator getContentValidator(){
        try{
            lock.lock();
            return contentValidator;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Проверка скачанного контента
     * @param cv проверяющий объект
     */
    public void setContentValidator(ContentValidator cv){
        try{
            lock.lock();
            
            contentValidatorCS.closeAll();
            
            this.contentValidator = cv;
            
            if( cv instanceof BindHttpDownloader ){
                Closeable cl = ((BindHttpDownloader)cv).bindHttpDownloader(this);
                if( cl!=null ){
                    contentValidatorCS.add(cl);
                }
            }
            
            // TODO log it
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="fragmentValidator">
    /**
     * Проверка получения фрагмента
     */
    protected volatile FragmentValidator fragmentValidator;
    private final CloseableSet fragmentValidatorCS = new CloseableSet();

    /**
     * Проверка получения фрагмента
     * @return проверяющий объект
     */
    public FragmentValidator getFragmentValidator(){
        FragmentValidator fv = fragmentValidator;
        if( fv!=null )return fv;

        try{
            lock.lock();
            
            fragmentValidatorCS.closeAll();
            
            fv = fragmentValidator;
            
            if( fv==null ){
                fragmentValidator = new BasicFragmentValidator();
                fv = fragmentValidator;
                if( fv instanceof BindHttpDownloader ){
                    Closeable cl = ((BindHttpDownloader)fv).bindHttpDownloader(this);
                    if( cl!=null )fragmentValidatorCS.add(cl);
                }
            }
            
            // TODO log it
        }finally{
            lock.unlock();
        }

        return fv;
    }

    /**
     * Проверка получения фрагмента
     * @param fv проверяющий объект
     */
    public void setFragmentValidator(FragmentValidator fv){
        try{
            lock.lock();
            fragmentValidatorCS.closeAll();
            this.fragmentValidator = fv;
            if( fv instanceof BindHttpDownloader ){
                Closeable cl = ((BindHttpDownloader)fv).bindHttpDownloader(this);
                if( cl!=null )fragmentValidatorCS.add(cl);
            }
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="fragments - Фрагменты данных">
    protected Lock fragmentsLock = new ReentrantLock();

    protected ContentFragments fragments = null;

    /**
     * Фрагменты данных
     * @return фрагменты
     */
    public ContentFragments getFragments(){
        try{
            lock.lock();
            if( fragments==null ){
                fragments = new ContentFragments(fragmentsLock);
                // TODO log it
            }
            return fragments;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Установка фрагментов
     * @param fragments фрагменты
     */
    protected void setFragments(ContentFragments fragments){
        if( fragments==null )throw new IllegalArgumentException( "fragments==null" );
        try{
            lock.lock();
            this.fragments = fragments;
            // TODO log it
        }finally{
            lock.unlock();
        }
    }

    /**
     * Создание фрагмента
     * @param begin начало
     * @param end конец исключительно
     * @return фрагмент
     */
    protected ContentFragment createFragment( long begin, long end ){
        try{
            lock.lock();
            return new ContentFragment(begin, end, getFragments().getLock());
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="defaultFragmentSize : int - размер фрагмента по умолчанию">
    /**
     * размер фрагмента по умолчанию (64К)
     */
    protected int defaultFragmentSize = 1024 * 64;

    /**
     * Вовзаращает размер фрагмента по умолчанию (64К)
     * @return размер фрагмента по умолчанию
     */
    public int getDefaultFragmentSize() {
        try{
            lock.lock();
            return defaultFragmentSize;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает размер фрагмента по умолчанию
     * @param defFragmentSize размер фрагмента по умолчанию, 1 и больше
     */
    public void setDefaultFragmentSize(int defFragmentSize) {
        if( defFragmentSize<1 )throw new IllegalArgumentException("defFragmentSize<1");
        try{
            lock.lock();
            this.defaultFragmentSize = defFragmentSize;
            // TODO log it
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="started : Date - время начала">
    protected Date started = null;

    /**
     * Дата/время начала закачки (переход в состояние Starting)
     * @return время начала
     */
    public Date getStarted() {
        return started;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="finished : Date - время завершения">
    protected Date finished = null;

    /**
     * Дата/время завершения закачки (перехода в состояние Finished)
     * @return время завершения
     */
    public Date getFinished() {
        return finished;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="state : State - состояния объекта">
    protected State state = State.Prepare;

    /**
     * Получение состояния объекта
     * @return состояние
     */
    public State getState(){
//        try{
//            lock.lock();
            return state;
//        }finally{
//            lock.unlock();
//        }
    }

    protected void setState(State newState){
        State old = null;
        try{
            lock.lock();
            old = this.state;
            this.state = newState;
            if( old!=newState ){
                if( State.Starting.equals(newState) ){
                    started = new Date();
                }
                if( State.Finished.equals(newState) ){
                    finished = new Date();
                }
            }
            // TODO log it
        }finally{
            lock.unlock();
        }
        fireEvent(new StateChangedEvent(this, old, newState));
    }

    protected StateChangedEvent setStateAndRetEvent(State newState){
        State old = null;
        try{
            lock.lock();
            old = this.state;
            this.state = newState;
            if( old!=newState ){
                if( State.Starting.equals(newState) ){
                    started = new Date();
                }
                if( State.Finished.equals(newState) ){
                    finished = new Date();
                }
            }
            // TODO log it
        }finally{
            lock.unlock();
        }
        return new StateChangedEvent(this, old, newState);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isFinished()">
    /**
     * Объект находится в конечном состоянии, загрузка не производится
     * @return true - конечное состояние
     */
    @Override
    public boolean isFinished(){
        try{
            lock.lock();
            return State.Finished.equals(this.state);
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="signal - сигналы паузы/возобновления">
    /**
     * Сигнал при паралельной работы
     */
    protected static enum Signal {
        Pause,
        Resume
    }

    protected Signal signal = null;

    protected Signal getSignal() {
        try{
            lock.lock();
            return signal;
        }finally{
            lock.unlock();
        }
    }

    protected void setSignal(Signal signal) {
        try{
            lock.lock();
            this.signal = signal;
            // TODO log it
        }finally{
            lock.unlock();
        }
    }

    protected Signal checkSignal(){
        try{
            lock.lock();
            Signal s = signal;
            signal = null;
            return s;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Сигнал паузы
     */
    public void pause(){
        setSignal(Signal.Pause);
        // TODO log it
    }

    /**
     * Сигнал продолжения - снятие с паузы
     */
    public void resume(){
        setSignal(Signal.Resume);
        // TODO log it
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="partialMode : boolean - Режим закачки по частям">
    private boolean partialMode = false;

    public boolean isPartialMode(){
        return partialMode;
    }

    protected void setPartialMode(boolean mode){
        partialMode = mode;
        // TODO log it
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="start()">
    /**
     * Запуск на закачивание.
     * Можно вызывать в состоянии Prepare, Finished
     * @see State
     */
    public void start(){
        start(null);
    }

    /**
     * Возобновление скачивания
     * Можно вызывать в состоянии Prepare, Finished
     * @param fragments Фрагменты которые требуется докачать
     * @see State
     */
    public void start(ContentFragments fragments){
        StateChangedEvent ev = null;
        try{
            lock.lock();
            State s = getState();

            switch( s ){
                case Prepare:
                case Finished:
                    break;
                default:
                    throw new IllegalStateException("state not in ( Prepare, Finished )");
            }

            setPartialMode( false );
            resetCounters();

//            final HttpRequest req = request.clone();
//            logFine("start {0}", req.getUrl());

            StringBuilder sbStartUrls = new StringBuilder();
            for( HttpRequest req : mirrors.getMirrors() ){
                if( req==null )continue;
                
                URL u = req.getUrl();
                if( u==null )continue;

                if( sbStartUrls.length()>0 )sbStartUrls.append(", ");
                sbStartUrls.append(u.toString());
            }
            logFine( "start {0}",sbStartUrls.toString() );
            
            ev = setStateAndRetEvent( State.Starting );

            final ContentFragments cfs = fragments;
//            final Mirrors mirrors = new Mirrors(lock,req);

            Runnable r = new Runnable() {
                @Override
                public void run() {
                    HttpDownloader.this.run(mirrors,cfs);
                }
            };

            if( isAsync() ){
                HttpClient hc = null;
                
                HttpRequest[] reqsts = mirrors.getMirrors();
                if( reqsts!=null ){
                    for( HttpRequest req : reqsts ){
                        if( req==null )continue;
                        hc = req.getHttpClient();
                        break;
                    }
                }
                if( hc==null ){
                    hc = new HttpClient();
                }

                thread = hc.createDownloaderThread().apply(this, r);
                thread.start();
            }else{
                r.run();
            }
        }finally{
            lock.unlock();
        }

        if( ev!=null )fireEvent(ev);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="run()">
    /**
     * Начало закачки
     * @param mirrors запросы на скачивание
     * @param fragments ранее сохраненное состояние (возможно null)
     */
    protected void run( final Mirrors mirrors, ContentFragments fragments ){
        if( mirrors==null )throw new IllegalArgumentException( "mirrors==null" );
        
        final State thisState = State.Starting;
        
        CatchFirstStatus sts = null;
        int cycleCount = 0;
        int nrequest = 0;
        
        while( true ){
            cycleCount++;
            @SuppressWarnings("null")
            final Mirrors.NRequest nreq = mirrors.getNRequest(nrequest);
            final HttpRequest request = nreq.getRequest();
            
//            setCurrentRequest(request);

            // Завершение работы при цикле redirect
            if( Thread.interrupted() ){
                finishStreamDownload();
                return;
            }

            // Пауза работы при цикле redirect
            Signal s = checkSignal();
            if( Signal.Pause.equals(s) ){
                setState(State.Pause);
                while( true ){
                    s = checkSignal();
                    if( s==null )continue;
                    if( Signal.Resume.equals(s) ){
                        setState(thisState);
                        break;
                    }
                    Thread.yield();
                }
            }

            // Выполнение первого запроса в асинх
            logFine("run {0}", request.getUrl());

            HttpRequest req = request;
            req.setContentBuffer(getContentBuffer());
            req.setContentOffsetStart(0);
            req.getHttpHeaders().setRangeHeader(null);
            req.setAsync(true);

            // TODO возложить обработку redirect на HttpResponse
            HttpResponse res = req.createResponse();
            res.setFollowRedirect(isFollowRedirect());
            
            HttpListenerAdapter redirectListener = new HttpListenerAdapter(){
                @Override
                protected void responseRedirect(HttpResponse.RedirectEvent event, HttpResponse response, URL from, URL to) 
                {
                    HttpDownloader.RedirectEvent re = new RedirectEvent(HttpDownloader.this, from, to);
                    HttpDownloader.this.fireEvent(re);
                    if( mirrors!=null ){
                        mirrors.redirect(nreq.getMirrorId(), from, to);
                    }
                }
            };
            res.addListener(redirectListener);

            fireEvent(new ResponseEvent(this, request, res));
            res.start();

            while( true ){
                sts = null;
                
                // Завершение работы при цикле redirect
                if( Thread.interrupted() ){
                    res.stop();
                    finishStreamDownload();
                    return;
                }

                // Пауза работы при цикле redirect
                s = checkSignal();
                if( Signal.Pause.equals(s) ){
                    res.pause();
                    setState(State.Pause);
                    while( true ){
                        s = checkSignal();
                        if( s==null )continue;
                        if( Signal.Resume.equals(s) ){
                            res.resume();
                            setState(thisState);
                            break;
                        }
                        Thread.yield();
                    }
                }

                if( res.isFinished() )break;

                HttpResponse.State st = res.getState();
                
                // TODO хм... вот тут надо подумать
                if( !HttpResponse.State.Downloading.equals(st) ){
                    Thread.yield();
                    continue;
                }

                // Скачивание началось
                HttpHeaders hh = res.getHttpHeaders();
                if( hh!=null ){
                    sts = catchFirstResponse(mirrors, request, res, fragments);
                    break;
                }
            }

            if( sts==null )sts = catchFirstResponse(mirrors, request, res, fragments);
            if( sts==null )break;
            if( sts.isRepeat() ){
//                URL redirect = sts.getRedirect();
//                if( redirect!=null ){
//                    URL from = req.getUrl();
//                    URL to = redirect;
//
//                    logFine("redirect {0} -> {1}", from, to);
//
//                    fireEvent(new RedirectEvent(this, from, to));
//
//                    HttpRequest reqredir = req.clone();
//                    reqredir.getHttpHeaders().setReferer(from.toString());
//                    reqredir.setUrl(to);
//                    
//                    mirrors.setNRequestRedirect(nrequest, res, reqredir);
//                }else{
                    nrequest++;
//                }
                continue;
            }else{
                break;
            }
        }
    }
    //</editor-fold>

    protected static HttpStatusHelper httpStatusHelper = new HttpStatusHelper();

    //<editor-fold defaultstate="collapsed" desc="catchFirstResponse()">
    protected static class CatchFirstStatus {
        public CatchFirstStatus(boolean repeat){
            this.repeat = repeat;
        }
        
//        public CatchFirstStatus(boolean repeat, URL redirect){
//            this.repeat = repeat;
//            this.redirect = redirect;
//        }
        
        protected boolean repeat;

        public boolean isRepeat() {
            return repeat;
        }

        public void setRepeat(boolean repeat) {
            this.repeat = repeat;
        }
        
//        protected URL redirect;
//
//        public URL getRedirect() {
//            return redirect;
//        }
//
//        public void setRedirect(URL redirect) {
//            this.redirect = redirect;
//        }
    }
    
    /**
     * Захвать первого ответа
     * @param mirrors зеркала с которых производится скачивание
     * @param request запрос
     * @param response ответ
     * @param fragments ранее сохраненное состояние (возможно null)
     * @return Обработка ответа - repeat=true - повтор запроса
     */
    protected CatchFirstStatus catchFirstResponse(
        Mirrors mirrors,
        HttpRequest request,
        HttpResponse response,
        final ContentFragments fragments
    ){
        int statusCode= response.getStatusCode();
        HttpHeaders resHH = response.getHttpHeaders();

        if( HttpResponse.State.Finished.equals( response.getState() ) &&
            response.getErrors().size()>0
        ){
            String errMessage = "";

            int errCount = response.getErrors().size();
            logFine("response has errors ({0}):", errCount);
            int i=-1;
            for( Throwable err : response.getErrors() ){
                i++;
                logFine("response error {0}/{1}: {2}", i+1, errCount, err.getMessage());
                
                errMessage += ( errMessage.length()>0 ? "\n" : "" )
                              + (
                                    (i+1)+"/"+(response.getErrors().size())
                                    +" "+(err.getClass().getName()+":")+err.getMessage()
                              )
                    ;
            }
            
            if( !response.isFinished() ){
                response.stop();
            }

            if( errCount==1 ){
                Throwable err = response.getErrors().get(0);
                //<editor-fold defaultstate="collapsed" desc="обработка ошибок">
                if( err instanceof java.net.UnknownHostException ){
                    MaxCounter sct = getTryStartCounter().increment();
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getUnknownHostCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof URISyntaxException ){
                    MaxCounter sct = getTryStartCounter().increment();
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getUriSyntaxErrCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof UnknownServiceException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getUnknownServiceCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof SocketTimeoutException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getSocketTimeoutCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof ProtocolException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getProtocolErrCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof PortUnreachableException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getPortUnreachableCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof NoRouteToHostException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getNoRouteToHostCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof MalformedURLException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getMalformedURLCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof HttpRetryException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getHttpRetryCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof ConnectException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getConnectErrCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof BindException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getBindErrCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }else if( err instanceof SocketException ){
                    MaxCounter tct = getTryCounter().increment();
                    MaxCounter cnt = getSocketErrCounter().increment();
                    MaxCounter sct = getTryStartCounter().increment();
                    if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                        fireEvent(
                            new RetryEvent(
                                this,
                                "retry "
                                    +(cnt.get()+"/"+cnt.getMax())
                                    +" on "+err.getClass().getName()+": "+err.getMessage()));
                        return new CatchFirstStatus(true);
                    }
                }
                //</editor-fold>
            }else{
                MaxCounter tct = getTryCounter().increment();
                MaxCounter cnt = getMultipleErrorsCounter().increment();
                MaxCounter sct = getTryStartCounter().increment();
                if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                    fireEvent(
                        new RetryEvent(
                            this,
                            "retry "
                                +(cnt.get()+"/"+cnt.getMax())
                                +" on "+errMessage));
                    return new CatchFirstStatus(true);
                }
            }

            finishAsError("response error = "+errMessage);
            return new CatchFirstStatus(false);
        }

        if( resHH==null ){
            logFine("no response headers, stop response");
            if( !response.isFinished() )response.stop();
            
            MaxCounter tct = getTryCounter().increment();
            MaxCounter cnt = getHeadersNotExistsCounter().increment();
            MaxCounter sct = getTryStartCounter().increment();
            if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                fireEvent(
                    new RetryEvent(
                        this, 
                        "retry "
                            +(cnt.get()+"/"+cnt.getMax())
                            +" on no response headers"));
                return new CatchFirstStatus(true);
            }
            finishAsError("server error, status code = "+statusCode);
            return new CatchFirstStatus(false);
        }

        if( statusCode==-1 ){
            logFine("status code = -1, stop response");
            if( !response.isFinished() )response.stop();
            
            MaxCounter tct = getTryCounter().increment();
            MaxCounter cnt = getBadStatusCounter().increment();
            MaxCounter sct = getTryStartCounter().increment();
            if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                fireEvent(
                    new RetryEvent(
                        this, 
                        "retry "
                            +(cnt.get()+"/"+cnt.getMax())
                            +" on status code = -1"));
                return new CatchFirstStatus(true);
            }
            finishAsError("server error, status code = "+statusCode);
            return new CatchFirstStatus(false);
        }

        this.contentLength = resHH.getContentLength();
        logFine("content length = {0}", contentLength);

        logFine(
            "catchFirstResponse\n"
                + "  url = {2}"
                + "  status = {0}"
                + "\n{1}"
            ,
            statusCode,
            Text.indent(response.getHttpHeaders().toString(),"\n","[Response] " ),
            request.getUrl()
        );
        
        String statusMessage = response.getStatusMessage();
        
        if( httpStatusHelper.isSuccess(statusCode) ){
            boolean suppContinue = isSupportedContinueDownload(response);
            setServerContinueSupported(suppContinue);
            if( suppContinue && isForcePartialDownload() ){
                switchPartialDownload(mirrors, request, response, fragments);
            }else{
                continueStreamDownload(request, response);
            }
//        }else if( httpStatusHelper.isRedirect(statusCode) ){
//            if( isFollowRedirect() ){
//                String target = resHH.getLocation();
//                if( target!=null ){
//                    if( !response.isFinished() )response.stop();
//                    try {
//                        URL targetURL = new URL(target);
////                        redirect(request, response, targetURL, fragments);
//                        return new CatchFirstStatus(true, targetURL);
//                    } catch (MalformedURLException ex) {
//                        Logger.getLogger(HttpDownloader.class.getName()).log(Level.SEVERE, null, ex);
//                        MaxCounter cnt = getBadLocationCounter().increment();
//                        if( !cnt.overflow() ){
//                            // TODO rewrite from recursion to cycle - ok
//                            fireEvent(
//                                new RetryEvent(
//                                    this, 
//                                    "retry "
//                                        +(cnt.get()+"/"+cnt.getMax())
//                                        +" on bad location"));
////                            run(request, fragments);
//                            return new CatchFirstStatus(true);
//                        }
//                        finishAsError("redirect failed, location("+target+") not parsed:"+ex.getMessage());
//                    }
//                }else{
//                    if( !response.isFinished() )response.stop();
//                    MaxCounter cnt = getLocationNotSetCounter().increment();
//                    if( !cnt.overflow() ){
//                        // TODO rewrite from recursion to cycle - ok
//                        fireEvent(
//                            new RetryEvent(
//                                this, 
//                                "retry "
//                                    +(cnt.get()+"/"+cnt.getMax())
//                                    +" on location not set"));
////                        run(request, fragments);
//                        return new CatchFirstStatus(true);
//                    }
//                    finishAsError("redirect, but location not set");
//                }
//            }else{
//                finishAsError("redirect not allowed");
//                if( !response.isFinished() )response.stop();
//            }
        }else if( httpStatusHelper.isServerError(statusCode) ){
            if( !response.isFinished() )response.stop();
            
            MaxCounter tct = getTryCounter().increment();
            MaxCounter cnt = getServerErrorCounter().increment();
            MaxCounter sct = getTryStartCounter().increment();
            if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                fireEvent(
                    new RetryEvent(
                        this, 
                        "retry "
                            +(cnt.get()+"/"+cnt.getMax())
                            +" on server error = "+statusCode+" "+statusMessage));
                return new CatchFirstStatus(true);
            }
            finishAsError("server error, status code = "+statusCode+" "+statusMessage);
        }else if( httpStatusHelper.isClientError(statusCode) ){
            if( !response.isFinished() )response.stop();
            
            MaxCounter tct = getTryCounter().increment();
            MaxCounter cnt = getClientErrorCounter().increment();
            MaxCounter sct = getTryStartCounter().increment();
            if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                fireEvent(
                    new RetryEvent(
                        this, 
                        "retry "
                            +(cnt.get()+"/"+cnt.getMax())
                            +" on client error = "+statusCode+" "+statusMessage));
                return new CatchFirstStatus(true);
            }
            finishAsError("client error, status code = "+statusCode+" "+statusMessage);
        }else if( httpStatusHelper.isInfoStatus(statusCode) ){
            if( !response.isFinished() )response.stop();
            
            MaxCounter tct = getTryCounter().increment();
            MaxCounter sct = getTryStartCounter().increment();
            MaxCounter cnt = getInfoStatusCounter().increment();
            
            if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                fireEvent(
                    new RetryEvent(
                        this, 
                        "retry "
                            +(infoStatusCounterName+"="+cnt.get()+"/"+cnt.getMax())
                            +(" "+tryStartCounterName+"="+sct.get()+"/"+sct.getMax())
                            +(" "+tryCounterName+"="+tct.get()+"/"+tct.getMax())
                            +" on info status = "+statusCode+" "+statusMessage));
                return new CatchFirstStatus(true);
            }
            finishAsError("info response status code = "+statusCode+" "+statusMessage);
        }else{
            if( !response.isFinished() )response.stop();
            
            MaxCounter tct = getTryCounter().increment();
            MaxCounter sct = getTryStartCounter().increment();
            MaxCounter cnt = getUnSupportedStatusCounter().increment();
            
            if( !cnt.overflow() && !tct.overflow() && !sct.overflow() ){
                fireEvent(
                    new RetryEvent(
                        this, 
                        "retry "
                            +(unSupportedStatusCounterName+"="+cnt.get()+"/"+cnt.getMax())
                            +(" "+tryStartCounterName+"="+sct.get()+"/"+sct.getMax())
                            +(" "+tryCounterName+"="+tct.get()+"/"+tct.getMax())
                            +" on unknow status = "+statusCode+" "+statusMessage));
                return new CatchFirstStatus(true);
            }
            
            finishAsError("un supported response status code = "+statusCode+" "+statusMessage);
        }
        
        return new CatchFirstStatus(false);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isSupportedContinueDownload()">
    public static boolean isSupportedContinueDownload(HttpResponse response){
        HttpHeaders hh = response.getHttpHeaders();
        String acptRngs = hh.getAcceptRanges();
        long contentLen = hh.getContentLength();
        if( acptRngs!=null && acptRngs.trim().equalsIgnoreCase("bytes") &&
            contentLen >= 0 ){
            // Поддерживается фрагментарное скачивание
            return true;
        }else{
            // фрагментарное скачивание не поддерживается
            return false;
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="Цикл частичной(partial) загрузки">
    //<editor-fold defaultstate="collapsed" desc="switchPartialDownload() - Переключение в режим фрагментарной загруки">
    /**
     * Переключение в режим фрагментарной загруки из потокового режима
     * @param mirrors зеркала с которых производится скачивание
     * @param request исходный запрос
     * @param firstRes первый ответ
     * @param fragments состояние скаченных фрагментов или null
     */
    protected void switchPartialDownload(
        Mirrors mirrors,
        HttpRequest request,
        HttpResponse firstRes,
        final ContentFragments fragments
    ){
        logFine("switchPartialDownload");
        setState(State.SwitchPartialDownloading);
        
        // Остановка первого запроса
        logFine("stop first response");
        if( !firstRes.isFinished() )firstRes.stop();
        
        // Проверка скаченного объема
        this.downloadedSize = firstRes.getDownloadedSize();
        logFine("stream downloaded {0} total {1}", downloadedSize, contentLength);
        fireEvent(new ProgressEvent(HttpDownloader.this));
        
        // Получен весь объем
        if( downloadedSize>=contentLength ){
            finishStreamDownload();
            return;
        }
        
        // Инициализация фрагментов
        if( fragments==null ){
            InitFragments initFragments
                //            = new FastEqualsFragments(this);
                = new EqualsFragments(this);
            
            initFragments.initFragments(getFragments(), firstRes, contentLength);
            logFinest("fragments:\n{0}",getFragments());
        }else{
            setFragments(fragments);
        }
        
        // Переход к скачиванию фрагментов
        continuePartialDownload(
            mirrors,
            request
        );
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="continuePartialDownload() - Продолжение фрагментарной загрузки">
    /**
     * Продолжение фрагментарной загрузки
     * @param mirrors зеркала
     * @param request текущий запрос
     */
    protected void continuePartialDownload(
        Mirrors mirrors,
        HttpRequest request
    ){
        // Переключение в режим по фрагментарной закачки
        setState(State.DownloadingPartial);
        setPartialMode(true);
        
        // Основной цикл посылки запросов
        GetPartBuilder gpBuilder = createGetPartBuilder(mirrors, request);
        
        // Паралельные запросы
        WeakHashMap<GetPart, ContentFragment> getPartFragment = new WeakHashMap<GetPart, ContentFragment>();
        WeakHashMap<ContentFragment,GetPart> fragmentGetPart = new WeakHashMap<ContentFragment,GetPart>();
//        GetPartList gparts = new GetPartList();
gparts.clear();

// Очередь сообщений
final Queue<Runnable> qe = new LinkedBlockingQueue<Runnable>();

while( true ){
    // Разослать уведомления
    fireQueueEvents(qe);
    
    // Сигнал остановки
    if( checkStopSignal(gparts, getPartFragment) ){
        return;
    }
    
    // Сигнал паузы
    checkPauseSignal(gparts);
    
    // Проверка полученых фрагментов
    if (checkFinishedParts(gparts, getPartFragment, fragmentGetPart)) {
        // завершаем работу
        gparts.stopAll();
        finishPartialDownload();
        return;
    }
    
    List<ContentFragment> lfragments = getFragments().getNotDownloaded();
    if( lfragments.isEmpty() && gparts.isEmpty() ){
        // все запросы завершены и не осталось больше фрагментов для закачки
        break;
    }
    
    // Есть не дополученные части
    if( !lfragments.isEmpty() ){
        // Создать запросы на получение
        createNewParts(lfragments, gparts, gpBuilder, fragmentGetPart, getPartFragment, qe);
    }
    
    Thread.yield();
}

// все запросы завершены и не осталось больше фрагментов для закачки
// завершаем работу
gparts.stopAll();

// нужнали проверка содержимого ?
ContentValidator cv = getContentValidator();
if( cv!=null ){
    validateContent( mirrors, request, cv );
    return;
}

finishPartialDownload();
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createGetPartBuilder() -  Создание построителя GetPart">
    /**
     * Создание построителя GetPart
     * @param mirrors зеркала
     * @param request Запрос
     * @return построитель
     */
    protected GetPartBuilder createGetPartBuilder(
        Mirrors mirrors,
        HttpRequest request
    ){
        GetPartBuilder gpBuilder = new GetPartBuilder(
            this,
            mirrors,
            request,
            getContentBuffer()
        );
        
        return gpBuilder;
    }
    //</editor-fold>

    /**
     * Паралельные загрузки
     */
    private final GetPartList gparts = new GetPartList(lock);

    //<editor-fold defaultstate="collapsed" desc="getGetParts() - Возвращает паралельные загрузки">
    /**
     * Возвращает паралельные загрузки
     * @return загрузки
     */
    public GetPart[] getGetParts(){
        return gparts.toArray(new GetPart[]{});
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createNewParts() - Создает паралельные запросы на полчение недостающих частей">
    /**
     * Создает паралельные запросы на полчение недостающих частей
     * @param lfragments Список недостающих фрагментов
     * @param gparts Паралельные потоки закачек
     * @param gpBuilder Создание паралельного потока закачки
     * @param fragmentGetPart Соответствие недостающий фрагмент - паралельные поток
     * @param getPartFragment Соответствие паралельные поток - недостающий фрагмент
     * @param qe Очередь сообщений
     * @return Созданые части
     */
    protected CreatedNewParts createNewParts(
        List<ContentFragment> lfragments,
        GetPartList gparts,
        GetPartBuilder gpBuilder,
        WeakHashMap<ContentFragment, GetPart> fragmentGetPart,
        WeakHashMap<GetPart, ContentFragment> getPartFragment,
        final Queue<Runnable> qe
    ) {
        CreatedNewParts res = new CreatedNewParts();

        res.createdGetParts = new LinkedHashMap<GetPart, ContentFragment>();
        res.skippedFragmentsInWork = new LinkedHashSet<ContentFragment>();
        res.skippedFragmentsInWorkMap = new LinkedHashMap<GetPart, ContentFragment>();

        // есть фрагменты необходиые для закачки
//        for( ContentFragment f : fragmentGetPart.keySet() ){
//            if( f!=null ){
//                // этот фрагмент в работе
//                lfragments.remove(f);
//
//                res.skippedFragmentsInWork.add(f);
//                res.skippedFragmentsInWorkMap.put(fragmentGetPart.get(f), f);
//            }
//        }
        for( GetPart gp : gparts.toArray(new GetPart[]{}) ){
            if( !gp.isFinished() ){
                for( ContentFragment cf : lfragments.toArray(new ContentFragment[]{}) ){
                    if( cf.equalsRange(gp.getFragment()) ){
                        lfragments.remove(cf);
                    }
                }
            }
        }

        // есть фрагменты которые не в работе еще и ожидают докачки
        if( !lfragments.isEmpty() ){
            // не привышен лими паралельных запросов
            int max_paralles_get_parts = getMaxParallesGetParts();
            if( gparts.size() < max_paralles_get_parts ){

                // сколько создадим запросов
                int newParallesCO = max_paralles_get_parts - gparts.size();
                for( int i=0; i< newParallesCO; i++ ){
                    if( lfragments.isEmpty() )break;

                    // фрагмент который надо получить
                    ContentFragment cf = lfragments.get(0);
                    lfragments.remove(cf);

                    // Запрос
                    final GetPart gp = gpBuilder.getPart(cf);
                    gparts.add(gp);

                    res.createdGetParts.put(gp, cf);

                    // устанавливаем связь между фрагментом и запросом
                    getPartFragment.put(gp, cf);
                    fragmentGetPart.put(cf, gp);

                    addGetPartProgressListener(qe, gp, cf);

                    // запускаем запрос
                    gp.start();
                    logFiner( "start getpart {0}"+cf );

                    // уведомление о начале
                    fireEvent(new GetPartStartedEvent(this, gp, cf));

                    HttpResponse gpres = gp.getResponse();
                    if( gpres!=null ){
                        // добавление запроса
//                        addHttpResponse(gpres);

                        // уведомление о запросе
                        fireEvent(new ResponseEvent(this, gp.getRequest(), gpres));
                    }
                }
            }
        }

        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="checkFinishedParts()">
    /**
     * Повтор закачки указанного фрагмента
     * @param cf фрагмент
     */
    protected void repeatDownload( ContentFragment cf ){
        cf.setDownloadedSize(0);
        if( isLogFiner )logFiner("repeat {0}", cf);
        if( isLogFinest )logFinest("fragments:\n{0}", getFragments());
        fireEvent(new RepeatFragmentEvent(this, cf));
    }

    /**
     * Пропуск закачки указанного фрагмента
     * @param cf фрагмент
     */
    protected void skipDownload( ContentFragment cf ){
        cf.setDownloadedSize(cf.getSize());
        if( isLogFiner )logFiner("skip {0}", cf);
        if( isLogFinest )logFinest("fragments:\n{0}",getFragments());
        fireEvent(new SkippedFragmentEvent(this, cf));
    }

    /**
     * Отметка фрагмента как закаченного
     * @param cf фрагмент
     */
    protected void markDownloaded( ContentFragment cf ){
        cf.setDownloadedSize(cf.getSize());
        if( isLogFiner )logFiner("downloaded {0}", cf);
        if( isLogFinest )logFinest("fragments:\n{0}",getFragments());
        fireEvent(new DownloadedFragmentEvent(this, cf));
    }

    /**
     * Проверить наличие законченных запросов, и удаляет успешно заверненные.
     * @param gparts Список запросов
     * @param getPartFragment Карта запрос - фрагмент
     * @param fragmentGetPart Карта фрагмент - запрос
     * @return true - прекратить выполнение цикла; false - продолжить выполнение
     */
    protected boolean checkFinishedParts(
        GetPartList gparts,
        WeakHashMap<GetPart, ContentFragment> getPartFragment,
        WeakHashMap<ContentFragment, GetPart> fragmentGetPart
    ) {
        FragmentValidator fragmentValidator = getFragmentValidator();

        for (GetPart gpart : gparts.toArray(new GetPart[]{})) {
            if (gpart.isFinished()) {
                ContentFragment cf = getPartFragment.get(gpart);

                try{
                    if( cf!=null ){
                        try{
                            boolean valid = fragmentValidator.validate(gpart, cf);
                            if( !valid ){
                                
                                OverflowCounter
                                    inc = getTryFagmentCounter().increment();
                                
                                MaxCounter tct = getTryCounter().increment();
                                if( tct.overflow() ){
                                    throw new Error( 
                                        "overflow try counter (current="+tct.get()+" max="+tct.getMax()+")" 
                                    );
                                }

                                if( inc.overflow() ){
                                    switch( inc.getOverflowAction() ){
                                        case Skip:
                                            skipDownload(cf);
                                            break;
                                        case Stop:
                                            logFine("repeat overflow (count={0}, max={1})",
                                                inc.get(), inc.getMax());
                                            fireEvent(new RepeatOverflowEvent(this, 
                                                inc.get(), inc.getMax()));
                                            return true;
                                    }
                                }else{
                                    repeatDownload(cf);
                                }
                            }else{
                                markDownloaded(cf);
                            }
                        }catch( Throwable err ){
                            logException(err);
                            fireEvent(new FragmentErrorEvent(this, fragmentValidator, cf, gpart, err));
                            return true;
                        }
                    }
                }
                finally{
                    // Запрос успешно получил все необходимые данные
                    // удалется из списка активных запросов
                    gparts.remove(gpart);

                    // удаление отработанных связей

                    // fragmentGetPart.remove(cf); - это связь остается для поиска откуда был получен фрагмент
                    getPartFragment.remove(gpart);
                }
                // уведомление о завершнии
                fireEvent(new GetPartFinishedEvent(this, gpart, cf));
            }
            // уведомление о прогрессе
            fireEvent(new ProgressEvent(HttpDownloader.this));
        }
        
        for( GetPart gp : getPartFragment.keySet().toArray(new GetPart[]{}) ){
            if( gp==null )continue;
            if( gp.isFinished() ){
                ContentFragment cf = getPartFragment.get(gp);
                getPartFragment.remove(gp);
                if( cf!=null ){
                    fragmentGetPart.remove(cf);
                }
            }
        }
        
        return false;
    }
    //</editor-fold>
    
    /**
     * Проверенные, валидные фрагменты
     */
    protected EventSet<ContentFragment> validatedFragments
        = new BasicEventSet<ContentFragment>();

    //<editor-fold defaultstate="collapsed" desc="validatedFragments - фрагменты которые валидны">
    /**
     * Возвращает фрагменты которые валидны
     * @return валидные фрагменты
     */
    public EventSet<ContentFragment> getValidatedFragments(){
        return validatedFragments;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="validateContent() - Проверка содержимого">
    /**
     * Проверка содержимого
     * @param mirrors зеркала с которых производится скачивание
     * @param request исходный запрос
     * @param cv проверяющий
     */
    protected void validateContent( Mirrors mirrors, HttpRequest request, ContentValidator cv ){
        if( cv==null ){
            finishPartialDownload();
            return;
        }
        
        ContentFragments cfs = getFragments();
        if( cfs==null ){
            finishPartialDownload();
            return;
        }
        
        ContentBuffer cbuff = getContentBuffer();
        if( cbuff==null ){
            finishPartialDownload();
            return;
        }
        
        setState(State.Validate);
        
        Set<ContentFragment> badFragments;
        ContentValidation cvres;
        
        CloseableSet cscv = new CloseableSet();
        if( cv instanceof ContentValidatorSender ){
            ContentValidatorListener lst = new ContentValidatorListener() {
                @Override
                public void contentValidatorEvent(ContentValidatorEvent event) {
                    if( event!=null && event instanceof ContentValidatorProgressEvent ){
                        ContentValidatorProgressEvent pe = (ContentValidatorProgressEvent)event;
                        
                        ContentValidateProgressEvent cvpe
                            = new ContentValidateProgressEvent(
                                HttpDownloader.this,
                                pe );
                        
                        HttpDownloader.this.fireEvent(cvpe);
                    }
                }
            };
            
            cscv.add(
                ((ContentValidatorSender)cv).addListener(lst)
            );
        }
        
        try{
            cvres = cv.validate(cbuff, cfs);
            if( cvres.getBad() instanceof Set ){
                badFragments = (Set)cvres.getBad();
            }else{
                badFragments = new LinkedHashSet<ContentFragment>();
                for( ContentFragment cf : cvres.getBad() ){
                    if( cf!=null )badFragments.add(cf);
                }
            }
        }catch( Throwable err ){
            logException(err);
            finishAsError("content validate error: "+err.getMessage());
            cscv.closeAll();
            return;
        }
        cscv.closeAll();
        
        Set<ContentFragment> valids;
        if( cvres.getGood() instanceof Set ){
            valids = (Set)cvres.getGood();
        }else{
            valids = new LinkedHashSet<ContentFragment>();
            for( ContentFragment cf : cvres.getGood() ){
                if( cf!=null )valids.add(cf);
            }
        }
        
        if( badFragments==null || badFragments.isEmpty() ){
            EventSet<ContentFragment> vfrags = getValidatedFragments();
            for( ContentFragment cf : valids ){
                boolean addedGood = vfrags.add(cf);
                if( addedGood ){
                    fireEvent(new AddedValidFragmentEvent(this, cfs, vfrags, cf));
                }
            }
            
            finishPartialDownload();
            return;
        }
        
        Set<ContentFragment> repeats = new LinkedHashSet<ContentFragment>();
        
        for( ContentFragment badcf : badFragments.toArray(new ContentFragment[]{}) ){
            int i = cfs.indexOf(badcf);
            if( i>=0 ){
                Fragment ocf = cfs.get(i);
                if( ocf instanceof ContentFragment ){
                    ContentFragment cf = (ContentFragment)ocf;
//                    cf.setDownloadedSize(0);
                    repeats.add(cf);
                }
            }else{
                ContentFragment cf = badcf.clone();
                cfs.add(cf);
//                cf.setDownloadedSize(0);
                repeats.add(cf);
            }
        }
        
        EventSet<ContentFragment> vfrags = getValidatedFragments();
        for( ContentFragment cf : repeats ){
            valids.remove(cf);
            repeatDownload(cf);
            boolean removedBad = vfrags.remove(cf);
            if( removedBad ){
                fireEvent(new RemovedInvalidFragmentEvent(this, cfs, vfrags, cf));
            }
        }
        
        for( ContentFragment cf : valids ){
            boolean addedGood = vfrags.add(cf);
            if( addedGood ){
                fireEvent(new AddedValidFragmentEvent(this, cfs, vfrags, cf));
            }
        }
        
        MaxCounter tct = getTryCounter().clone();
        if( tct.overflow() ){
            finishAsError("overflow try counter: current="+tct.get()+" max="+tct.getMax());
            return;
        }
        
        run(mirrors, cfs);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="addGetPartProgressListener() - Добавляет подписчика на уведомление о прогрессе GetPart">
    /**
     * Добавляет подписчика на уведомление о прогрессе
     * @param qe Очередь сообщений
     * @param gp Паралельный поток
     * @param cf Фрагмент данных
     */
    protected void addGetPartProgressListener(
        final Queue<Runnable> qe,
        final GetPart gp,
        final ContentFragment cf
    ) {
        // добавляем подписчика на получение прогресса
        final CloseableSet closeListener = new CloseableSet();
        HttpListenerAdapter gpPartListener = new HttpListenerAdapter(){
            CloseableSet cs = closeListener;
            @Override
            protected void responseProgress(HttpResponse.ProgressEvent event, HttpResponse response) {
                // уведомляем о прогрессе
                Runnable r = new Runnable(){
                    @Override
                    public void run() {
                        HttpDownloader.this.fireEvent(new ProgressEvent(HttpDownloader.this));
                    }
                };
                qe.add(r);

                long size = response.getDownloadedSize();
                cf.setDownloadedSize(size);
            }

            @Override
            protected void responseStateChanged(HttpResponse.StateChangedEvent event, HttpResponse response, HttpResponse.State oldState, HttpResponse.State newState) {
                if( newState==HttpResponse.State.Finished ){
                    if( cs!=null ){
                        cs.closeAll();
                        cs = null;
                    }
                }
            }
        };
        closeListener.add(
            gp.addResponseListener(gpPartListener,false)
        );
    }
    //</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="fireQueueEvents()">
    /**
     * Разослать все уведомления в очереди
     * @param qe очередь уведомлений
     */
    protected void fireQueueEvents(final Queue<Runnable> qe) {
        // Очередь сообщений
        while( !qe.isEmpty() ){
            Runnable r = qe.poll();
            if( r==null )break;
            r.run();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="checkStopSignal()">
    /**
     * Проверяет сигнал остановки
     * @param gparts паралельные потоки
     * @param getPartFragment Карта поток &lt; = &gt; фрагмент
     * @return true - завершить работу цикла; false - продолжить
     */
    protected boolean checkStopSignal(GetPartList gparts, WeakHashMap<GetPart, ContentFragment> getPartFragment) {
        // принудитеьлно завершаем работу
        if (Thread.interrupted()) {
            gparts.stopAll();

            // уведомление о завершении
            for( GetPart gp2 : gparts ){
                fireEvent(new GetPartFinishedEvent(this, gp2, getPartFragment.get(gp2)));
            }
            gparts.clear();

            finishAsError("forced stop");
            return true;
        }
        return false;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="checkPauseSignal()">
    /**
     * Проверяет сигнал паузы
     * @param gparts паралельные потоки
     */
    protected void checkPauseSignal(GetPartList gparts) {
        // пауза ?
        Signal sgnl = checkSignal();
        if( Signal.Pause.equals(sgnl) ){
            gparts.pauseAll();
            setState(State.Pause);
            while( true ){
                sgnl = checkSignal();
                if( Signal.Resume.equals(sgnl) ){
                    gparts.resumeAll();
                    setState(State.DownloadingPartial);
                    break;
                }
                Thread.yield();

                if( Thread.currentThread().isInterrupted() ){
                    setState(State.DownloadingPartial);
                    break;
                }
            }
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="continueStreamDownload()">
    /**
     * Продолжение потоковой загрузки
     * @param request запрос
     * @param firstRes первый ответ
     */
    protected void continueStreamDownload(HttpRequest request, HttpResponse firstRes){
        logFine("continueStreamDownload");

        setState(State.DownloadingStream);

        final Queue<Runnable> qe = new LinkedBlockingQueue<Runnable>();

        CloseableSet closeset = new CloseableSet();
        HttpListener l = new HttpListenerAdapter(){
            @Override
            protected void responseProgress(HttpResponse.ProgressEvent event, HttpResponse response) {
                //HttpDownloader.this.setStreamDownloadedSize(response.getDownloadedSize());
                final HttpResponse fresponse = response;
                qe.add(new Runnable() {
                    @Override
                    public void run() {
                        HttpDownloader.this.setStreamDownloadedSize(fresponse.getDownloadedSize());
                    }
                });
            }
        };
        Closeable c = firstRes.addListener(l,true);
        closeset.add(c);

//        boolean interput = false;

        while( true ){
            if( firstRes.isFinished() )break;

            if( Thread.interrupted() ){
                firstRes.stop();
                break;
            }

            while (!qe.isEmpty()){
                Runnable r = qe.poll();
                if( r==null )break;
                r.run();
            }

            // Пауза работы
            Signal s = checkSignal();
            if( Signal.Pause.equals(s) ){
                firstRes.pause();
                setState(State.Pause);
                while( true ){
                    s = checkSignal();
                    if( s==null )continue;
                    if( Signal.Resume.equals(s) ){
                        firstRes.resume();
                        setState(State.DownloadingStream);
                        break;
                    }
//                    Thread.yield();
                }
            }

//            Thread.yield();
        }

        closeset.closeAll();

//        try{
//            lock.lock();
//            this.downloadedSize = firstRes.getDownloadedSize();
//        }finally{
//            lock.unlock();
//        }

        HttpDownloader.this.setStreamDownloadedSize(firstRes.getDownloadedSize());

        finishStreamDownload();
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="счётчики">
    //<editor-fold defaultstate="collapsed" desc="progressCounters">
    private volatile ProgressCounters progressCounters;
    
    /**
     * Общий счетчик попыток
     * @return Счетчик попыток
     */
    public ProgressCounters getProgressCounters(){
        if( progressCounters!=null ){
            return progressCounters;
        }
        synchronized( this ){
            if( progressCounters==null ){
                progressCounters = new ProgressCounters(this);
            }
        }
        return progressCounters;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="tryCounter - Общий (0/100)">
    private volatile MaxCounter tryCounter;
    private static final String tryCounterName = "try";

    /**
     * Общий счетчик попыток
     * @return Счетчик попыток
     */
    public MaxCounter getTryCounter(){
        if( tryCounter!=null ){
            return tryCounter;
        }
        synchronized( this ){
            if( tryCounter==null ){
                tryCounter = new MaxCounter(0,100,lock);
                counters.put(tryCounterName, tryCounter);
            }
        }
        return tryCounter;
    }
    
    /**
     * Общий счетчик попыток
     * @param cnt счетчик
     */
    public void setTryCounter( MaxCounter cnt ){
        synchronized(this){
            getTryCounter().assign(cnt);
        }
    }
    
    /**
     * Счетчик пропущенного заголовка перенаправления
     * @return Счетчик пропущенного заголовка перенаправления
     */
    /*
    public MaxCounter getLocationNotSetCounter(){
        if( locationNoSetCounter!=null ){
            return locationNoSetCounter;
        }
        synchronized( this ){
            if( locationNoSetCounter==null ){
                locationNoSetCounter = new MaxCounter(0,100,lock);
                counters.put("err.locationNoSet", locationNoSetCounter);
            }
        }
        return badLocationCounter;
    }
    
    public void setLocationNotSetCounter( MaxCounter cnt ){
        synchronized(this){
            if( cnt==null )
                cnt = new MaxCounter(0,100,lock);
            else 
                cnt = new MaxCounter(cnt, lock);
            locationNoSetCounter = cnt;
            counters.put("err.locationNoSet", locationNoSetCounter);
        }
    }
    */
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="tryStartCounter - Счетчик попыток старта (0/-1)">
    private volatile MaxCounter tryStartCounter;
    
    private static final String tryStartCounterName = "try.start";
    
    /**
     * Счетчик попыток старта
     * @return Счетчик попыток
     */
    public MaxCounter getTryStartCounter(){
        if( tryStartCounter!=null ){
            return tryStartCounter;
        }
        synchronized( this ){
            if( tryStartCounter==null ){
                tryStartCounter = new MaxCounter(0,-1,lock);
                counters.put(tryStartCounterName, tryStartCounter);
            }
        }
        return tryStartCounter;
    }
    
    /**
     * Счетчик попыток старта
     * @param cnt счетчик
     */
    public void setTryStartCounter( MaxCounter cnt ){
        synchronized(this){
            getTryStartCounter().assign(cnt);
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="счетчик tryFagmentCounter - (0/-1)">
    private volatile OverflowCounter tryFagmentCounter;
    private static final String tryFragmentCounterName = "try.fragment";
    
    public OverflowCounter getTryFagmentCounter(){
        if( tryFagmentCounter!=null ){
            return tryFagmentCounter;
        }
        synchronized( this ){
            if( tryFagmentCounter==null ){
                tryFagmentCounter = new OverflowCounter(0,-1,this.lock);
                counters.put(tryFragmentCounterName, tryFagmentCounter);
            }
        }
        return tryFagmentCounter;
    }
    
    public void setTryFagmentCounter(OverflowCounter cnt){
        synchronized( this ){
            getTryFagmentCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="multipleErrorsCounter - Счетчик ошибок (более 1 ошибки за раз) старта (0/-1)">
    private volatile MaxCounter multipleErrorsCounter;
    private static final String multipleErrorsCounterName = "try.multipleErrors";
    
    /**
     *  Счетчик ошибок (более 1 ошибки за раз) старта
     * @return Счетчик попыток
     */
    public MaxCounter getMultipleErrorsCounter(){
        if( multipleErrorsCounter!=null ){
            return multipleErrorsCounter;
        }
        synchronized( this ){
            if( multipleErrorsCounter==null ){
                multipleErrorsCounter = new MaxCounter(0,-1,lock);
                counters.put(multipleErrorsCounterName, multipleErrorsCounter);
            }
        }
        return multipleErrorsCounter;
    }
    
    /**
     * Счетчик ошибок (более 1 ошибки за раз) старта
     * @param cnt счетчик
     */
    public void setMultipleErrorsCounter( MaxCounter cnt ){
        synchronized(this){
            getMultipleErrorsCounter().assign(cnt);
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="unknownHostCounter - (0/-1)">
    private volatile MaxCounter unknownHostCounter;
    private static final String unknownHostCounterName = "err.unknownHost";
    
    /**
     * Счетчик ошибок UnknownHostException
     * @return счетчик ошибок UnknownHostException
     */
    public MaxCounter getUnknownHostCounter(){
        if( unknownHostCounter!=null ){
            return unknownHostCounter;
        }
        synchronized( this ){
            if( unknownHostCounter==null ){
                unknownHostCounter = new MaxCounter(0,-1,lock);
                counters.put(unknownHostCounterName, unknownHostCounter);
            }
        }
        return unknownHostCounter;
    }
    
    public void setUnknownHostCounter( MaxCounter cnt ){
        synchronized(this){
            getUnknownHostCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="headersNotExistsCounter - (0/-1)">
    private volatile MaxCounter headersNotExistsCounter;
    private final static String headersNotExistsCounterName = "err.headersNotExists";
    
    /**
     * Счетчик ошибок отсуствия заголовка
     * @return Счетчик ошибок отсуствия заголовка
     */
    public MaxCounter getHeadersNotExistsCounter(){
        if( headersNotExistsCounter!=null ){
            return headersNotExistsCounter;
        }
        synchronized( this ){
            if( headersNotExistsCounter==null ){
                headersNotExistsCounter = new MaxCounter(0,-1,lock);
                counters.put(headersNotExistsCounterName, headersNotExistsCounter);
            }
        }
        return headersNotExistsCounter;
    }
    
    public void setHeadersNotExistsCounter( MaxCounter cnt ){
        synchronized(this){
            getHeadersNotExistsCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="badStatusCounter - (0/-1)">
    private volatile MaxCounter badStatusCounter;
    private static final String badStatusCounterName = "err.badStatus";
    
    /**
     * Счетчик плохого статуса HTTP ответа
     * @return Счетчик плохого статуса HTTP ответа
     */
    public MaxCounter getBadStatusCounter(){
        if( badStatusCounter!=null ){
            return badStatusCounter;
        }
        synchronized( this ){
            if( badStatusCounter==null ){
                badStatusCounter = new MaxCounter(0,-1,lock);
                counters.put(badStatusCounterName, badStatusCounter);
            }
        }
        return badStatusCounter;
    }
    
    public void setBadStatusCounter( MaxCounter cnt ){
        synchronized(this){
            getBadStatusCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="serverErrorCounter - (0/-1)">
    private volatile MaxCounter serverErrorCounter;
    private static final String serverErrorCounterName = "err.serverError";
    
    /**
     * Счетчик ошибок сервера, 500 &gt;= HTTP статус 
     * @return Счетчик ошибок сервера, 500 &gt;= HTTP статус 
     */
    public MaxCounter getServerErrorCounter(){
        if( serverErrorCounter!=null ){
            return serverErrorCounter;
        }
        synchronized( this ){
            if( serverErrorCounter==null ){
                serverErrorCounter = new MaxCounter(0,-1,lock);
                counters.put(serverErrorCounterName, serverErrorCounter);
            }
        }
        return serverErrorCounter;
    }

    public void setServerErrorCounter( MaxCounter cnt ){
        synchronized(this){
            getServerErrorCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="clientErrorCounter - (0/-1)">
    private volatile MaxCounter clientErrorCounter;
    private static final String clientErrorCounterName = "err.clientError";
    
    /**
     * Счетчик ошибок клиента, 400 &gt;= HTTP статус &lt; 500
     * @return Счетчик ошибок клиента, 400 &gt;= HTTP статус  &lt; 500
     */
    public MaxCounter getClientErrorCounter(){
        if( clientErrorCounter!=null ){
            return clientErrorCounter;
        }
        synchronized( this ){
            if( clientErrorCounter==null ){
                clientErrorCounter = new MaxCounter(0,-1,lock);
                counters.put(clientErrorCounterName, clientErrorCounter);
            }
        }
        return clientErrorCounter;
    }

    public void setClientErrorCounter( MaxCounter cnt ){
        synchronized(this){
            getClientErrorCounter().assign(cnt);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="infoStatusCounter - (0/-1)">
    private volatile MaxCounter infoStatusCounter;
    private final static String infoStatusCounterName = "err.infoStatus";
    
    /**
     * Счетчик информационных сообщений, 100 &gt;= HTTP статус &lt; 200
     * @return Счетчик ошибок клиента, 100 &gt;= HTTP статус  &lt; 200
     */
    public MaxCounter getInfoStatusCounter(){
        if( infoStatusCounter!=null ){
            return infoStatusCounter;
        }
        synchronized( this ){
            if( infoStatusCounter==null ){
                infoStatusCounter = new MaxCounter(0,100,lock);
                counters.put(infoStatusCounterName, infoStatusCounter);
            }
        }
        return infoStatusCounter;
    }

    public void setInfoStatusCounter( MaxCounter cnt ){
        synchronized(this){
            getInfoStatusCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="unSupportedStatusCounter - (0/-1)">
    private volatile MaxCounter unSupportedStatusCounter;
    private static final String unSupportedStatusCounterName = "err.unSupportedStatus";
    
    /**
     * Счетчик не поддерживаемых статусов HTTP ответа
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getUnSupportedStatusCounter(){
        if( unSupportedStatusCounter!=null ){
            return unSupportedStatusCounter;
        }
        synchronized( this ){
            if( unSupportedStatusCounter==null ){
                unSupportedStatusCounter = new MaxCounter(0,-1,lock);
                counters.put(unSupportedStatusCounterName, unSupportedStatusCounter);
            }
        }
        return unSupportedStatusCounter;
    }

    public void setUnSupportedStatusCounter( MaxCounter cnt ){
        synchronized(this){
            getUnSupportedStatusCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="uriSyntaxErrCounter - (0/-1)">
    private volatile MaxCounter uriSyntaxErrCounter;
    private static final String uriSyntaxErrCounterName = "err.uriSyntaxErr";
    
    /**
     * Счетчик URISyntaxException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getUriSyntaxErrCounter(){
        if( uriSyntaxErrCounter!=null ){
            return uriSyntaxErrCounter;
        }
        synchronized( this ){
            if( uriSyntaxErrCounter==null ){
                uriSyntaxErrCounter = new MaxCounter(0,-1,lock);
                counters.put(uriSyntaxErrCounterName, uriSyntaxErrCounter);
            }
        }
        return uriSyntaxErrCounter;
    }

    public void setUriSyntaxErrCounterCounter( MaxCounter cnt ){
        synchronized(this){
            getUriSyntaxErrCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="unknownServiceCounter - (0/-1)">
    private volatile MaxCounter unknownServiceCounter;
    private static final String unknownServiceCounterName = "err.unknownService";
    
    /**
     * Счетчик UnknownServiceException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getUnknownServiceCounter(){
        if( unknownServiceCounter!=null ){
            return unknownServiceCounter;
        }
        synchronized( this ){
            if( unknownServiceCounter==null ){
                unknownServiceCounter = new MaxCounter(0,-1,lock);
                counters.put(unknownServiceCounterName, unknownServiceCounter);
            }
        }
        return unknownServiceCounter;
    }

    public void setUnknownServiceCounter( MaxCounter cnt ){
        synchronized(this){
            getUnknownServiceCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="socketTimeoutCounter - (0/-1)">
    private volatile MaxCounter socketTimeoutCounter;
    private static final String socketTimeoutCounterName = "err.socketTimeout";
    
    /**
     * Счетчик SocketTimeoutException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getSocketTimeoutCounter(){
        if( socketTimeoutCounter!=null ){
            return socketTimeoutCounter;
        }
        synchronized( this ){
            if( socketTimeoutCounter==null ){
                socketTimeoutCounter = new MaxCounter(0,-1,lock);
                counters.put(socketTimeoutCounterName, socketTimeoutCounter);
            }
        }
        return socketTimeoutCounter;
    }

    public void setSocketTimeoutCounter( MaxCounter cnt ){
        synchronized(this){
            getSocketTimeoutCounter().assign(cnt);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="protocolErrCounter - (0/-1)">
    private volatile MaxCounter protocolErrCounter;
    private static final String protocolErrCounterName = "err.protocolException";
    
    /**
     * Счетчик ProtocolException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getProtocolErrCounter(){
        if( protocolErrCounter!=null ){
            return protocolErrCounter;
        }
        synchronized( this ){
            if( protocolErrCounter==null ){
                protocolErrCounter = new MaxCounter(0,-1,lock);
                counters.put(protocolErrCounterName, protocolErrCounter);
            }
        }
        return protocolErrCounter;
    }

    public void setProtocolErrCounter( MaxCounter cnt ){
        synchronized(this){
            getProtocolErrCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="portUnreachableCounter - (0/-1)">
    private volatile MaxCounter portUnreachableCounter;
    private static final String portUnreachableCounterName = "err.portUnreachable";
    
    /**
     * Счетчик PortUnreachableException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getPortUnreachableCounter(){
        if( portUnreachableCounter!=null ){
            return portUnreachableCounter;
        }
        synchronized( this ){
            if( portUnreachableCounter==null ){
                portUnreachableCounter = new MaxCounter(0,-1,lock);
                counters.put(portUnreachableCounterName, portUnreachableCounter);
            }
        }
        return portUnreachableCounter;
    }

    public void setPortUnreachableCounter( MaxCounter cnt ){
        synchronized(this){
            getPortUnreachableCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="malformedURLCounter - (0/-1)">
    private volatile MaxCounter malformedURLCounter;
    private static final String malformedURLCounterName = "err.malformedURL";
    
    /**
     * Счетчик MalformedURLException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getMalformedURLCounter(){
        if( malformedURLCounter!=null ){
            return malformedURLCounter;
        }
        synchronized( this ){
            if( malformedURLCounter==null ){
                malformedURLCounter = new MaxCounter(0,-1,lock);
                counters.put(malformedURLCounterName, malformedURLCounter);
            }
        }
        return malformedURLCounter;
    }

    public void setMalformedURLCounter( MaxCounter cnt ){
        synchronized(this){
            getMalformedURLCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="httpRetryCounter - (0/-1)">
    private volatile MaxCounter httpRetryCounter;
    private static final String httpRetryCounterName = "err.httpRetry";
    
    /**
     * Счетчик HttpRetryException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getHttpRetryCounter(){
        if( httpRetryCounter!=null ){
            return httpRetryCounter;
        }
        synchronized( this ){
            if( httpRetryCounter==null ){
                httpRetryCounter = new MaxCounter(0,-1,lock);
                counters.put(httpRetryCounterName, httpRetryCounter);
            }
        }
        return httpRetryCounter;
    }

    public void setHttpRetryCounter( MaxCounter cnt ){
        synchronized(this){
            getHttpRetryCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="socketErrCounter - (0/-1)">
    private volatile MaxCounter socketErrCounter;
    private final static String socketErrCounterName = "err.socketException";
    
    /**
     * Счетчик SocketException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getSocketErrCounter(){
        if( socketErrCounter!=null ){
            return socketErrCounter;
        }
        synchronized( this ){
            if( socketErrCounter==null ){
                socketErrCounter = new MaxCounter(0,-1,lock);
                counters.put(socketErrCounterName, socketErrCounter);
            }
        }
        return socketErrCounter;
    }

    /**
     * Счетчик SocketException
     * @param cnt Счетчик ошибок клиента
     */
    public void setSocketErrCounter( MaxCounter cnt ){
        synchronized(this){
            getSocketErrCounter().assign(cnt);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="bindErrCounter - (0/-1)">
    private volatile MaxCounter bindErrCounter;
    private static final String bindErrCounterName = "err.bindException";
    
    /**
     * Счетчик BindException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getBindErrCounter(){
        if( bindErrCounter!=null ){
            return bindErrCounter;
        }
        synchronized( this ){
            if( bindErrCounter==null ){
                bindErrCounter = new MaxCounter(0,-1,lock);
                counters.put(bindErrCounterName, bindErrCounter);
            }
        }
        return bindErrCounter;
    }

    /**
     * Счетчик BindException
     * @param cnt Счетчик ошибок клиента
     */
    public void setBindErrCounter( MaxCounter cnt ){
        synchronized(this){
            getBindErrCounter().assign(cnt);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="connectErrCounter - (0/-1)">
    private volatile MaxCounter connectErrCounter;
    private final static String connectErrCounterName = "err.connectException";
    
    /**
     * Счетчик ConnectException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getConnectErrCounter(){
        if( connectErrCounter!=null ){
            return connectErrCounter;
        }
        synchronized( this ){
            if( connectErrCounter==null ){
                connectErrCounter = new MaxCounter(0,-1,lock);
                counters.put(connectErrCounterName, connectErrCounter);
            }
        }
        return connectErrCounter;
    }

    public void setConnectErrCounter( MaxCounter cnt ){
        synchronized(this){
            getConnectErrCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="noRouteToHostCounter - (0/-1)">
    private volatile MaxCounter noRouteToHostCounter;
    private static final String noRouteToHostCounterName = "err.noRouteToHost";
    
    /**
     * Счетчик NoRouteToHostException
     * @return Счетчик ошибок клиента
     */
    public MaxCounter getNoRouteToHostCounter(){
        if( noRouteToHostCounter!=null ){
            return noRouteToHostCounter;
        }
        synchronized( this ){
            if( noRouteToHostCounter==null ){
                noRouteToHostCounter = new MaxCounter(0,-1,lock);
                counters.put(noRouteToHostCounterName, noRouteToHostCounter);
            }
        }
        return noRouteToHostCounter;
    }

    public void setNoRouteToHostCounter( MaxCounter cnt ){
        synchronized(this){
            getNoRouteToHostCounter().assign(cnt);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="workTime">
    private volatile LongCounter workTimeCounter;
    private final static String workTimeCounterName = "workTime";
    
    public LongCounter getWorkTimeCounter(){
        if( workTimeCounter!=null ){
            return workTimeCounter;
        }
        synchronized( this ){
            if( workTimeCounter==null ){
                workTimeCounter = new LongCounter(this.lock){
                    @Override
                    public Long get() {
                        Date dStarted = HttpDownloader.this.getStarted();
                        Date dFinished = HttpDownloader.this.getFinished();
                        if( dStarted==null )return 0L;
                        if( dFinished==null || dStarted.after(dFinished) )return new Date().getTime() - dStarted.getTime();
                        return dFinished.getTime() - dStarted.getTime();
                    }
                };
                counters.put(workTimeCounterName, workTimeCounter);
            }
        }
        return workTimeCounter;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="requests">
    private volatile IntCounter requestsCounter;
    private static final String requestsCounterName = "requests";
    
    public IntCounter getRequestsCounter(){
        if( requestsCounter!=null ){
            return requestsCounter;
        }
        synchronized( this ){
            if( requestsCounter==null ){
                requestsCounter = new IntCounter(this.lock);
                counters.put(requestsCounterName, requestsCounter);
            }
        }
        return requestsCounter;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="counters map">
    private volatile boolean countersMapInit = false;
    
    private final Map<String,Counter> counters = new WeakHashMap<String, Counter>();
    
    /**
     * Счетчики
     * @return счетчики
     */
    @Override
    public Map<String,Counter> getCounters(){
        if( !countersMapInit ){
            synchronized(this){
                if( !countersMapInit ){
                    countersMapInit = true;
                    getTryCounter();
                    getTryStartCounter();
                    
                    getSocketErrCounter();
                    getBindErrCounter();

                    getTryFagmentCounter();
                    
//                    getBadLocationCounter();
                    getBadStatusCounter();
                    getClientErrorCounter();
                    getHeadersNotExistsCounter();
                    getInfoStatusCounter();
//                    getLocationNotSetCounter();
                    getServerErrorCounter();
                    getUnSupportedStatusCounter();
                    getUnknownHostCounter();
                    getUriSyntaxErrCounter();
                    getUnknownServiceCounter();
                    getSocketTimeoutCounter();
                    getProtocolErrCounter();
                    getPortUnreachableCounter();
                    getNoRouteToHostCounter();
                    getMalformedURLCounter();
                    getHttpRetryCounter();
                    getConnectErrCounter();
                    
                    getWorkTimeCounter();
                    getRequestsCounter();
                    
                    getFragmentValidator();
                }
            }
        }
        return counters;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="resetCounters()">
    /**
     * Сброс счетчиков
     */
    @Override
    public void resetCounters() {
        try{
            lock.lock();
//            getRepeatCounter().reset();
//            getUnknownHostCounter().reset();
            if( counters!=null ){
                for( Counter c : counters.values() ){
                    if( c!=null && c instanceof ResetCounter ){
                        ((ResetCounter)c).reset();
                    }
                }
            }
        }
        finally{
            lock.unlock();
        }

        FragmentValidator fv = getFragmentValidator();
        if( fv instanceof ResetCounters ){
            ((ResetCounters)fv).resetCounters();
        }

        ContentValidator cv = getContentValidator();
        if( cv instanceof ResetCounters ){
            ((ResetCounters)cv).resetCounters();
        }

        validatedFragments.clear();

        fireEvent(new CountersResetedEvent(this));
    }
//</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="finishStreamDownload()">
    /**
     * Завершение работы потокогового скачивания
     */
    protected void finishStreamDownload(){
        setState(State.Finished);
        logFine("finish download");
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="finishAsError()">
    /**
     * Завершить работу паралельных потоков с сообщением. <br>
     * Вызывает finishAsError(String message)
     * @param gparts паралельные потоки
     * @param getPartFragment соответствие паралельный поток - фрагмент
     * @param message сообщение
     * @see #finishAsError(java.lang.String)
     */
    protected void finishAsError(GetPartList gparts, WeakHashMap<GetPart, ContentFragment> getPartFragment, String message){
        gparts.stopAll();
        // уведомление о завершении
        for( GetPart gp2 : gparts ){
            fireEvent(new GetPartFinishedEvent(this, gp2, getPartFragment.get(gp2)));
        }
        gparts.clear();
        finishAsError(message);
    }

    /**
     * Завершение работы с сообщением
     * @param message сообщение
     */
    protected void finishAsError(String message){
        fireEvent(new FinishWithErrorEvent(this, message));
        setState(State.Finished);
        logWarning("finish download with error: {0}",message);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="finishPartialDownload()">
    /**
     * Завершение работы фрагментарного скачивания
     */
    protected void finishPartialDownload(){
        setState(State.Finished);
        logFine("finish download");
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="waitForFinished()">
    /**
     * Ожидание завершения загрузки.
     * можно вызывать когда работает в параллельном/асинхронном режиме.
     * @see #isAlive()
     */
    public void waitForFinished(){
        Thread tt = Thread.currentThread();
        Thread t = null;
        try{
            if( lock!=null )lock.lock();
            t = thread;
        }finally{
            if( lock!=null )lock.unlock();
        }
        if( t==null )throw new IllegalStateException("not async started");
        if( t.equals(tt) ){
            throw new Error("thread can be locked");
        }
        while( !isFinished() || t.isAlive() ){
            Thread.yield();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isAlive()">
    /**
     * Возвращает признак паралельной работы,
     * соответ возможность вызвать методы waitForFinished() и stop()
     * @return true - объект работает в параллельном/асинхронном режиме.
     */
    public boolean isAlive(){
        Thread t = null;
        try{
            if( lock!=null )lock.lock();
            t = thread;
        }finally{
            if( lock!=null )lock.unlock();
        }
        if( t==null )return false;
        return t.isAlive();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="stop()">
    /**
     * прерываение загрузки,
     * можно вызывать когда работает в параллельном/асинхронном режиме.
     * @see #isAlive()
     */
    @SuppressWarnings("CallToThreadYield")
    public void stop(){
        Thread tt = Thread.currentThread();
        Thread t = null;
        try{
            if( lock!=null )lock.lock();
            t = thread;
        }finally{
            if( lock!=null )lock.unlock();
        }
        if( t==null )throw new IllegalStateException("not async started");
        if( t.equals(tt) ){
            throw new Error("thread can be locked");
        }
        while( !isFinished() || t.isAlive() ){
            t.interrupt();
            Thread.yield();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="onFinished()">
    /**
     * Вызывать код когда запрос перейдет в состояние finished. <br>
     * Вызов можеть производиться из другого потока (если был асихронный вызов). <br>
     * Код будет вызван не более одного раза.
     * @param runOnFinsihed код который надо вызвать
     * @return закрытие объектов
     */
    public Closeable onFinished( final Func1<Object,HttpDownloader> runOnFinsihed ){
        if( runOnFinsihed==null )throw new IllegalArgumentException( "runOnFinsihed==null" );

        final CloseableSet cset = new CloseableSet();

        HttpListener listener = new HttpListenerAdapter(){
            private CloseableSet cs = cset;
            private Func1<Object,HttpDownloader> call = runOnFinsihed;

            @Override
            protected void downloaderFinished(HttpDownloader.StateChangedEvent event, HttpDownloader downloader) {
                if( cs!=null ){
                    cs.closeAll();
                    cs = null;
                }

                if( call!=null ){
                    call.apply(downloader);
                    call = null;
                }
            }
        };

        cset.add(
            addListener(listener, false)
        );

        Closeable cl = new Closeable() {
            @Override
            public void close() throws IOException {
                cset.closeAll();
            }
        };

        return cl;
    }
//</editor-fold>
}
