/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;


import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import xyz.cofe.text.Text;

/**
 * Http заголовки
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class HttpHeaders {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(HttpHeaders.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(HttpHeaders.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(HttpHeaders.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(HttpHeaders.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(HttpHeaders.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(HttpHeaders.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(HttpHeaders.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    private Map<String,List<String>> headers = null;
    
    protected HttpHeaders(Map<String,List<String>> multiMap){
        if( multiMap==null )throw new IllegalArgumentException( "multiMap==null" );
        this.headers = multiMap;
    }
    
    public HttpHeaders(){
        headers = new LinkedHashMap<String, List<String>>();
    }

    public HttpHeaders( HttpHeaders source ){
        if( source!=null ){
            this.headers = source.cloneHeaders();
        }
    }
    
    public static HttpHeaders createFromMultiMap(
        Map<String,List<String>> multiMap ){
        if( multiMap==null )throw new IllegalArgumentException( "multiMap==null" );
        return new HttpHeaders(multiMap);
    }

    public Map<String, List<String>> getMultiMap() {
        return this.headers;
    }

    //<editor-fold defaultstate="collapsed" desc="clone">
    @Override
    public HttpHeaders clone(){
        return new HttpHeaders(this);
    }
    
    public Map<String, List<String>> cloneHeaders() {
        Map<String,List<String>> clone = new LinkedHashMap<String, List<String>>();
        if( headers!=null ){
            for( Map.Entry<String,List<String>> e : this.headers.entrySet() ){
                String sk = e.getKey();
                List<String> sv = e.getValue();
                List<String> cv = new ArrayList<String>();
                cv.addAll(sv);
                clone.put(sk, cv);
            }
        }
        return clone;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="get / set">
    /**
     * Возвращает значение указанного заголовка
     * @param key имя заголовка, регистр не имеет значения
     * @return значение или null, если заголовок не указан
     */
    public String getFirst( String key ){
        if( headers==null )return null;
        if( key==null && headers.containsKey(null) ){
            List<String> values = headers.get(key);
            if( values!=null && values.size()>0 ){
                return values.get(0);
            }
            return null;
        }
        for( Map.Entry<String,List<String>> e : this.headers.entrySet() ){
            String sk = e.getKey();
            if( key!=null && sk!=null && key.equalsIgnoreCase(sk) ){
                List<String> values = e.getValue();
                if( values!=null && values.size()>0 ){
                    return values.get(0);
                }
            }
        }
        return null;
    }
    
    /**
     * Возвращает значение указанного заголовка
     * @param key имя заголовка, регистр не имеет значения
     * @return значение или null, если заголовок не указан
     */
    public String get(String key){
        return getFirst(key);
    }
    
    protected static Set<String> getKeysIgnoreCase( Map<String, List<String>> headers, String key ){
        if( key==null )throw new IllegalArgumentException( "key==null" );
        
        Set<String> keys = new LinkedHashSet<String>();
        for( String k : headers.keySet() ){
            if( key.equalsIgnoreCase(k) ){
                keys.add( k );
            }
        }
        
        return keys;
    }
    
//    /**
//     * Установка заголовка
//     * @param key Заголовок
//     * @param value Значение или null - для удаления
//     * @return Заголовки с новыми данными
//     */
//    public HttpHeaders set(String key,String value){
//        if( key==null )throw new IllegalArgumentException( "key==null" );
//        
//        Map<String, List<String>> h = this.cloneHeaders();
//        if( value==null ){
//            // Удаление заголовка
//            Set<String> keys = getKeysIgnoreCase(h, key);
//            for( String k : keys )h.remove(key);
//            return createFromMultiMap(h);
//        }else{
//            // Установка заголовка
//            Set<String> keys = getKeysIgnoreCase(h, key);
//            int idx = -1;
//            for( String k : keys ){
//                idx++;
//                if( idx==0 ){
//                    // Установка
//                    List<String> values = new ArrayList<String>();
//                    values.add(value);
//                    h.put(k, values);
//                }else{
//                    // Удаление лишних
//                    h.remove(k);
//                }
//            }
//            return createFromMultiMap(h);
//        }
//    }
    
    /**
     * Установка заголовка
     * @param key Заголовок
     * @param value Значение или null - для удаления
     * @return Заголовки с новыми данными (этот же объект)
     */
    public HttpHeaders set(String key,String value){
        if( key==null )throw new IllegalArgumentException( "key==null" );
        
        Map<String, List<String>> h = this.headers;
        if( value==null ){
            // Удаление заголовка
            Set<String> keys = getKeysIgnoreCase(h, key);
            for( String k : keys )h.remove(key);
//            return createFromMultiMap(h);
            return this;
        }else{
            // Установка заголовка
            Set<String> keys = getKeysIgnoreCase(h, key);
                if( !keys.isEmpty() ){
                int idx = -1;
                for( String k : keys ){
                    idx++;
                    if( idx==0 ){
                        // Установка
                        List<String> values = new ArrayList<String>();
                        values.add(value);
                        h.put(k, values);
                    }else{
                        // Удаление лишних
                        h.remove(k);
                    }
                }
            }else{
                List<String> lv = new ArrayList<String>();
                lv.add(value);
                h.put(key, lv);
            }
//            return createFromMultiMap(h);
            return this;
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="toString()">
    @Override
    public String toString(){
        StringBuilder buff = new StringBuilder();
        for( Map.Entry<String,List<String>> e : headers.entrySet() ){
            String header = e.getKey();
            List<String> values = e.getValue();
            if( values!=null && header!=null && !values.isEmpty() ){
                for( String value: values ){
                    buff
                        .append(header)
                        .append(": ")
                        .append(value!=null ? value.trim() : null)
                        .append("\n");
                }
            }
        }
        return buff.toString();
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Content-type + charset">
    public static final String CONTENT_TYPE = "Content-Type";
    
    /**
     * Заголовок Content-type
     * @return значение Content-type, пример:
     * Content-Type: text/html;charset=utf-8
     */
    public String getContentTypeHeader(){
        return getFirst(CONTENT_TYPE);
    }
    
    /**
     * Кодировка символов (Content-Type: text/html;charset=utf-8)
     * @return Кодировка или null
     */
    public Charset getContentTypeCharset(){
        String charset = null;
        String contentType = getContentTypeHeader();
        if( contentType!=null ){
            for (String param : contentType.replace(" ", "").split(";")) {
                if (param.toLowerCase().startsWith("charset=")) {
                    charset = param.split("=", 2)[1];
                    break;
                }
            }
            if( charset!=null ){
                return Charset.forName(charset);
            }
        }
        return null;
    }

    /**
     * mime тип (Content-Type: text/html)
     * @return mime тип или null
     */
    public String getContentType(){
        String contentTypeHeader = getContentTypeHeader();
        if( contentTypeHeader!=null ){
            if( contentTypeHeader.contains(";") ){
                int i = contentTypeHeader.indexOf(";");
                return contentTypeHeader.substring(0,i).trim();
            }
        }
        return null;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Content-Length">
    /**
     * Размер содержимого в байтах (Content-Length)
     * @return Размер содержимого в байтах или -1, если не указан
     */
    public long getContentLength(){
        String val = get("Content-Length");
        if( val!=null ){
            try{
                return Long.parseLong(val);
            }catch(NumberFormatException ex){
                return -1;
            }
        }
        return -1;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Content-Disposition">
    public String getContentDispositionHeader(){
        return get("Content-Disposition");
    }
    
    /**
     * парсинг экранированной строки. <br>
     * <font style="font-family:monospace">
     * quotedString ::= ws* quote qchar* quote . <br>
     * quote ::= '"' <br>
     * qhcar ::= escapeChar | nonEscapeChar <br>
     * escapeChar ::= '\' char <br>
     * nonEscapeChar ::= <i>char except quote</i> <br>
     * </font>
     * @param text текст
     * @param offset начало
     * @return декодированная строка
     */
    protected String parseQuotedString( String text, int offset ){
        int state = 0;
        StringBuilder sb = new StringBuilder();
        for( int i=offset; i<text.length(); i++ ){
            String ltext = Text.lookupText(text, i, 2);
            if( ltext.length()==0 ){
                break;
            }
            if( ltext.length()==1 ){
                switch( state ){
                    case 1:
                        sb.append(ltext);
                        break;
                }
            }
            if( ltext.length()==2 ){
                switch( state ){
                    case 0:
                        if( ltext.startsWith("\"") ){
                            state = 1;
                        }
                        break;
                    case 1:
                        if( ltext.startsWith("\"") ){
                            state = 2;
                        }else if( ltext.equals("\\r") ){
                            i++;
                            sb.append("\r");
                        }else if( ltext.equals("\\t") ){
                            i++;
                            sb.append("\t");
                        }else if( ltext.equals("\\n") ){
                            i++;
                            sb.append("\n");
                        }else if( ltext.equals("\\\"") ){
                            i++;
                            sb.append("\"");
                        }else if( ltext.equals("\\\\") ){
                            i++;
                            sb.append("\\");
                        }else if( ltext.startsWith("\\") ){
                            i++;
                            sb.append(ltext.charAt(1));
                        }else{
                            sb.append(ltext.charAt(0));
                        }
                        break;
                }
            }
        }
        return sb.toString();
    }
    
    /**
     * Имя файла вложения
     * @return имя файла вложения или null
     */
    public String getContentDispositionFileName(){
        String header = getContentDispositionHeader();
        if( header==null )return null;
        if( !header.contains(";") )return null;
        String[] kvline = header.split(";");
        if( kvline!=null && kvline.length>0 ){
            for( int i=1; i<kvline.length; i++ ){
                String[] kv = kvline[i].split("\\s*=\\s*",2);
                if( kv!=null && kv.length==2 ){
                    String k = kv[0].trim();
                    String v = kv[1].trim();
                    if( k.equalsIgnoreCase("filename") && v.length()>0 ){
                        if( v.startsWith("\"") ){
                            int i1 = v.indexOf("\"");
                            return parseQuotedString(v, i1);
//                            int i2 = v.lastIndexOf("\"");
//                            if( i1>=0 && i2>i1 ){
//                                int l = i2-i1;
//                                if( l>0 ){
//                                    return v.substring(i1,i2);
//                                }
//                            }
                        }else{
                            return v;
                        }
                    }
                    //TODO см. http://tools.ietf.org/html/rfc6266
//                    if( k.equalsIgnoreCase("filename*") && v.toLowerCase().startsWith("utf-8''") ){
//                    }
                }
            }
        }
        return null;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Accept-Ranges">
    /**
     * Принимает запросы на фрагментарное скачивание (Accept-Ranges: bytes)
     * @return как указываются диапазоны докачки или null
     */
    public String getAcceptRanges(){
        return get("Accept-Ranges");
    }
//</editor-fold>
    
    //    //<editor-fold defaultstate="collapsed" desc="isAllowContinueDownload">
//    /**
//     * Возвращает признак доступности докачки
//     * @return true - докачка возможна; false - не возможна
//     */
//    public boolean isAllowContinueDownload(){
//        long contentLen = getContentLength();
//        String acceptRanges = getAcceptRanges();
//        return acceptRanges!=null && acceptRanges.trim().equalsIgnoreCase("bytes") && contentLen >= 0;
//    }
////</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Range">
    /**
     * Диапазон запрашиваемых байтов фрагмента
     */
    public static class Range {
        /**
         * С какого байта (от нуля)
         */
        public long from = -1;
        
        /**
         * По какой байт включительно, или -1
         */
        public long to = -1;
    }
    
    public String getRangeHeader(){
        return get("Range");
    }
    
    /**
     * Указывает размер запрашиваемого фрагмента
     * @return Описание фрагмента или null
     */
    public Range getRange(){
        String val = get( "Range" );
        if( val==null )return null;
        
        val = val.trim();
        if( !val.toLowerCase().startsWith("bytes=") )return null;
        
        val = val.substring("bytes=".length());
        String[] ft = val.split("\\-",2);
        if( ft.length!=2 )return null;
        
        if( ft[0].trim().length()>0 &&
            ft[1].trim().length()>0
            ){
            Range r = new Range();
            r.from = Long.parseLong(ft[0]);
            r.to = Long.parseLong(ft[1]);
            return r;
        }
        
        if( ft[0].trim().length()>0 ){
            Range r = new Range();
            r.from = Long.parseLong(ft[0]);
            r.to = -1;
            return r;
        }
        
        return null;
    }
    
    /**
     * Указывает размер запрашиваемого фрагмента
     * @param from с какого байта начать (от нуля)
     * @param to по какой байт включительно закончить (от нуля), или -1 - то до конца
     */
    public void setRange(long from,long to){
        if( from<0 )throw new IllegalArgumentException("from<0");
        if( to>=0 && to<from )throw new IllegalArgumentException("to>=0 && to<from");
        if( to>=0 ){
            set( "Range", "bytes="+Long.toString(from)+"-"+Long.toString(to) );
        }else{
            set( "Range", "bytes="+Long.toString(from)+"-" );
        }
    }

    /**
     * указывает заголовок Range
     * @param val Значение заголовка
     */
    public void setRangeHeader(String val){
        set( "Range", null );
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Content-Range">
    /**
     * Content-Range: bytes 88080384-160993791/160993792
     * @return Значение заголовка
     */
    public String getContentRangeHeader(){
        return get("Content-Range");
    }
    
    /**
     * Размер запрашивой порции
     */
    public static class ContentRange {
        /**
         * С какого байта (от 0) начинается порция
         */
        public long from = -1;
        
        /**
         * По какой байт включительно (от 0) заканчивается порция
         */
        public long to = -1;
        
        /**
         * Общее кол-во байтов
         */
        public long total = -1;
        
        @Override
        public String toString(){
            return Long.toString(from)+"-"+Long.toString(to)+"/"+Long.toString(total);
        }
    }
    
    private static Pattern contentRangePattern = Pattern.compile(
        "(?is)^"
            + "\\s*bytes\\s+"
            + "(\\d+)\\s*-\\s*(\\d+)"
            + "\\s*/\\s*(\\d+)"
            + ".*?$");
    
    /**
     * Размер части данных
     * @return размер запрашивой порции или null
     */
    public ContentRange getContentRange(){
        String val = getContentRangeHeader();
        if( val!=null ){
            Matcher m = contentRangePattern.matcher(val);
            if( m.matches() ){
                String fromStr = m.group(1);
                String toStr = m.group(2);
                String totalStr = m.group(3);
                
                ContentRange cr = new ContentRange();
                cr.from = Long.parseLong(fromStr);
                cr.to = Long.parseLong(toStr);
                cr.total = Long.parseLong(totalStr);
                
                return cr;
            }
        }
        return null;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Accept">
    public String getAccept(){ return get("Accept"); }
    
    // text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8
    public void setAccept(String accept){ set( "Accept", accept ); }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Accept-Language">
    public String getAcceptLanguage(){ return get("Accept-Language"); }
    /**
     * ru-RU,ru;q=0.8,en-US;q=0.6,en;q=0.4
     * @param accept Значение заголовка
     */
    public void setAcceptLanguage(String accept){ set( "Accept-Language", accept ); }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="User-Agent">
    public String getUserAgent(){ return get("User-Agent"); }
    /**
     * HttpClient.class.getName()+"/"+"0.1-SNAPSHOT"
     * @param useragent Значение заголовка
     */
    public void setUserAgent(String useragent){ set( "User-Agent", useragent ); }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="Accept-Charset">
    public Charset getAcceptCharset(){ 
        String val = get("Accept-Charset"); 
        return val!=null ? Charset.forName(val) : null;
    }
    
    /**
     * Accept-Charset: UTF-8
     * @param accept Значение заголовка
     */
    public void setAcceptCharset(Charset accept){
        set( "Accept-Charset", accept!=null ? accept.name() : null );
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="Location">
    /**
     * Указывает redirect
     * @return куда переход или null
     */
    public String getLocation(){ 
        String val = get("Location"); 
        return val;
    }
    
    /**
     * Указывает redirect
     * @param location куда переход
     */
    public void setLocation(String location){
        set( "Location", location );
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="referer">
    /**
     * Указывает referer, откуда пришел
     * @return откуда пришел или null
     */
    public String getReferer(){ 
        String val = get("Referer"); 
        return val;
    }
    
    /**
     * Указывает referer, откуда пришел
     * @param referer откуда пришел
     */
    public void setReferer(String referer){
        set( "Referer", referer );
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Set-Cookie">
    public String getSetCookie(){
        return get("Set-Cookie");
    }
    
    public void setSetCookie(String setCookie){
        set("Set-Cookie",setCookie);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="Cookie">
    public String getCookie(){
        return get("Cookie");
    }
    
    public void setCookie(String setCookie){
        set("Cookie",setCookie);
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Last-Modified">
    /**
     * Возвращает время последнего изменения объекта (заголовок Last-Modified)
     * в локальном времени.
     *
     * Если время не указано, вернет null.
     *
     * @return Локальное время изменения объекта (или null)
     */
    public Date getLastModifiedLocal(){
        String lastModifiedHeader = get("Last-Modified");
        if( lastModifiedHeader==null )return null;
        return new DateTime().parse(lastModifiedHeader,false);
    }
    
    /**
     * Устанавливает время последнего изменения объекта (заголовок Last-Modified)
     * @param date лоакльное время изменения или null что бы удалить заголовок Last-Modified
     */
    public void setLastModifiedLocal( Date date ){
        if( date==null ){
            set( "Last-Modified", null );
        }else{
            set( "Last-Modified", new DateTime().toDate(date,false) );
        }
    }
    
    /**
     * Возвращает время по гринвичу последнего изменения объекта (заголовок Last-Modified)
     *
     * Если время не указано, вернет null.
     *
     * @return Время, по гринвичу, изменения объекта (или null)
     */
    public Date getLastModifiedGMT(){
        String lastModifiedHeader = get("Last-Modified");
        if( lastModifiedHeader==null )return null;
        
        return new DateTime().parse(lastModifiedHeader,true);
    }
    
    /**
     * Устанавливает время последнего изменения объекта (заголовок Last-Modified)
     * @param date время, по гринвичу, изменения или null что бы удалить заголовок Last-Modified
     */
    public void setLastModifiedGMT( Date date ){
        if( date==null ){
            set( "Last-Modified", null );
        }else{
            set( "Last-Modified", new DateTime().toDate(date,true) );
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Last-Modified">
    /**
     * Возвращает время для проверки: изменился ли объект с указаного времени (заголовок If-Modified-Since)
     * в локальном времени.
     *
     * Если время не указано, вернет null.
     *
     * @return Локальное время изменения объекта (или null)
     */
    public Date getIfModifiedSinceLocal(){
        String lastModifiedHeader = get("If-Modified-Since");
        if( lastModifiedHeader==null )return null;
        return new DateTime().parse(lastModifiedHeader,false);
    }
    
    /**
     * Устанавливает время для проверки: изменился ли объект с указаного времени (заголовок If-Modified-Since). <br>
     * <br>
     * При запросе GET, если объект не изменился и сервер поддерживает заголовок, то: <br>
     * вернет ответ 304 - объет не изменился, вместо 200 <br>
     * вернет ответ другое значение - если дата указана не верно, или что либо еще ... <br>
     * <br>
     * Если объект изменился то вернет 200 или другое значение, но не 304
     * @param date лоакльное время изменения или null что бы удалить заголовок If-Modified-Since
     */
    public void setIfModifiedSinceLocal( Date date ){
        if( date==null ){
            set( "If-Modified-Since", null );
        }else{
            set( "If-Modified-Since", new DateTime().toDate(date,false) );
        }
    }
    
    /**
     * Возвращает время для проверки: изменился ли объект с указаного времени (заголовок If-Modified-Since)
     *
     * Если время не указано, вернет null.
     *
     * @return Время, по гринвичу, изменения объекта (или null)
     */
    public Date getIfModifiedSinceGMT(){
        String lastModifiedHeader = get("If-Modified-Since");
        if( lastModifiedHeader==null )return null;
        
        return new DateTime().parse(lastModifiedHeader,true);
    }
    
    /**
     * Устанавливает время для проверки: изменился ли объект с указаного времени (заголовок If-Modified-Since).<br>
     * <br>
     * При запросе GET, если объект не изменился и сервер поддерживает заголовок, то: <br>
     * вернет ответ 304 - объет не изменился, вместо 200 <br>
     * вернет ответ другое значение - если дата указана не верно, или что либо еще ... <br>
     * <br>
     * Если объект изменился то вернет 200 или другое значение, но не 304
     * @param date время, по гринвичу, изменения или null что бы удалить заголовок If-Modified-Since
     */
    public void setIfModifiedSinceGMT( Date date ){
        if( date==null ){
            set( "If-Modified-Since", null );
        }else{
            set( "If-Modified-Since", new DateTime().toDate(date,true) );
        }
    }
    //</editor-fold>
}
