/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;


import java.io.Closeable;
import java.io.IOException;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Помошник в реализации HttpListener
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class HttpListenersHelper {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(HttpListenersHelper.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(HttpListenersHelper.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(HttpListenersHelper.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(HttpListenersHelper.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(HttpListenersHelper.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(HttpListenersHelper.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(HttpListenersHelper.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    private Object sync = null;
    
    public HttpListenersHelper(){
        this( true );
    }
    
    public HttpListenersHelper(boolean sync){
        if( sync ){
            this.sync = this;
        }
    }
    
    public HttpListenersHelper(boolean sync, Object synObj){
        if( sync ){
            this.sync = synObj;
        }
    }
    
    protected HashSet<HttpListener> listeners = new HashSet<HttpListener>();
    protected WeakHashMap<HttpListener,Object> weakLinks = new WeakHashMap<HttpListener,Object>();
    
    public Set<HttpListener> getListeners() { 
        HashSet<HttpListener> res = new LinkedHashSet<HttpListener>();
        if( sync!=null ){
            synchronized(sync){
                res.addAll(listeners);
                res.addAll(weakLinks.keySet());
            }
        }else{
            res.addAll(listeners);
            res.addAll(weakLinks.keySet());
        }
        return res; 
    }
    
    public Closeable addListener(HttpListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException("listener==null");
        }

        if( sync!=null ){
            synchronized (sync) {
                return addListener0(listener);
            }
        }
        return addListener0(listener);
    }

    private Closeable addListener0(HttpListener listener) {
        listeners.add(listener);
        final HttpListener lstr = listener;
        return new Closeable() {
            private HttpListener l_str = lstr;
            
            @Override
            public void close() throws IOException {
                if (l_str != null) {
                    HttpListenersHelper.this.removeListener(l_str);
                    l_str = null;
                }
            }
        };
    }
    
    public Closeable addListener(HttpListener listener, boolean weakLink) {
        if( listener==null )throw new IllegalArgumentException( "listener==null" );
        if( sync!=null ){
            synchronized(sync){
                if( weakLink ){
                    return addListenerWeak0(listener);
                }else{
                    return addListener0(listener);
                }
            }
        }else{
            if( weakLink ){
                return addListenerWeak0(listener);
            }else{
                return addListener0(listener);
            }
        }
    }
    
    private Closeable addListenerWeak0(HttpListener listener) {
        weakLinks.put(listener, true);
        final HttpListener lstr = listener;
        return new Closeable() {
            private HttpListener l_str = lstr;
            @Override
            public void close() throws IOException {
                if (l_str != null) {
                    HttpListenersHelper.this.removeListener(l_str);
                    l_str = null;
                }
            }
        };
    }
    
    public void removeListener(HttpListener listener) {
        if( sync!=null ){
            synchronized (sync) {
                if (listener != null) {
                    listeners.remove(listener);
                    weakLinks.remove(listener);
                }
            }
        }else{
            if (listener != null) {
                listeners.remove(listener);
                weakLinks.remove(listener);
            }
        }
    }
    
    public void fireEvent(HttpEvent event) {
        HttpListener[] lstnrs = null;
        if( sync!=null ){
            synchronized (this) {
                lstnrs = getListeners().toArray(new HttpListener[]{});
            }
        }else{
            lstnrs = getListeners().toArray(new HttpListener[]{});
        }
        if (event != null) {
            for (HttpListener l : lstnrs) {
                if( l!=null )l.httpEvent(event);
            }
        }
    }
}
