/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;


import java.io.Closeable;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.ProtocolException;
import java.net.Proxy;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.cbuffer.ContentBuffer;
import xyz.cofe.collection.Func1;

/**
 * Http Запрос
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class HttpRequest 
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(HttpRequest.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(HttpRequest.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(HttpRequest.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(HttpRequest.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(HttpRequest.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(HttpRequest.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(HttpRequest.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected static final AtomicLong sequenceID = new AtomicLong();

    protected final Lock lock;

    /**
     * Идентификатор объекта в пределах приложения
     */
    public final long id = sequenceID.incrementAndGet();

    public HttpRequest(URL url){
        if( url==null )throw new IllegalArgumentException( "url==null" );
        this.url = url;
        this.lock = new ReentrantLock();
    }

    public HttpRequest( URL url, HttpClient httpClient ){
        if( url==null )throw new IllegalArgumentException( "url==null" );
        if( httpClient==null )throw new IllegalArgumentException( "httpClient==null" );
        this.httpClient = httpClient;
        this.url = url;
        this.lock = new ReentrantLock();
    }

    public HttpRequest( HttpRequest source ){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        try{
            source.lock.lock();
            this.httpClient = source.getHttpClient();
            this.httpHeaders = source.getHttpHeaders().clone();
            this.method = source.getMethod();
            this.url = source.getUrl();
            this.async = source.isAsync();
            this.connectTimeout = source.getConnectTimeout();
            this.readTimeout = source.getReadTimeout();
            this.lock = new ReentrantLock();
            this.data = source.getData();
        }finally{
            source.lock.unlock();
        }
    }

    @Override
    public HttpRequest clone(){
        return new HttpRequest(this);
    }

    /**
     * Получение идентификатора
     * @return идентификатор запроса
     */
    public final long getId() {
        return id;
    }
    
    //<editor-fold defaultstate="collapsed" desc="url">
    private URL url = null;

    /**
     * Указывает запрашиваемый адрес
     * @return запрашиваемый адрес
     */
    public URL getUrl() {
        try{
            lock.lock();
            return url;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает запрашиваемый адрес
     * @param url запрашиваемый адрес
     */
    public void setUrl(URL url) {
        if( url==null )throw new IllegalArgumentException( "url==null" );
        try{
            lock.lock();
            this.url = url;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="method">
    private String method = "GET";

    /**
     * Возвращает HTTP метод, по умолчанию GET
     * @return HTTP метод
     */
    public String getMethod() {
        try{
            lock.lock();
            if( method==null )method = "GET";
            return method;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает HTTP метод
     * @param method HTTP метод
     */
    public void setMethod(String method) {
        if( method==null )throw new IllegalArgumentException( "method==null" );
        try{
            lock.lock();
            this.method = method;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="httpClient">
    private HttpClient httpClient = null;

    public HttpClient getHttpClient() {
        try{
            lock.lock();
            if( httpClient==null )httpClient = new HttpClient();
            return httpClient;
        }finally{
            lock.unlock();
        }
    }

    public void setHttpClient(HttpClient httpClient) {
        try{
            lock.lock();
            this.httpClient = httpClient;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="httpHeaders">
    private HttpHeaders httpHeaders = null;

    public HttpHeaders getHttpHeaders() {
        try{
            lock.lock();
            if( httpHeaders==null )httpHeaders = new HttpHeaders();
            return httpHeaders;
        }finally{
            lock.unlock();
        }
    }

    public void setHttpHeaders(HttpHeaders headers) {
        try{
            lock.lock();
            this.httpHeaders = headers;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="applyHeadersTo()">
    public void applyHeadersTo( URLConnection connection ) {
        try{
            lock.lock();
            if( connection==null )throw new IllegalArgumentException( "connection==null" );
            for( Map.Entry<String,List<String>> e : getHttpHeaders().getMultiMap().entrySet() ){
                String k = e.getKey();
                List<String> lv = e.getValue();
                if( k!=null && lv!=null && !lv.isEmpty() ){
                    String v = lv.get(0);
                    connection.setRequestProperty(k, v);
                }
            }
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="user agent">
    public String getUserAgent(){
        try{
            lock.lock();
            HttpHeaders hh = getHttpHeaders();
            String ua = hh.getUserAgent();

            if( ua==null ){
                if( httpClient!=null ){
                    String ua1 = httpClient.getUserAgent();
                    if( ua1!=null ){
                        hh.setUserAgent(ua1);
                        return ua1;
                    }else{
                        return null;
                    }
                }else{
                    return null;
                }
            }else{
                return ua;
            }
        }finally{
            lock.unlock();
        }
    }

    public void setUserAgent( String userAgent ){
        try{
            lock.lock();
            HttpHeaders hh = getHttpHeaders();
            hh.setUserAgent(userAgent);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="connectTimeout">
    protected Integer connectTimeout = null;

    public int getConnectTimeout(){
        try{
            lock.lock();
            if( connectTimeout==null ){
                if( httpClient!=null ){
                    connectTimeout = httpClient.getConnectTimeout();
                }else{
                    connectTimeout = 30000;
                }
                return connectTimeout;
            }else{
                return connectTimeout;
            }
        }finally{
            lock.unlock();
        }
    }

    public void setConnectTimeout( int connectTimeoutMS ){
        if( connectTimeoutMS<0 )throw new IllegalArgumentException("connectTimeoutMS<0");
        try{
            lock.lock();
            connectTimeout = connectTimeoutMS;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="readTimeout">
    protected Integer readTimeout = null;

    public int getReadTimeout(){
        try{
            lock.lock();
            if( readTimeout==null ){
                if( httpClient!=null ){
                    readTimeout = httpClient.getReadTimeout();
                }else{
                    readTimeout = 30000;
                }
                return readTimeout;
            }else{
                return readTimeout;
            }
        }finally{
            lock.unlock();
        }
    }

    public void setReadTimeout( int readTimeoutMS ){
        if( readTimeoutMS<0 )throw new IllegalArgumentException("readTimeoutMS<0");
        try{
            lock.lock();
            readTimeout = readTimeoutMS;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="listeners">
    protected final HttpListenersHelper httpListenerHelper = new HttpListenersHelper();

    public Set<HttpListener> getListeners() {
        return httpListenerHelper.getListeners();
    }

    public Closeable addListener(HttpListener listener) {
        return httpListenerHelper.addListener(listener);
    }

    public Closeable addListener(HttpListener listener, boolean weakLink) {
        return httpListenerHelper.addListener(listener, weakLink);
    }

    public void removeListener(HttpListener listener) {
        httpListenerHelper.removeListener(listener);
    }

    protected void fireEvent(HttpEvent event) {
        httpListenerHelper.fireEvent(event);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="async">
    protected Boolean async = null;
    /**
     * Указывает асинхронный режим
     * @return true - асинхронный; false (по умолч) - синхронный
     */
    public boolean isAsync(){
        try{
            lock.lock();
            if( async==null )async = false;
            return async;
        }finally{
            lock.unlock();
        }
    }
    /**
     * Указывает асинхронный режим
     * @param async true - асинхронный; false - синхронный
     */
    public void setAsync(boolean async){
        try{
            lock.lock();
            this.async = async;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentBuffer">
    protected ContentBuffer contentBuffer = null;

    /**
     * Указывает буффер для content-а
     * @return contentBuffer буффер
     */
    public ContentBuffer getContentBuffer() {
        try{
            lock.lock();
            return contentBuffer;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает буффер для content-а
     * @param contentBuffer буффер
     */
    public void setContentBuffer(ContentBuffer contentBuffer) {
        try{
            lock.lock();
            this.contentBuffer = contentBuffer;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentOffset">
    protected long contentOffsetStart = -1;

    /**
     * Указывает с какой позиции записывать в content buffer
     * @return позиция
     */
    public long getContentOffsetStart() {
        try{
            lock.lock();
            return contentOffsetStart;
        }finally{
            lock.unlock();
        }
    }
    /**
     * Указывает с какой позиции записывать в content buffer
     * @param contentOffsetStart позиция
     */
    public void setContentOffsetStart(long contentOffsetStart) {
        try{
            lock.lock();
            this.contentOffsetStart = contentOffsetStart;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="maxDownloadSize">
    protected long maxDownloadSize = -1;

    /**
     * Возвращает максимум загужаемых данных, по умолчанию без ограничения
     * @return макс загружаемых данных, maxDownloadSize &lt; 0  - без ограничений
     */
    public long getMaxDownloadSize(){
        try{
            lock.lock();
            return maxDownloadSize;
        }
        finally{
            lock.unlock();
        }
    }

    /**
     * Устанавливает максимум загужаемых данных.
     * @param maxSize макс загружаемых данных, maxSize &lt; 0  - без ограничений
     */
    public void setMaxDownloadSize(long maxSize){
        try{
            lock.lock();
            maxDownloadSize = maxSize;
        }
        finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="followRedirect">
    protected boolean followRedirect = true;

    /**
     * Переходить по redirect
     * @return true - переход по redirect
     */
    public boolean isFollowRedirect() {
        try{
            lock.lock();
            return followRedirect;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Переходить по redirect
     * @param followRedirect true - переход по redirect
     */
    public void setFollowRedirect(boolean followRedirect) {
        try{
            lock.lock();
            this.followRedirect = followRedirect;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="data">
    protected Func1<Object,URLConnection> data;

    public Func1<Object, URLConnection> getData() {
        try{
            lock.lock();
            return data;
        }finally{
            lock.unlock();
        }
    }

    public void setData( Func1<Object, URLConnection> data ) {
        try{
            lock.lock();
            this.data = data;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    /**
     * Указание user-agent
     * @param connection Коннект
     */
    protected void prepareUserAgent(URLConnection connection){
        HttpHeaders hh = getHttpHeaders();
        String ua = hh.getUserAgent();

        if( ua==null ){
            if( httpClient!=null ){
                String ua1 = httpClient.getUserAgent();
                if( ua1!=null ){
                    hh.setUserAgent(ua1);
                }
            }
        }
    }

    /**
     * Подготовка обязательных заголовков
     * @param connection Открытое соединение
     * @see #prepareUserAgent(java.net.URLConnection) 
     */
    protected void prepareRequiredHeaders(URLConnection connection){
        logFine("prepareRequiredHeaders");
        if( connection!=null ){
            if( httpHeaders==null ){
                // создать headers из connection request header
                httpHeaders = new HttpHeaders();
            }

            // копирование существующей cookie
            String cookie = connection.getRequestProperty("Cookie");
            if( cookie!=null ){
                String existsCookie = httpHeaders.getCookie();
                if( existsCookie==null )httpHeaders.setCookie(cookie);
            }

        }
        prepareUserAgent(connection);
    }

    /**
     * Открытие соединение, с учетом настроек: <br>
     * proxy, connectTimeout, readTimeout, httpHeaders, method
     * @return соединение
     * @throws IOException не может открыть соединение
     */
    public URLConnection openURLConnection()
        throws IOException
    {
        HttpClient client = null;
        List<Proxy> lproxy = null;
        URLConnection connection = null;

        try{
            lock.lock();
            client = getHttpClient();
            try {
                lproxy = client.getProxySelector().select(url.toURI());
            } catch (URISyntaxException ex) {
                Logger.getLogger(HttpRequest.class.getName()).log(Level.SEVERE, null, ex);
            }


            if( lproxy==null ||  lproxy.isEmpty() ){
                logFine( "open url connection: {0}",url );
                connection = url.openConnection();
            }else{
                IOException err = null;
                for( Proxy p : lproxy ){
                    if( connection!=null )break;
                    try {
                        err = null;
                        logFine( "open url connection: {0} through proxy {1}",url, p );
                        connection = url.openConnection(p);
                    }catch (IOException ex){
                        err = ex;
                        connection = null;
                        logException(err);
                    }
                }
                if( err!=null ){
                    throw err;
                }
            }
            if( connection==null ){
                throw new IOException("can't open connection to "+url);
            }

            prepareRequiredHeaders(connection);
            applyHeadersTo(connection);

            if( connection instanceof HttpURLConnection ){
                HttpURLConnection httpConnection = (HttpURLConnection)connection;
                if( method!=null ){
                    try {
                        httpConnection.setRequestMethod(method);
                    } catch (ProtocolException ex) {
                        Logger.getLogger(HttpRequest.class.getName()).log(Level.SEVERE, null, ex);
                    }
                }

                httpConnection.setInstanceFollowRedirects(false);

                httpConnection.setConnectTimeout(getConnectTimeout());
                httpConnection.setReadTimeout(getReadTimeout());
            }

            return connection;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Создание объекта HttpResponse
     * @return объекта HttpResponse
     * @see HttpResponse
     */
    public HttpResponse createResponse(){
        try {
            URLConnection conn = openURLConnection();
            return new HttpResponse(this, conn);
        } catch( IOException ex ) {
            Logger.getLogger(HttpRequest.class.getName()).log(Level.SEVERE, null, ex);
            return new HttpResponse(this,ex);
        }
    }
}
