/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.cbuffer.ContentBuffer;
import xyz.cofe.cbuffer.ContentBufferInputStream;
import xyz.cofe.cbuffer.MemContentBuffer;
import xyz.cofe.collection.BasicPair;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Func4;
import xyz.cofe.collection.Pair;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.text.Text;

/**
 * Http ответ
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class HttpResponse
implements IsFinished
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(HttpResponse.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(HttpResponse.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(HttpResponse.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(HttpResponse.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(HttpResponse.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected static final HttpStatusHelper httpStatusHelper = new HttpStatusHelper();

    protected static final AtomicLong sequenceID = new AtomicLong();

    /**
     * Идентификатор объекта в пределах приложения
     */
    public final long id = sequenceID.incrementAndGet();

    /**
     * Блокировка объекта
     */
    protected final Lock lock;

//    /**
//     * Конструктор
//     * @param request запрос
//     */
//    public HttpResponse(HttpRequest request){
//        if( request==null )throw new IllegalArgumentException( "request==null" );
//        this.request = request;
//        this.lock = new ReentrantLock();
//    }

    /**
     * Конструктор копирования. <br>
     * Копируются свойства: <br>
     * request - ссылка <br>
     * state - клон <br>
     * errors - клон <br>
     * async - клон <br>
     * downloadBufferSize - клон <br>
     * contentBuffer - клон <br>
     * contentCharset - клон <br>
     * contentWriterPos - клон <br>
     * downloadedSize - клон <br>
     * httpHeaders - клон <br>
     * started - клон <br>
     * finished - клон <br>
     * statusCode - клон <br>
     * statusMessage - клон <br>
     * text - клон <br>
     * @param source исходный объект
     * @param cloneContentBuffer клонировать содержимое
     * @param cloneHeaders клонировать заголовки
     * @param req Переопределить запрос
     */
    protected HttpResponse( HttpResponse source, HttpRequest req, boolean cloneHeaders, boolean cloneContentBuffer){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        this.lock = new ReentrantLock();
        try{
            source.lock.lock();

            this.request = req!=null ? req : source.request;
            this.state = source.state;

            this.errors.clear();
            this.errors.addAll(source.errors);

            this.async = source.async;
            this.downloadBufferSize = source.downloadBufferSize;

            this.contentBuffer = source.contentBuffer!=null
                ? ( cloneContentBuffer
                        ? source.contentBuffer.clone()
                        : source.contentBuffer )
                : this.contentBuffer;

            this.contentCharset = source.contentCharset;
            this.contentWriterPos = source.contentWriterPos;

            this.downloadedSize = source.downloadedSize;
            this.httpHeaders =
                    source.httpHeaders!=null
                        ? (cloneHeaders ? source.httpHeaders.clone() : source.httpHeaders)
                        : this.httpHeaders;

            this.started = source.started;
            this.finished = source.finished;
            this.statusCode = source.statusCode;
            this.statusMessage = source.statusMessage;
            this.followRedirect = source.followRedirect;
            this.currentRequest = source.currentRequest;

            this.text = source.text;

            this.data = source.getData();
//            this.method = source.method;
        }finally{
            source.lock.unlock();
        }
    }

    /**
     * Конструктор копирования.
     * Вызывает более длинный конструктор
     * @param source исходный объект
     * @param cloneContentBuffer клонировать содержимое
     * @param cloneHeaders клонировать заголовки
     * @see #HttpResponse(xyz.cofe.http.HttpResponse, xyz.cofe.http.HttpRequest, boolean, boolean)
     */
    public HttpResponse( HttpResponse source, boolean cloneHeaders, boolean cloneContentBuffer){
        this(source, null, true, true);
    }

    /**
     * Посылать вновь подцепившимся сообщение о изменении состояния на Finished
     */
    private boolean sendFinishedOnAddListener = false;

    /**
     * Конструктор с ошибкой открытия соединения
     * @param request запрос
     * @param error описание ошибки
     */
    public HttpResponse( HttpRequest request, Throwable error ){
        if( request==null )throw new IllegalArgumentException( "request==null" );
        if( error==null )throw new IllegalArgumentException( "error==null" );
        this.lock = new ReentrantLock();
        this.request = request;
        this.errors.add(error);
        setState(State.Finished);
        sendFinishedOnAddListener = true;
        this.data = request.getData();
    }

    public HttpResponse( HttpRequest request, URLConnection connection ){
        if( request==null )throw new IllegalArgumentException( "request==null" );
        this.lock = new ReentrantLock();
        this.request = request;
        this.connection = connection;

        try{
            request.lock.lock();

            this.contentBuffer = request.contentBuffer;
            this.followRedirect = request.followRedirect;
            this.contentWriterPos = request.contentOffsetStart;
            if( this.contentWriterPos!=null ){
                if( this.contentWriterPos<0 ){
                    this.contentWriterPos = (long)0;
                }
            }
            this.async = request.async;
            this.maxDownloadSize = request.maxDownloadSize;
            this.data = request.getData();
        }finally{
            request.lock.unlock();
        }
    }

    /**
     * Получение блокировки
     * @return блокировка
     */
    public Lock getLock(){
        return lock;
    }

    /**
     * Получение идентификатора
     * @return идентификатор запроса
     */
    public final long getId() {
        return id;
    }
    
    /**
     * Создание клона
     * @return клон
     * @see #HttpResponse(xyz.cofe.http.HttpResponse, boolean, boolean)
     */
    @Override
    @SuppressWarnings( "CloneDeclaresCloneNotSupported" )
    public HttpResponse clone(){
        return new HttpResponse(this, true, true);
    }

    /**
     * Создание клона
     * @param cloneHeaders клонировать заголовки
     * @param cloneBuffer клонировать буфер
     * @return клон
     * @see #HttpResponse(xyz.cofe.http.HttpResponse, boolean, boolean)
     */
    public HttpResponse clone(boolean cloneHeaders, boolean cloneBuffer){
        return new HttpResponse(this, cloneHeaders, cloneBuffer);
    }

    /**
     * Создание клона
     * @param req Запрос для клона
     * @param cloneHeaders клонировать заголовки
     * @param cloneBuffer клонировать буфер
     * @return клон
     */
    protected HttpResponse clone(HttpRequest req, boolean cloneHeaders, boolean cloneBuffer){
        return new HttpResponse(this, req, cloneHeaders, cloneBuffer);
    }

    protected URLConnection connection = null;

    /**
     * Запуск скачивания.
     * можно вызывать когда объект находится в состоянии Prepare
     */
    public void start(){
        State s = getState();
        if( !State.Prepare.equals( s ) ){
            throw new IllegalStateException("state is not prepare");
        }

        Runnable r = new Runnable() {
            @Override
            public void run() {
                if( connection==null ){
                    try {
                        connection = request.openURLConnection();
                        start(request,request.getUrl(), connection);
                    } catch( IOException ex ) {
                        Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
                        addError(ex);
                        setState(State.Finished);
                    }
                }else{
                    start(request, request.getUrl(), connection);
                }
            }
        };

        if( isAsync() ){
            if( thread!=null && thread.isAlive() ){
                stop();
            }

            thread = request.getHttpClient().createResponseThread().apply(this, r);
//            thread.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
//                @Override
//                public void uncaughtException(Thread t, Throwable e) {
//                    logException(e);
//                    addError(e);
//                    setState(State.Finished);
//
//                    long tbegin = System.currentTimeMillis();
//                    long tmax = 1000 * 15; // 15 seconds for stop
//                    long sleep = 50; // 50 milli second sleep
//                    while( t.isAlive() ){
//                        t.interrupt();
//                        long tnow = System.currentTimeMillis();
//                        long diff = tnow - tbegin;
//                        if( diff > tmax ){
//                            t.stop();
//                        }else{
//                            try {
//                                Thread.sleep(sleep);
//                            } catch( InterruptedException ex ) {
//                                t.stop();
//                                break;
//                            }
//                        }
//                    }
//                }
//            });

            String thName = getThreadName();
            if( thName!=null )thread.setName(thName);

            setState(State.Started);
            thread.start();
        }else{
            setState(State.Started);
            r.run();
        }
    }

    //<editor-fold defaultstate="collapsed" desc="method">
    // already method set in request
//    private String method;
//
//    /**
//     * Возвращает HTTP метод, по умолчанию GET
//     * @return HTTP метод
//     */
//    public String getMethod() {
//        try{
//            lock.lock();
//            if( method==null ){
//                HttpRequest req = getCurrentRequest();
//                req = req==null ? request : req;
//                if( req!=null ){
//                    method = req.getMethod();
//                }else{
//                    method = "GET";
//                }
//            }
//            return method;
//        }finally{
//            lock.unlock();
//        }
//    }
//
//    /**
//     * Указывает HTTP метод
//     * @param method HTTP метод
//     */
//    public void setMethod(String method) {
//        if( method==null )throw new IllegalArgumentException( "method==null" );
//        try{
//            lock.lock();
//            this.method = method;
//        }finally{
//            lock.unlock();
//        }
//    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="data">
    protected Func1<Object,URLConnection> data;

    public Func1<Object, URLConnection> getData() {
        try{
            lock.lock();
            return data;
        }finally{
            lock.unlock();
        }
    }

    public void setData( Func1<Object, URLConnection> data ) {
        try{
            lock.lock();
            this.data = data;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="ProgressEvent">
    public static class ProgressEvent extends HttpEvent {
        protected HttpResponse httpResponse = null;

        public ProgressEvent( HttpResponse httpResponse ){
            this.httpResponse = httpResponse;
        }

        public HttpResponse getHttpResponse() {
            return httpResponse;
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="request">
    private final HttpRequest request;

    public HttpRequest getRequest() {
        return request;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="errors">
    private final List<Throwable> errors = new ArrayList<Throwable>();
    public List<Throwable> getErrors(){
        try{
            lock.lock();
            ArrayList<Throwable> l = new ArrayList<Throwable>();
            l.addAll(errors);
            return l;
        }finally{
            lock.unlock();
        }
    }
    protected void addError( Throwable err ){
        try{
            lock.lock();
            errors.add( err );
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="listeners">
    protected final HttpListenersHelper httpListenerHelper = new HttpListenersHelper();

    public Set<HttpListener> getListeners() {
        return httpListenerHelper.getListeners();
    }

    public Closeable addListener(HttpListener listener) {
        Closeable c = httpListenerHelper.addListener(listener);
        if( sendFinishedOnAddListener && listener!=null ){
            listener.httpEvent(new StateChangedEvent(this, State.Prepare, State.Finished));
        }
        return c;
    }

    public Closeable addListener(HttpListener listener, boolean weakLink) {
        Closeable c = httpListenerHelper.addListener(listener, weakLink);
        if( sendFinishedOnAddListener && listener!=null ){
            listener.httpEvent(new StateChangedEvent(this, State.Prepare, State.Finished));
        }
        return c;
    }

    public void removeListener(HttpListener listener) {
        httpListenerHelper.removeListener(listener);
    }

    protected void fireEvent(HttpEvent event) {
        httpListenerHelper.fireEvent(event);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="downloadedSize">
    protected long downloadedSize = 0;

    /**
     * Возвращает кол-во загруженых байтов
     * @return кол-во загруженых байтов
     */
    public long getDownloadedSize() {
        try{
            lock.lock();
            return downloadedSize;
        }
        finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="maxDownloadSize">
    protected Long maxDownloadSize = null;

    /**
     * Возвращает максимум загужаемых данных, по умолчанию без ограничения
     * @return макс загружаемых данных, maxDownloadSize &lt; 0  - без ограничений
     */
    public long getMaxDownloadSize(){
        try{
            lock.lock();
            if( maxDownloadSize==null ){
                maxDownloadSize = request.getMaxDownloadSize();
//                maxDownloadSize = (long)-1;
            }
            return maxDownloadSize;
        }
        finally{
            lock.unlock();
        }
    }

    /**
     * Устанавливает максимум загужаемых данных.
     * @param maxSize макс загружаемых данных, maxSize &lt; 0  - без ограничений
     */
    public void setMaxDownloadSize(long maxSize){
        try{
            lock.lock();
            maxDownloadSize = maxSize;
        }
        finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="started">
    protected Date started = null;

    public Date getStarted() {
        try{
            lock.lock();
            return started;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="finished">
    protected Date finished = null;

    public Date getFinished() {
        try{
            lock.lock();
            return finished;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    /**
     * Проверяет что:
     * <ul>
     * <li>есть ответ сервера</li>
     * <li>статус ответа сервера в диапазоне 200 - 299</li>
     * <li>есть заголовки сервера</li>
     * <li>нет ошибок - свойство errors</li>
     * </ul>
     * @return Ответ не содержит ошибок ( в диапазоне 200-299 )
     */
    public boolean isErrorsNotExists(){
        try{
            lock.lock();
            if( errors!=null && !errors.isEmpty() )return false;
            if( httpHeaders==null )return false;
            if( !(statusCode>=200 && statusCode<=299) )return false;
            return true;
        }
        finally{
            lock.unlock();
        }
    }

    //<editor-fold defaultstate="collapsed" desc="State">
    /**
     * Состояние объекта
     */
    public static enum State {
        /**
         * Начальное состояние
         */
        Prepare,

        /**
         * Посылка запроса
         */
        Started,

        /**
         * Скачивание данных
         */
        Downloading,

        /**
         * Пауза
         */
        Pause,

        /**
         * Конечное состояние, скачивание не производится
         */
        Finished
    }

    public static class StateChangedEvent extends HttpEvent {
        public StateChangedEvent( HttpResponse downloader, State oldState, State newState ){
            this.httpResponse = downloader;
            this.oldState = oldState;
            this.newState = newState;
        }

        protected HttpResponse httpResponse = null;
        protected State oldState = null;
        protected State newState = null;

        public HttpResponse getHttpResponse() {
            return httpResponse;
        }

        public State getOldState() {
            return oldState;
        }

        public State getNewState() {
            return newState;
        }
    }

    protected State state = State.Prepare;

    /**
     * Возвращает текущее состояние объекта
     * @return текущее состояние
     */
    public State getState(){
        try{
            lock.lock();
            return state;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Устанавливает текущее состояние
     * @param newState текущее состояние
     */
    protected void setState(State newState){
        State old = null;
        boolean fire = false;
        try{
            lock.lock();
            old = this.state;
            this.state = newState;
            if( old!=newState ){
                if( State.Started.equals(newState) ){
                    started = new Date();
                }
                if( State.Finished.equals(newState) ){
                    finished = new Date();
                }
                fire = true;
            }
        }finally{
            lock.unlock();
        }
        if( fire )fireEvent(new StateChangedEvent(this, old, newState));
    }

    /**
     * Возвращает прзнак что объект завершил работу и находится в конечном состоянии
     * @return конечное состояние
     */
    @Override
    public boolean isFinished(){
        try{
            lock.lock();
            return State.Finished.equals(this.state);
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="statusCode">
    protected int statusCode = -1;

    /**
     * Код ответа сервера
     * @return код ответа сервера или -1 если еще не завершено
     */
    public int getStatusCode() {
        try{
            lock.lock();
            return statusCode;
        }finally{
            lock.unlock();
        }
    }

    protected void setStatusCode(int newStatusCode) {
        try{
            lock.lock();
            this.statusCode = newStatusCode;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="statusMessage">
    protected String statusMessage = "";
    public String getStatusMessage() {
        try{
            lock.lock();
            return statusMessage;
        }finally{
            lock.unlock();
        }
    }
    protected void setStatusMessage(String statusMessage) {
        try{
            lock.lock();
            this.statusMessage = statusMessage;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="thread">
    protected Thread thread = null;
    public Thread getThread(){
        try{
            lock.lock();
            return thread;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="async">
    protected Boolean async = null;
    public boolean isAsync(){
        try{
            lock.lock();
            if( async==null ){
                if( request!=null ){
                    async = request.isAsync();
                }else{
                    async = false;
                }
            }
            return async;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="downloadBufferSize">
    protected int downloadBufferSize = 1024 * 64;
    public int getDownloadBufferSize(){
        try{
            lock.lock();
            return downloadBufferSize;
        }finally{
            lock.unlock();
        }
    }
    public void setDownloadBufferSize( int buffSize ){
        try{
            lock.lock();
            if( buffSize<1 )throw new IllegalArgumentException("buffSize < 1");
            this.downloadBufferSize = buffSize;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="httpHeaders">
    private HttpHeaders httpHeaders = null;

    public HttpHeaders getHttpHeaders() {
        try{
            lock.lock();
            return httpHeaders;
        }finally{
            lock.unlock();
        }
    }
    protected void setHttpHeaders( HttpHeaders headers ){
        try{
            lock.lock();
            this.httpHeaders = headers;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentBuffer">
    protected ContentBuffer contentBuffer = null;
    public ContentBuffer getContentBuffer(){
        try{
            lock.lock();
            if( contentBuffer==null ){
                if( request!=null ){
                    ContentBuffer cb = request.getContentBuffer();
                    if( cb!=null ){
                        contentBuffer = cb;
                    }else{
                        contentBuffer = new MemContentBuffer();
                    }
                }else{
                    contentBuffer = new MemContentBuffer();
                }
            }
            return contentBuffer;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentCharset">
    protected Charset contentCharset = null;
    public Charset getContentCharset(){
        try{
            lock.lock();
            if( contentCharset!=null )return contentCharset;
            if( httpHeaders!=null ){
                Charset cs = httpHeaders.getContentTypeCharset();
                if( cs!=null ){
                    contentCharset = cs;
                    return cs;
                }
            }else{
                if( request!=null ){
                    HttpClient hc = request.getHttpClient();
                    if( hc!=null ){
                        Charset cs = hc.getDefaultCharset();
                        return cs;
                    }
                }
            }

            return Charset.forName("ISO-8859-1");
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="text">
    protected String text = null;

    public String getText(){
        try{
            lock.lock();
            if( text!=null )return text;

            Charset cs = getContentCharset();
            if( cs==null )return null;

            StringBuilder buff = new StringBuilder();
            char[] cbuf = new char[getDownloadBufferSize()];
            InputStreamReader reader = new InputStreamReader(
                    new ContentBufferInputStream( getContentBuffer(), 0, getDownloadedSize() ),
                    cs);
            while( true ){
                int r;
                try {
                    r = reader.read(cbuf);
                } catch (IOException ex) {
                    Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
                    break;
                }
                if( r<0 )break;
                if( r>0 )buff.append(cbuf,0,r);
            }

            String txt = buff.toString();

            if( isFinished() ){
                text = txt;
            }

            return txt;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentWriterPos">
    protected Long contentWriterPos = null;

    public long getContentWriterPos() {
        try{
            lock.lock();
            if( contentWriterPos==null ){
                if( request!=null ){
                    contentWriterPos = request.getContentOffsetStart();
                    if( contentWriterPos<0 )contentWriterPos = (long)0;
                }else{
                    contentWriterPos = (long)0;
                }
            }
            return contentWriterPos;
        }finally{
            lock.unlock();
        }
    }

    protected void setContentWriterPos(long pos) {
        try{
            lock.lock();
            contentWriterPos = pos;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="threadName">
    /**
     * Имя паралельного потока
     */
    protected String threadName = null;

    /**
     * Имя паралельного потока
     * @return Имя паралельного потока, возможно null
     */
    public String getThreadName() {
        try{
            lock.lock();
            return threadName;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Имя паралельного потока
     * @param threadName Имя паралельного потока, возможно null
     */
    public void setThreadName(String threadName) {
        try{
            lock.lock();
            this.threadName = threadName;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="followRedirect">
    protected Boolean followRedirect = null;

    /**
     * Переходить по redirect
     * @return true - переход по redirect
     */
    public boolean isFollowRedirect() {
        try{
            lock.lock();
            if( followRedirect==null )followRedirect = request.isFollowRedirect();
            return followRedirect;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Переходить по redirect
     * @param followRedirect true - переход по redirect
     */
    public void setFollowRedirect(boolean followRedirect) {
        try{
            lock.lock();
            this.followRedirect = followRedirect;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="redirects">
    protected final List<HttpResponse> redirects = new ArrayList<HttpResponse>();
    protected final List<Pair<URL,URL>> redirectUrls = new ArrayList<Pair<URL,URL>>();

    /**
     * Получение списка ответов редиректа
     * @return список редиректов
     */
    public List<HttpResponse> getRedirectResponses(){
        List<HttpResponse> res = new ArrayList<HttpResponse>();
        try{
            lock.lock();
            res.addAll(redirects);
        }finally{
            lock.unlock();
        }
        return res;
    }

    /**
     * Получение списка редиректов
     * @return список редиректов
     */
    public List<Pair<URL,URL>> getRedirectUrls(){
        List<Pair<URL,URL>> res = new ArrayList<Pair<URL,URL>>();
        try{
            lock.lock();
            res.addAll(redirectUrls);
        }finally{
            lock.unlock();
        }
        return res;
    }

    /**
     * Возвращает последнее перенаправление
     * @return перенаправление или null
     */
    public Pair<URL,URL> getLastRedirectUrl(){
        try{
            lock.lock();
            int s = redirectUrls.size();
            if( s<1 )return null;
            return redirectUrls.get(s-1);
        }
        finally{
            lock.unlock();
        }
    }

    /**
     * Возвращает последний адрес куда было перенаправление
     * @return куда последний раз перенаправило или null
     */
    public URL getLastRedirectUrlTo(){
        try{
            lock.lock();
            Pair<URL,URL> purl = getLastRedirectUrl();
            if( purl!=null ){
                return purl.B();
            }
            return null;
        }
        finally{
            lock.unlock();
        }
    }

    /**
     * Добавление в список редирект
     * @param response редирект
     * @param from откуда переход
     * @param to куда переход
     */
    protected void addRedirect( HttpResponse response, URL from, URL to ){
        try{
            lock.lock();
            redirects.add(response);
            redirectUrls.add(new BasicPair<URL, URL>(from, to));
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="redirectValidate">
    protected Func4<Boolean,HttpHeaders,URL,URL,List<Pair<URL,URL>>> redirectValidate = null;

    /**
     * Возвращает проверку redirect.
     * @return Текущий запрос
     */
    public Func4<Boolean,HttpHeaders,URL,URL,List<Pair<URL,URL>>> getRedirectValidate() {
        try{
            lock.lock();
            if( redirectValidate==null )return redirectValidate;
            return redirectValidate;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Устанавливает проверку redirect.
     * @param redirectValidate функция проверки: <br>
     * boolean fn( HttpHeaders hh, URL from, URL to, List&lt;URL,URL&gt; history )
     */
    public void setRedirectValidate( Func4<Boolean,HttpHeaders,URL,URL,List<Pair<URL,URL>>> redirectValidate ) {
        try{
            lock.lock();
            this.redirectValidate = redirectValidate;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="RedirectEvent">
    public static class RedirectEvent extends HttpEvent {
        protected HttpResponse response = null;
        protected URL from = null;
        protected URL to = null;

        public RedirectEvent( HttpResponse response, URL from, URL to ){
            this.response = response;
            this.from = from;
            this.to = to;
        }

        public HttpResponse getHttpResponse() {
            return response;
        }

        public URL getFrom() {
            return from;
        }

        public URL getTo() {
            return to;
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="currentRequest">
    protected HttpRequest currentRequest = null;

    /**
     * Возвращает текущий запрос.
     * Текущий запрос может отличатся от оригинального в случаии если был redirect
     * @return Текущий запрос
     */
    public HttpRequest getCurrentRequest() {
        try{
            lock.lock();
            if( currentRequest==null )return request;
            return currentRequest;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Устанавливает текущий запрос
     * @param currentRequest текущий запрос
     */
    protected void setCurrentRequest( HttpRequest currentRequest ) {
        try{
            lock.lock();
            this.currentRequest = currentRequest;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="start()">
    /**
     * Запускает процесс скачивания
     * @param sourceRequest Исходный запрос
     * @param connection Соединение
     * @param urlFrom Исходный URL адресса
     */
    @SuppressWarnings( "CallToThreadYield" )
    protected void start(HttpRequest sourceRequest, URL urlFrom, URLConnection connection){
        if( connection==null )throw new IllegalArgumentException( "connection==null" );
        final URLConnection conn = connection;

        setCurrentRequest(sourceRequest);

        // Передача данных на сервер
        try{
            lock.lock();
            if( data!=null ){
                try{
                    data.apply(conn);
                }catch( Throwable err ){
                    logException(err);
                    setState(State.Finished);
                    errors.add(err);
                }
            }
        }finally{
            lock.unlock();
        }

        // остановка
        if( Thread.interrupted() ){
            setState(State.Finished);
            return;
        }

        // Есть сигнал паузы
        if( hasPauseSignal() ){
            setState(State.Pause);
            while( true ){
                if( hasContinueSignal() ){
                    setState(State.Started);
                    break;
                }
                Thread.yield();
            }
        }

        logFine("start {0}",
                HttpResponse.this.getRequest().getUrl());

        InputStream inputStream = null;
        int statusCodeL = -1;
        String statusMessageL = null;

        if( conn instanceof HttpURLConnection ){
            HttpURLConnection httpConnection = (HttpURLConnection)conn;

            try {
                statusCodeL = httpConnection.getResponseCode();
            } catch( IOException ex ) {
                Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
                addError(ex);
                setState(State.Finished);
                return;
            }
            HttpResponse.this.setStatusCode( statusCodeL );

            try {
                statusMessageL = httpConnection.getResponseMessage();
            } catch( IOException ex ) {
                Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
                addError(ex);
                setState(State.Finished);
                return;
            }
            HttpResponse.this.setStatusMessage(statusMessageL);

            if( statusCodeL>=400 ){
                inputStream = httpConnection.getErrorStream();
            }
        }

        Map<String,List<String>> hhz = conn.getHeaderFields();
        HttpHeaders hh = HttpHeaders.createFromMultiMap(hhz);
        HttpResponse.this.setHttpHeaders( hh );

        // Перенаправление
        if( isFollowRedirect() && httpStatusHelper.isRedirect(statusCodeL) ){
            try {
                String loc = hh.getLocation();
                if( loc!=null ){
                    URL urlTo = new URL(loc);

                    boolean validRedirect = true;
                    Func4<Boolean,HttpHeaders,URL,URL,List<Pair<URL,URL>>>
                        redirectValidFun = getRedirectValidate();

                    if( redirectValidFun!=null ){
                        validRedirect =
                            redirectValidFun.apply(hh,urlFrom,urlTo,getRedirectUrls());
                    }

                    if( validRedirect ){
                        HttpRequest req = sourceRequest.clone();
                        req.setUrl(urlTo);
                        req.getHttpHeaders().setReferer(urlFrom.toString());

                        addRedirect(clone(sourceRequest.clone(), true, false), urlFrom, urlTo);

                        try {
                            URLConnection uconn = req.openURLConnection();
                            fireEvent(new RedirectEvent(this, urlFrom, urlTo));
                            start(req, urlTo, uconn);
                            return;

                        } catch( IOException ex ) {
                            Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
                            addError(ex);
                            setState(State.Finished);
                            return;
                        }
                    }
                }
            } catch( MalformedURLException ex ) {
                Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
            }
        }

        if( inputStream==null ){
            try {
                inputStream = conn.getInputStream();
            } catch( IOException ex ) {
                Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
                addError(ex);
                setState(State.Finished);
                return;
            }
        }

        logFine("response:\n"
                + "status {0} {1}\n"
                + "headers\n"
                + "{2}", statusCodeL, statusMessageL,
                Text.indent(hh.toString(), "\n", "[Response] ")
        );

        try {
            startDownloading(inputStream);
        } catch( IOException ex ) {
            Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
            addError(ex);
            setState(State.Finished);
            return;
        }

        getContentBuffer().flush();
        setState(State.Finished);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="pause / resume">
    protected volatile AtomicBoolean needPause = new AtomicBoolean(false);

    public void pause(){
        try{
            lock.lock();
            needPause.set(true);
            continueSignal.set(false);
            logFine("pause thread.isAlive={0}", this.thread!=null ? thread.isAlive() : false );
        }finally{
            lock.unlock();
        }
    }

    protected boolean hasPauseSignal(){
        try{
            lock.lock();
            if( needPause.get() ){
                needPause.set(false);
//                logFine("hasPauseSignal thread.isAlive={0}", this.thread!=null ? thread.isAlive() : false );
                return true;
            }else{
//                logFine("hasPauseSignal thread.isAlive={0}", this.thread!=null ? thread.isAlive() : false );
            }
            return false;
        }finally{
            lock.unlock();
        }
    }

    protected volatile AtomicBoolean continueSignal = new AtomicBoolean(false);

    public void resume(){
        try{
            lock.lock();
            needPause.set(false);
            continueSignal.set(true);
            logFine("resume thread.isAlive={0}", this.thread!=null ? thread.isAlive() : false );
        }finally{
            lock.unlock();
        }
    }

    protected boolean hasContinueSignal(){
        try{
            lock.lock();
            if( continueSignal.get() ){
                continueSignal.set(false);
//                logFine("hasContinueSignal thread.isAlive={0}", this.thread!=null ? thread.isAlive() : false );
                return true;
            }else{
//                logFine("hasContinueSignal thread.isAlive={0}", this.thread!=null ? thread.isAlive() : false );
            }
            return false;
        }finally{
            lock.unlock();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="acceptDownloaded()">
    /**
     * Принимает байты из сети
     * @param buffer данные
     * @param off смещение в buffer
     * @param len объем в buffer
     */
    protected void acceptDownloaded( byte[] buffer, int off, int len ){
        try{
            lock.lock();

            boolean skipWrite = false;

            long maxDownload = getMaxDownloadSize();
            int writeLen = len;

            if( maxDownload>=0 ){
                // есть ограничение по размеру
//                if( downloadedSize >= maxDownload ){
//                    // ограничение достигнуто
//                    skipWrite = true;
//                }else{
                    // целевой размер закаченных данных
                    long targetDownloaded = downloadedSize + writeLen;
                    if( targetDownloaded > maxDownload ){
                        // целевой размер превышает лимит
                        // на сколько превышает
                        long overload = targetDownloaded - maxDownload;
                        writeLen = (int)( targetDownloaded - overload );
                        if( writeLen<0 ){
                            skipWrite = true;
                        }
//                    }else{
//                        // целевой размер не превышает лимит
                    }
//                }
            }

            if( !skipWrite ){
                long pos = getContentWriterPos();
                getContentBuffer().set(pos, buffer, off, writeLen);
                pos += writeLen;
                setContentWriterPos(pos);

                downloadedSize += writeLen;
            }
        }finally{
            lock.unlock();
        }
        fireEvent(new ProgressEvent(this));
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="startDownloading(InputStream)">
    protected void startDownloading(InputStream inputStream) throws IOException {
        setState(State.Downloading);
        logFine("State.Downloading");

        int st = 0;
        boolean stopCycle = false;

        while( !stopCycle ){
            switch( st ){
                case 0:{
                    long maxSize = getMaxDownloadSize();
                    long downloaded = getDownloadedSize();
                    if( maxSize>=0 && downloaded>=maxSize ){
                        stopCycle = true;
                        logFine("stop download");
                        break;
                    }

                    byte[] buff = new byte[getDownloadBufferSize()];
                    int readed = inputStream.read(buff);
                    if( readed<0 ){
                        stopCycle = true;
                        logFine("stop reading ");
                        break;
                    }
                    if( readed>0 ){
                        acceptDownloaded(buff, 0, readed);
                        logFine("readed {0}",readed);
                    }
                    if( Thread.interrupted() ){
                        stopCycle = true;
                        logFine("stop download");
                        break;
                    }
                    if( hasPauseSignal() ){
                        st = 1;
                        setState(State.Pause);
                        logFine("set State.Pause");
                        break;
                    }
                }
                break;
                case 1:{
                    if( hasContinueSignal() ){
                        st = 0;
                        setState(State.Downloading);
                        logFine("set State.Downloading");
                        break;
                    }
                    if( Thread.interrupted() ){
                        stopCycle = true;
                        logFine("stop download");
                        break;
                    }
                    Thread.yield();
                }
                break;
            }
        }

        logFine("end work trhead, close stream downloaded {0}",getDownloadedSize());
        inputStream.close();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="stop()">
    /**
     * Остановка скаичвания
     * @param waitStopMS время через которое повторно посылается сигнал остановки
     * @param forceStopMS время через которое принудительно закрывается thread скачивания
     * @param sleepTimeMS время засыпания потока, или 0 - для передачи Thread.yield()
     */
    public void stop(long waitStopMS, long forceStopMS, long sleepTimeMS){
        if( waitStopMS<0 )throw new IllegalArgumentException( "waitStopMS<0" );
        if( forceStopMS<0 )throw new IllegalArgumentException( "forceStopMS<0" );

        Thread t = thread;
        if( t==null )return;

        t.interrupt();

        Date d1 = new Date();
        while( t.isAlive() ){
            Date d2 = new Date();

            long ddiff = d2.getTime() - d1.getTime();
            if( ddiff>=waitStopMS ){
                t.interrupt();
            }

            if( ddiff>=forceStopMS ){
                t.stop();
            }

            if( sleepTimeMS<=0 ){
                Thread.yield();
            }else{
                try {
                    Thread.sleep(sleepTimeMS);
                    Thread.yield();
                } catch (InterruptedException ex) {
                    Logger.getLogger(HttpResponse.class.getName()).log(Level.FINE, null, ex);
                    break;
                }
            }
        }
    }

    /**
     * Остановка скаивания данных
     */
    public void stop(){
        int readtimeout = request.getReadTimeout();
        if( readtimeout>0 ){
            stop( 100, readtimeout * 3, 50 );
        }else{
            stop( 100, 1000 * 5, 50 );
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="waitForFinished()">
    /**
     * Ожидание завершения загрузки.
     * можно вызывать когда работает в параллельном/асинхронном режиме.
     * @see #getState() 
     */
    public void waitForFinished(){
        Thread tt = Thread.currentThread();
        Thread t = null;
        try{
            if( lock!=null )lock.lock();
            t = thread;
        }finally{
            if( lock!=null )lock.unlock();
        }

//        State s = getState();
//        switch( s ){
//            case Prepare:
//                throw new IllegalStateException("not started");
//            case
//        }

        if( t==null )throw new IllegalStateException("not async started");
        if( t.equals(tt) ){
            throw new Error("thread can be locked");
        }
        while( !isFinished() || t.isAlive() ){
            Thread.yield();
        }
    }
    //</editor-fold>

    /**
     * Вызывать код когда запрос перейдет в состояние finished. <br>
     * Вызов можеть производиться из другого потока (если был асихронный вызов). <br>
     * Код будет вызван не более одного раза.
     * @param runOnFinsihed код который надо вызвать
     * @return закрытие объектов
     */
    public Closeable onFinished( final Func1<Object,HttpResponse> runOnFinsihed ){
        if( runOnFinsihed==null )throw new IllegalArgumentException( "runOnFinsihed==null" );

        final CloseableSet cset = new CloseableSet();

        HttpListener listener = new HttpListenerAdapter(){
            private CloseableSet cs = cset;
            private Func1<Object,HttpResponse> call = runOnFinsihed;

//            @Override
//            protected void downloaderFinished(HttpDownloader.StateChangedEvent event, HttpDownloader downloader) {
//                if( cs!=null ){
//                    cs.closeAll();
//                    cs = null;
//                }
//
//                if( call!=null ){
//                    call.apply(downloader);
//                    call = null;
//                }
//            }

            @Override
            protected void responseStateChanged(StateChangedEvent event, HttpResponse response, State oldState, State newState) {
                if( State.Finished.equals(newState) ){
                    if( cs!=null ){
                        cs.closeAll();
                        cs = null;
                    }

                    if( call!=null ){
                        call.apply(response);
                        call = null;
                    }
                }
            }
        };

        cset.add(
            addListener(listener, false)
        );

        Closeable cl = new Closeable() {
            @Override
            public void close() throws IOException {
                cset.closeAll();
            }
        };

        return cl;
    }
}
