/*
 * The MIT License
 *
 * Copyright 2016 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.http;

import java.net.InetSocketAddress;
import java.util.logging.Level;
import java.util.logging.Logger;

import static java.net.InetSocketAddress.createUnresolved;

/**
 * Интернет адрес
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class InetAddress
implements Comparable<InetAddress>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(InetAddress.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

//    protected final Lock lock = new ReentrantLock();
//
//    public Lock getLock() {
//        return lock;
//    }

    /**
     * Конструктор по умолчанию
     */
    public InetAddress(){
    }

    /**
     * Конструктор копирования
     * @param src образец
     */
    public InetAddress(InetAddress src){
        if( src==null )throw new IllegalArgumentException( "src==null" );
        synchronized( src ){
            this.host = src!=null ? src.host : this.host;
            this.port = src!=null ? src.port : this.port;
        }
    }

    /**
     * Конструктор копирования
     * @param src образец
     */
    public InetAddress(InetSocketAddress src){
        if( src!=null ){
            this.host = src.getAddress().getHostAddress();
            this.port = src.getPort();
        }
    }

    public InetAddress( String host, int port ){
        this.host = host;
        this.port = port;
    }

    @Override
    @SuppressWarnings("CloneDoesntCallSuperClone")
    public InetAddress clone(){
        return new InetAddress(this);
    }

    //<editor-fold defaultstate="collapsed" desc="host">
    private String host;

    public final String getHost() {
        synchronized(this){
            return host;
        }
    }

    public final void setHost(String host) {
        synchronized(this){
            this.host = host;
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="port">
    private int port;

    public final int getPort() {
        synchronized(this){
            return port;
        }
    }

    public final void setPort(int port) {
        synchronized(this){
            this.port = port;
        }
    }
//</editor-fold>

    public InetSocketAddress getInetSocketAddress(){
        synchronized(this){
            if( host==null )throw new IllegalStateException("host property not set");
            if( port<0 )throw new IllegalStateException("port property less then 0");
            if( port>65535 )throw new IllegalStateException("port property more then 65535");
            return createUnresolved(host, port);
        }
    }

    @Override
    public int hashCode() {
        synchronized(this){
            int hash = 3;
            hash = 59 * hash + (this.host != null ? this.host.hashCode() : 0);
            hash = 59 * hash + this.port;
            return hash;
        }
    }

    @Override
    @SuppressWarnings( "NestedSynchronizedStatement" )
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }

        final InetAddress other = (InetAddress) obj;

        synchronized(this){
            synchronized(other){
                if ((this.host == null) ? (other.host != null) : !this.host.equals(other.host)) {
                    return false;
                }
                if (this.port != other.port) {
                    return false;
                }
                return true;
            }
        }
    }

    @Override
    @SuppressWarnings( "NestedSynchronizedStatement" )
    public int compareTo(InetAddress adr) {
        if( adr==null )return -1;

        synchronized(this){
            synchronized(adr){
                String h1 = host;
                String h2 = adr.host;

                int cmpp = port==adr.port ? 0 : (
                        port < adr.port ? -1 : 1
                        );

                if(         h1==null && h2==null ){
                    return cmpp;
                }else if(   h1!=null && h2==null ){
                    return -1;
                }else if(   h1==null && h2!=null ){
                    return 1;
                }

                @SuppressWarnings( "null" )
                int cmph = h1.compareToIgnoreCase(h2);

                return cmph==0 ? cmpp : cmph;
            }
        }
    }

    @Override
    public String toString() {
        synchronized(this){
            return "InetAddress{" + "host=" + host + ", port=" + port + '}';
        }
    }
}
