/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.map.ICaseStringMap;
//import xyz.cofe.files.FileUtil;

/**
 * Перечень mime типов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class MimeTypes {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(MimeTypes.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(MimeTypes.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(MimeTypes.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(MimeTypes.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(MimeTypes.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(MimeTypes.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(MimeTypes.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    public static class Qualifier {
        public boolean text = false;
        public boolean html = false;
        public boolean xml = false;
        public boolean javascript = false;
        
        public Qualifier(){
        }
        
        public Qualifier text(boolean text){
            this.text = text;
            return this;
        }
        public Qualifier html(boolean html){
            this.html = html;
            return this;
        }
    }
    
    protected final Map<String,Qualifier> mime2qualifier;
    protected final Map<String,String> extension2mime;
    
    protected final Lock lock;
    
    public MimeTypes(){
        lock = new ReentrantLock();
        
        ICaseStringMap<Qualifier> icaseMap = new ICaseStringMap<MimeTypes.Qualifier>(true);
        mime2qualifier = icaseMap;

        ICaseStringMap<String> icaseMap2 = new ICaseStringMap<String>(true);
        extension2mime = icaseMap2;
        
        boolean propsReaded = false;
        URL uProps = MimeTypes.class.getResource("mime.properties");
        if( uProps!=null ){
            InputStream instr = null;
            try {
                instr = uProps.openStream();
//                Properties p = FileUtil.readProperties(instr);
                Properties p = new Properties();
                p.load(instr);
                read(p);
                propsReaded = true;
            } catch (IOException ex) {
                Logger.getLogger(MimeTypes.class.getName()).log(Level.SEVERE, null, ex);
            } finally {
                try {
                    instr.close();
                } catch (IOException ex) {
                    Logger.getLogger(MimeTypes.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
        initDefault();
    }
    
    protected void initDefault(){
        mime2qualifier.put("text/plain", 
            new Qualifier().text(true));
        mime2qualifier.put("text/php", 
            new Qualifier().text(true));
        mime2qualifier.put("text/xml", 
            new Qualifier().text(true));
        mime2qualifier.put("text/javascript", 
            new Qualifier().text(true));
        mime2qualifier.put("text/html", 
            new Qualifier().text(true).html(true));
        mime2qualifier.put("text/csv", 
            new Qualifier().text(true));
        mime2qualifier.put("text/css", 
            new Qualifier().text(true));
        mime2qualifier.put("text/cmd", 
            new Qualifier().text(true));
        mime2qualifier.put("text/markdown", 
            new Qualifier().text(true));
        mime2qualifier.put("text/rtf", 
            new Qualifier().text(true));
        mime2qualifier.put("text/x-markdown", 
            new Qualifier().text(true));
        
        mime2qualifier.put("application/atom+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/json", 
            new Qualifier().text(true));
        mime2qualifier.put("application/javascript", 
            new Qualifier().text(true));
        mime2qualifier.put("application/soap+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/x-javascript", 
            new Qualifier().text(true));
        mime2qualifier.put("application/rdf+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/rss+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/xhtml+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/xml-dtd", 
            new Qualifier().text(true));
        mime2qualifier.put("application/smil+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/xml", 
            new Qualifier().text(true));
        mime2qualifier.put("application/xml", 
            new Qualifier().text(true));
        
        mime2qualifier.put("image/svg+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("message/http", 
            new Qualifier().text(true));
        mime2qualifier.put("message/imdn+xml", 
            new Qualifier().text(true));
        mime2qualifier.put("model/vrml", 
            new Qualifier().text(true));
    }

    public void read( Properties props ){
        if( props==null )throw new IllegalArgumentException( "props==null" );
        try{
            lock.lock();
            Map<String,String> m = new LinkedHashMap<String, String>();
            for( String k : props.stringPropertyNames() ){
                m.put(k,props.getProperty(k));
            }
            read(m);
        }finally{
            lock.unlock();
        }
    }

    public void read( Map<String,String> props ){
        if( props==null )throw new IllegalArgumentException( "props==null" );
        try{
            lock.lock();
            mime2qualifier.clear();
            extension2mime.clear();
            
            for( String k : props.keySet() ){
                if( k.startsWith("qualifier.") ){
                    String mime = k.substring("qualifier.".length());
                    String qualStr = props.get(k);
                    String[] qual = qualStr.split("\\s*,\\s*");
                    Qualifier q = new Qualifier();
                    for( String qu : qual ){
                        if( qu.equalsIgnoreCase("text") )q.text = true;
                        if( qu.equalsIgnoreCase("html") )q.html = true;
                        if( qu.equalsIgnoreCase("javascript") )q.javascript = true;
                        if( qu.equalsIgnoreCase("xml") )q.xml = true;
                    }
                    mime2qualifier.put(mime, q);
                }
                
                if( k.startsWith("extension.") ){
                    String ext = k.substring("extension.".length());
                    String mime = props.get(k);
                    extension2mime.put(ext, mime);
                }
            }
        }finally{
            lock.unlock();
        }
    }
    
    /**
     * Получает mime тип по расширению файла
     * @param ext расширение файла
     * @return mime тип или null
     */
    public String getMimeForExtension( String ext ){
        if( ext==null )throw new IllegalArgumentException( "ext==null" );
        try{
            lock.lock();
            return extension2mime.get(ext);
        }finally{
            lock.unlock();
        }
    }
    
    /**
     * Проверяет что mime тип относится к текстовым файлам
     * @param mime mime тип
     * @return true - относится к текстовым файлам
     */
    public boolean isText( String mime ){
        if( mime==null )throw new IllegalArgumentException( "mime==null" );
        try{
            lock.lock();
            Qualifier q = mime2qualifier.get(mime);
            if( q==null )return false;
            return q.text;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Проверяет что mime тип относится к javascript файлам
     * @param mime mime тип
     * @return true - относится к javascript файлам
     */
    public boolean isJavascript( String mime ){
        if( mime==null )throw new IllegalArgumentException( "mime==null" );
        try{
            lock.lock();
            Qualifier q = mime2qualifier.get(mime);
            if( q==null )return false;
            return q.javascript;
        }finally{
            lock.unlock();
        }
    }
    
    /**
     * Проверяет что mime тип относится к xml файлам
     * @param mime mime тип
     * @return true - относится к xml файлам
     */
    public boolean isXml( String mime ){
        if( mime==null )throw new IllegalArgumentException( "mime==null" );
        try{
            lock.lock();
            Qualifier q = mime2qualifier.get(mime);
            if( q==null )return false;
            return q.xml;
        }finally{
            lock.unlock();
        }
    }
    
    /**
     * Проверяет что mime тип относится к html файлам
     * @param mime mime тип
     * @return true - относится к html файлам
     */
    public boolean isHtml( String mime ){
        if( mime==null )throw new IllegalArgumentException( "mime==null" );
        try{
            lock.lock();
            Qualifier q = mime2qualifier.get(mime);
            if( q==null )return false;
            return q.html;
        }finally{
            lock.unlock();
        }
    }
    
    public boolean hasQualifier( String mime ){
        if( mime==null )throw new IllegalArgumentException( "mime==null" );
        try{
            lock.lock();
            Qualifier q = mime2qualifier.get(mime);
            if( q==null )return false;
            return true;
        }finally{
            lock.unlock();
        }
    }
}
