/*
 * The MIT License
 *
 * Copyright 2016 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.http;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Прокси адрес
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class Proxy
extends java.net.Proxy
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(Proxy.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    /**
     * Конструктор по умолчанию
     */
    public Proxy(){
        super(java.net.Proxy.Type.HTTP, new InetSocketAddress("localhost", 3128));
        this.address = new InetAddress("localhost", 3128);
        this.type = Type.direct;
    }

    /**
     * Конструктор копирования
     * @param src образец
     */
    public Proxy( Proxy src ){
        super(
            src==null ? java.net.Proxy.Type.DIRECT : src.type() ,
            src==null ? null : src.address()
        );

        if( src!=null ){
            synchronized(src){
                this.type = src.getType();
                this.address = src.getAddress().clone();
            }
        }
    }

    /**
     * Создает клона
     * @return клон
     */
    @Override
    protected Proxy clone() {
        return new Proxy(this);
    }

    /**
     * Конструктор
     * @param type тип
     * @param address адрес
     */
    public Proxy( Type type, InetAddress address ){
        super(
            type==null ? java.net.Proxy.Type.DIRECT : type.get(),
            address==null ? null : address.getInetSocketAddress()
        );

        this.type = type;
        this.address = address;
    }

    /**
     * Конструктор копирования
     * @param src образец
     */
    public Proxy( java.net.Proxy src ){
        super(
            src==null ? java.net.Proxy.Type.DIRECT : src.type(),
            src==null ? null : src.address()
        );

        if( src!=null ){
            Object oaddr = src.address();
            if( oaddr instanceof InetSocketAddress ){
                this.address = new InetAddress((InetSocketAddress)oaddr);
            }

            java.net.Proxy.Type t = src.type();

            if( t==java.net.Proxy.Type.DIRECT )this.type = Type.direct;
            if( t==java.net.Proxy.Type.HTTP )this.type = Type.http;
            if( t==java.net.Proxy.Type.SOCKS )this.type = Type.socks;
        }
    }

    //<editor-fold defaultstate="collapsed" desc="type">
    /**
     * Тип прокси
     */
    public static enum Type {
        /**
         * Прямое соединение
         */
        direct,

        /**
         * http / ftp прокси
         */
        http,

        /**
         * socks прокси протокол
         */
        socks;

        public java.net.Proxy.Type get(){
            switch( this ){
                case direct:
                    return java.net.Proxy.Type.DIRECT;
                case http:
                    return java.net.Proxy.Type.HTTP;
                case socks:
                    return java.net.Proxy.Type.SOCKS;
            }
            return java.net.Proxy.Type.DIRECT;
        }

        public static Type get( java.net.Proxy.Type t ){
            if( t==null )return null;
            switch( t ){
                case HTTP: return http;
                case SOCKS: return socks;
            }
            return direct;
        }
    }

    private Type type;

    /**
     * Указывает тип прокси
     * @return тип прокси
     */
    public Type getType() {
        synchronized(this){
            if( type==null )type = Type.direct;
            return type;
        }
    }

    /**
     * Указывает тип прокси
     * @param type тип прокси
     */
    public void setType(Type type) {
        synchronized(this){
            this.type = type;
        }
    }

    @Override
    public java.net.Proxy.Type type() {
        return getType().get();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="address">
    private InetAddress address;

    /**
     * Указывает адрес прокси
     * @return адрес прокси
     */
    public InetAddress getAddress() {
        synchronized(this){
            return address;
        }
    }

    /**
     * Указывает адрес прокси
     * @param address адрес прокси
     */
    public void setAddress(InetAddress address) {
        synchronized(this){
            this.address = address;
        }
    }

    @Override
    public SocketAddress address() {
        synchronized(this){
            if( address==null )return null;
            synchronized(address){
                SocketAddress sa = address.getInetSocketAddress();
                return sa;
            }
        }
    }
//</editor-fold>

    @Override
    public String toString() {
        synchronized(this){
            return "Proxy{" + "type=" + type + ", address=" + address + '}';
        }
    }
}
