/*
 * The MIT License
 *
 * Copyright 2016 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.http;

import java.net.URI;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import xyz.cofe.text.Text;

/**
 *
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class WildcardUrlPattern implements UrlPattern
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(WildcardUrlPattern.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    public WildcardUrlPattern(){
    }

    public WildcardUrlPattern( WildcardUrlPattern src ){
        if( src!=null ){
            synchronized(src){
                this.pattern = src.getPattern();
            }
        }
    }

    public WildcardUrlPattern( String pattern ){
        if( pattern==null )throw new IllegalArgumentException( "pattern==null" );
        this.pattern = pattern;
    }

    private String pattern = "";

    public String getPattern() {
        synchronized(this){
            if( pattern==null )pattern = "";
            return pattern;
        }
    }

    public void setPattern(String pattern) {
        synchronized(this){
            this.pattern = pattern;
            this.regex = null;
        }
    }

    private Pattern regex;

    public Pattern getRegex(){
        synchronized(this){
            if( regex!=null )return regex;

            String wc = getPattern();
            regex = Text.wildcard(wc, false, true);
            return regex;
        }
    }

    @Override
    public boolean matched(URI uri) {
        synchronized(this){
            if( uri==null )return false;
            String str = uri.toString();
            return getRegex().matcher(str).matches();
        }
    }

    @Override
    public WildcardUrlPattern clone() {
        return new WildcardUrlPattern(this);
    }

    @Override
    public String toString() {
        return "WildcardUrlPattern{" + "pattern=" + pattern + '}';
    }
}
