/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http.download;


import java.io.Closeable;
import java.io.IOException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.http.ContentFragment;
import xyz.cofe.http.HttpDownloader;
import xyz.cofe.http.HttpHeaders;
import xyz.cofe.http.HttpRequest;
import xyz.cofe.http.HttpResponse;
import xyz.cofe.http.HttpStatusHelper;

/**
 * Базовая проверка скаченного фрагмента
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class BasicFragmentValidator
implements FragmentValidator, ResetCounters, GetCounters, BindHttpDownloader
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(BasicFragmentValidator.class.getName());
    private static final Level logLevel = logger.getLevel();

    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected final Lock lock;
    protected static final String COUNTER_PREFIX = "fv.";
    
    public BasicFragmentValidator(){
        lock = new ReentrantLock();
    }

    public BasicFragmentValidator(Lock lock){
        this.lock = lock==null ? new ReentrantLock() : lock;
    }

    public BasicFragmentValidator(BasicFragmentValidator src, Lock lock){
        this.lock = lock==null ? new ReentrantLock() : lock;
        
        if( src!=null ){
            Lock lck = src.lock;
            try{
                if( lck!=null )lck.lock();
                errors = src.getErrors().clone(this.lock);
                downloadedNotMatchRequired = src.getDownloadedNotMatchRequired().clone(this.lock);
                invalidRangeContentStart = src.getInvalidRangeContentStart().clone(this.lock);
                notPartialContent = src.getNotPartialContent().clone(this.lock);
                nullHttpHeaders = src.getNullHttpHeaders().clone(this.lock);
                statusCodeInvalid = src.getStatusCodeInvalid().clone(this.lock);
            }
            finally{
                if( lck!=null )lck.unlock();
            }
        }
    }
    
    @Override
    public BasicFragmentValidator clone(){
        try {
            lock.lock();
            return new BasicFragmentValidator(this, null); 
        }finally{
            lock.unlock();
        }
    }

    public BasicFragmentValidator clone(Lock lock){
        try {
            lock.lock();
            return new BasicFragmentValidator(this, lock); 
        }finally{
            lock.unlock();
        }
    }
    
    protected boolean isFinished( GetPart gp ){
        if( gp==null )return false;

        if( !gp.isFinished() )return false;

        HttpResponse res = gp.getResponse();
        if( res==null ){
            throw new IllegalStateException(
                "HttpResponse not exists"+
                    " for GetPart"+
                    " id="+gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }

        // check httpresponse finished
        if( !res.isFinished() )return false;
        return true;
    }
    
    private WeakHashMap<HttpDownloader,Boolean> bindedDownloaders
        = new WeakHashMap<HttpDownloader, Boolean>();

    @Override
    public Closeable bindHttpDownloader(final HttpDownloader downloader) {
        if( downloader!=null ){
            Map<String,Counter> cntrs = getCounters();
            final LinkedHashSet<String> cntNames = new LinkedHashSet<String>();

            for( Map.Entry<String,Counter> me : cntrs.entrySet() ){
                String cname = me.getKey();
                Counter cnt = me.getValue();
                if( cname!=null && cnt!=null ){
                    cname = COUNTER_PREFIX+cname;
                    cntNames.add(cname);
                    
                    downloader.getCounters().put(cname, cnt);
                }
            }
            
            bindedDownloaders.put(downloader, true);
            
            return new Closeable() {
                @Override
                public void close() throws IOException {
                    Map<String,Counter> cnts = downloader.getCounters();
                    for( String n : cntNames ){
                        cnts.remove(n);
                    }
                    
                    bindedDownloaders.remove(downloader);
                }
            };
        }
        return null;
    }
    
    private Map<String,Counter> counters;

    @Override
    public Map<String, Counter> getCounters() {
        try{
            lock.lock();
            if( counters!=null )return counters;
            counters = new LinkedHashMap<String, Counter>();
            counters.put("errors", getErrors());
            counters.put("statusCodeInvalid", getStatusCodeInvalid());
            counters.put("nullHttpHeaders", getNullHttpHeaders());
            counters.put("notPartialContent", getNotPartialContent());
            counters.put("invalidRangeContentStart", getInvalidRangeContentStart());
            counters.put("downloadedNotMatchRequired", getDownloadedNotMatchRequired());
            return counters;
        }finally{
            lock.unlock();
        }
    }
    
    @Override
    public void resetCounters() {
        try{
            lock.lock();
            getErrors().reset();
            getStatusCodeInvalid().reset();
            getNullHttpHeaders().reset();
            getNotPartialContent().reset();
            getInvalidRangeContentStart().reset();
            getDownloadedNotMatchRequired().reset();
        }finally{
            lock.unlock();
        }
    }
    
    //<editor-fold defaultstate="collapsed" desc="hasErrors">
    //<editor-fold defaultstate="collapsed" desc="errors">
    protected OverflowCounter errors;
    public OverflowCounter getErrors(){
        try{
            lock.lock();
            if( errors==null )errors = new OverflowCounter(0,-1,lock);
            return errors;
        }finally{
            lock.unlock();
        }
    }
    public void setErrors(OverflowCounter cnt){
        try{
            lock.lock();
            getErrors().assign(cnt);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="hasErrors( downl, getpart, fragment )">
    protected boolean hasErrors(
//        HttpDownloader downl,
        GetPart gp,
        ContentFragment cf
    ){
        HttpResponse res = gp.getResponse();
        if( res==null ){
            throw new IllegalStateException(
                "HttpResponse not exists"+
                    " for GetPart"
                    + " id="+gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }

        HttpRequest req = res.getRequest();
        URL reqUrl = req!=null ? req.getUrl() : null;

        // check response errors
        List<Throwable> resErrs = res.getErrors();
        if( resErrs.size()>0 ){
            int errcnt = 0;
            if( isLogFine ){
                for( Throwable err : resErrs ){
                    if( err==null )continue;
                    errcnt++;
                    logFine("{0}. HttpResponse error{class={2} message={1}} status={{3} {4}} id={5} url={6}",
                            errcnt, // 0
                            err.getMessage(), // 1
                            err.getClass().getName(),// 2
                            res.getStatusCode(),     // 3
                            res.getStatusMessage(),  // 4
                            res.id, // 5
                            reqUrl  // 6
                    );
                }
            }
            if( errcnt>0 ){
                OverflowAction act;
                int max;
                int cur;
                
                if( gp instanceof GetPartMirror ){
                    markFailMirror((GetPartMirror)gp);
                }
                
                // increment counter
                OverflowCounter cnt = getErrors().increment();
                max = cnt.getMax();
                cur = cnt.get();
                act = cnt.getOverflowAction();

                if( cnt.overflow() ){
                    switch( act ){
                        case Skip:
                            break;
                        case Stop:
                        default:
                            throw new Error("Overflow HttpResponse errors count (current="+cur+" max="+max+")");
                    }
                }

                return true;
            }
        }

        return false;
    }
//</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="statusCodeInvalid">
    //<editor-fold defaultstate="collapsed" desc="statusCodeInvalid counter">
    protected OverflowCounter statusCodeInvalid;
    public OverflowCounter getStatusCodeInvalid(){
        try{
            lock.lock();
            if( statusCodeInvalid==null )statusCodeInvalid = new OverflowCounter(0,-1,lock);
            return statusCodeInvalid;
        }finally{
            lock.unlock();
        }
    }
    public void seStatusCodeInvalid(OverflowCounter cnt){
        try{
            lock.lock();
            getStatusCodeInvalid().assign(cnt);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="statusCodeInvalid()">
    protected boolean statusCodeInvalid(
//        HttpDownloader downl,
        GetPart gp,
        ContentFragment cf
    ){
        HttpResponse res = gp.getResponse();
        if( res==null ){
            throw new IllegalStateException(
                "HttpResponse not exists"
                    + " for GetPart id="+gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }

        // check status code in success range
        int statusCode = res.getStatusCode();
        if( !(statusCode>=200 && statusCode<300) ){
            if( gp instanceof GetPartMirror ){
                markFailMirror((GetPartMirror)gp);
            }

            OverflowAction act;
            int max;
            int cur;
            
            OverflowCounter cnt = getStatusCodeInvalid().increment();
            act = cnt.getOverflowAction();
            max = cnt.getMax();
            cur = cnt.get();
            
            if( max>0 ){
                if( cur<=max ){
//                    downl.repeatDownload(cf);
                }else{
                    switch( act ){
                        case Skip:
//                            downl.skipDownload(cf);
                            break;
                        case Stop:
                        default:
                            throw new Error(
                                "Overflow HttpResponse invalid status code count "
                                    +" (current="+cur+" max="+max+")"
                            );
                    }
                }
            }else{
//                downl.repeatDownload(cf);
            }
            return true;
        }

        return false;
    }
//</editor-fold>
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="nullHttpHeaders">
    //<editor-fold defaultstate="collapsed" desc="counter">
    protected OverflowCounter nullHttpHeaders;
    public OverflowCounter getNullHttpHeaders(){
        try{
            lock.lock();
            if( nullHttpHeaders==null )nullHttpHeaders = new OverflowCounter(0,-1,lock);
            return nullHttpHeaders;
        }finally{
            lock.unlock();
        }
    }
    public void setNullHttpHeaders(OverflowCounter cnt){
        try{
            lock.lock();
            getNullHttpHeaders().assign(cnt);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="nullHttpHeaders()">
    protected boolean nullHttpHeaders(
//        HttpDownloader downl,
        GetPart gp,
        ContentFragment cf
    ){
        HttpResponse res = gp.getResponse();
        if( res==null ){
            throw new IllegalStateException(
                "HttpResponse not exists"
                    + " for GetPart"
                    + " id="+gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }

        HttpHeaders hh = res.getHttpHeaders();
        if( hh==null ){
            if( gp instanceof GetPartMirror ){
                markFailMirror((GetPartMirror)gp);
            }

            OverflowCounter cnt = getNullHttpHeaders().increment();

            OverflowAction act = cnt.getOverflowAction();
            int max = cnt.getMax();
            int cur = cnt.get();
            
            if( max>0 ){
                if( cur<=max ){
                }else{
                    switch( act ){
                        case Skip:
                            break;
                        case Stop:
                        default:
                            throw new Error("Overflow HttpResponse null HttpHeaders count"
                            +" (current="+cur+" max="+max+")");
                    }
                }
            }else{
//                downl.repeatDownload(cf);
            }
            return true;
        }

        return false;
    }
//</editor-fold>
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="notPartialContent()">
    //<editor-fold defaultstate="collapsed" desc="counter">
    protected OverflowCounter notPartialContent;
    public OverflowCounter getNotPartialContent(){
        try{
            lock.lock();
            if( notPartialContent==null )notPartialContent = new OverflowCounter(0,-1,lock);
            return notPartialContent;
        }finally{
            lock.unlock();
        }
    }
    public void setNotPartialContent(OverflowCounter cnt){
        try{
            lock.lock();
            getNotPartialContent().assign(cnt);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="notPartialContent()">
    protected boolean notPartialContent(
//        HttpDownloader downl,
        GetPart gp,
        ContentFragment cf
    ) {
        HttpResponse res = gp.getResponse();
        if( res==null ){
            throw new IllegalStateException(
                "HttpResponse not exists"
                    + " for GetPart id="
                    +gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }
        
        int statusCode = res.getStatusCode();
        HttpHeaders hh = res.getHttpHeaders();
        if( hh==null ){
            throw new IllegalStateException(
                "HttpResponse HttpHeaders not exists "
                    + "for GetPart id="
                    +gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }
        
        boolean statcode_partialContent =  statusCode==HttpStatusHelper.PARTIAL_CONTENT;
        HttpHeaders.ContentRange contentRange = hh.getContentRange();
        boolean contentRange_exists = contentRange!=null & statcode_partialContent;
        if( !contentRange_exists ){
            if( gp instanceof GetPartMirror ){
                markFailMirror((GetPartMirror)gp);
            }
            
            OverflowCounter cnt = getNotPartialContent().increment();
            int max = cnt.getMax();
            int cur = cnt.get();
            if( max>0 && cur>max ){
                if( OverflowAction.Stop.equals(cnt.getOverflowAction()) ){
                    throw new Error(
                        "Overflow HttpResponse notPartialContent count"
                            +" (current="+cur+" max="+max+")"
                    );
                }
            }
            return true;
        }
        
        return false;
    }
//</editor-fold>
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="invalidRangeContentStart()">
    //<editor-fold defaultstate="collapsed" desc="counter">
    protected OverflowCounter invalidRangeContentStart;
    public OverflowCounter getInvalidRangeContentStart(){
        try{
            lock.lock();
            if( invalidRangeContentStart==null )invalidRangeContentStart = new OverflowCounter(0,-1,lock);
            return invalidRangeContentStart;
        }finally{
            lock.unlock();
        }
    }
    public void setInvalidRangeContentStart(OverflowCounter cnt){
        try{
            lock.lock();
            getInvalidRangeContentStart().assign(cnt);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="invalidRangeContentStart()">
    protected boolean invalidRangeContentStart(GetPart gp, ContentFragment cf) {
        HttpResponse res = gp.getResponse();
        if( res==null ){
            throw new IllegalStateException(
                "HttpResponse not exists"
                    + " for GetPart id="+gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }
        
        HttpHeaders hh = res.getHttpHeaders();
        if( hh==null ){
            throw new IllegalStateException(
                "HttpResponse HttpHeaders not exists "
                    + "for GetPart id="+gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }
        
        HttpHeaders.ContentRange contentRange = hh.getContentRange();
        if( contentRange==null ){
            throw new IllegalStateException(
                "HttpResponse Content-range not exists "
                    + "for GetPart id="+gp.id
                    +" from="+gp.getFragment().getBegin()
                    +" size="+gp.getFragment().getSize()
            );
        }
        
        if( contentRange.from != gp.getFragment().getBegin() ){
            if( gp instanceof GetPartMirror ){
                markFailMirror((GetPartMirror)gp);
            }
            
            OverflowCounter cnt = getInvalidRangeContentStart().increment();
            int max = cnt.getMax();
            int cur = cnt.get();
            if( max>0 && cur>max ){
                if( OverflowAction.Stop.equals(cnt.getOverflowAction()) ){
                    throw new Error("Overflow HttpResponse invalidRangeContentStart count"
                        +" (current="+cur+" max="+max+")"
                    );
                }
            }
            return true;
        }
        
        return false;
    }
//</editor-fold>
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="downloadedNotMatchRequired()">
    //<editor-fold defaultstate="collapsed" desc="counter">
    protected OverflowCounter downloadedNotMatchRequired;
    public OverflowCounter getDownloadedNotMatchRequired(){
        try{
            lock.lock();
            if( downloadedNotMatchRequired==null )downloadedNotMatchRequired = new OverflowCounter(0,-1,lock);
            return downloadedNotMatchRequired;
        }finally{
            lock.unlock();
        }
    }
    public void setDownloadedNotMatchRequired(OverflowCounter cnt){
        try{
            lock.lock();
            getDownloadedNotMatchRequired().assign(cnt);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="downloadedNotMatchRequired()">
    protected boolean downloadedNotMatchRequired(
        GetPart gp,
        ContentFragment cf
    ) {
        long downloaded = gp.getDownloadedSize();
        if( downloaded > Integer.MAX_VALUE ){
            throw new IllegalStateException("fetched more then Integer.max_value");
        }
        
        long cfsize = cf.getSize();
        if( downloaded!=cfsize ){
            if( gp instanceof GetPartMirror ){
                markFailMirror((GetPartMirror)gp);
            }
            
            OverflowCounter cnt = getDownloadedNotMatchRequired().increment();
            int max = cnt.getMax();
            int cur = cnt.get();
            if( max>0 && cur>max ){
                if( OverflowAction.Stop.equals(cnt.getOverflowAction()) ){
                    throw new Error("Overflow HttpResponse downloadedNotMatchRequired count"
                        +" (current="+cur+" max="+max+")"
                    );
                }
            }
            return true;
        }
        
        return false;
    }
//</editor-fold>
//</editor-fold>

    @Override
    public boolean validate(GetPart gp,
                            ContentFragment cf) {
        if( gp==null )return false;
        if( cf==null )return false;

        // check getpart/response finished
        if( !isFinished(gp) )return false;

        // check response errors
        if( hasErrors(gp,cf) )return false;

        // check status code in success range
        if( statusCodeInvalid(gp,cf) )return false;

        // check hash http headers
        if( nullHttpHeaders(gp,cf) )return false;

        // check hash content range
        if( notPartialContent(gp,cf) )return false;

        // check downloaded equaks fragment start
        if( invalidRangeContentStart(gp,cf) )return false;

        // check downloaded equals fragment size
        if( downloadedNotMatchRequired(gp,cf) )return false;

        if( gp instanceof GetPartMirror ){
            markSuccessMirror((GetPartMirror)gp);
        }
        
        return true;
    }
    
    protected void markFailMirror(GetPartMirror gpm){
        Mirrors.NRequest nreq = gpm.getNRequest();
        if( nreq==null ){
            return;
        }
        
        long mirrorID = nreq.getMirrorId();
        String cntName = COUNTER_PREFIX+"mirror(id="+mirrorID+").fail";
        
        for( HttpDownloader downl : bindedDownloaders.keySet().toArray(new HttpDownloader[]{}) ){
            Object ocnt = downl.getCounters().get(cntName);
            LongCounter cnt = null;
            if( !(ocnt instanceof LongCounter) ){
                cnt = new LongCounter(lock);
                downl.getCounters().put(cntName, cnt);
            }else{
                cnt = (LongCounter)ocnt;
            }
            cnt.increment();
        }
    }

    protected void markSuccessMirror(GetPartMirror gpm){
        Mirrors.NRequest nreq = gpm.getNRequest();
        if( nreq==null ){
            return;
        }
        
        long mirrorID = nreq.getMirrorId();
        String cntName = COUNTER_PREFIX+"mirror(id="+mirrorID+").fetch";
        
        for( HttpDownloader downl : bindedDownloaders.keySet().toArray(new HttpDownloader[]{}) ){
            Object ocnt = downl.getCounters().get(cntName);
            LongCounter cnt = null;
            if( !(ocnt instanceof LongCounter) ){
                cnt = new LongCounter(lock);
                downl.getCounters().put(cntName, cnt);
            }else{
                cnt = (LongCounter)ocnt;
            }
            cnt.increment();
        }
    }
}
