/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http.download;


import java.io.Closeable;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func1;
import xyz.cofe.http.ContentFragment;
import xyz.cofe.http.HttpHeaders;
import xyz.cofe.http.HttpListener;
import xyz.cofe.http.HttpRequest;
import xyz.cofe.http.HttpResponse;
import xyz.cofe.http.HttpStatusHelper;

/**
 * Обработака получения фрагмента
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class GetPart {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(GetPart.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected static HttpStatusHelper httpStatusHelper = new HttpStatusHelper();
    public static final AtomicLong idseq = new AtomicLong(0);
    public final long id = idseq.incrementAndGet();

    public GetPart(ContentFragment fragment,HttpRequest request,HttpResponse response){
        if( request==null )throw new IllegalArgumentException( "request==null" );
        if( response==null )throw new IllegalArgumentException( "response==null" );
//        if( from<0 )throw new IllegalArgumentException( "from < 0" );
//        if( size<1 )throw new IllegalArgumentException( "size < 1" );
        if( fragment==null )throw new IllegalArgumentException( "fragment==null" );
        if( fragment.getBegin()<0 )
            throw new IllegalArgumentException( "fragment.begin("+fragment.getBegin()+")<0" );
        if( fragment.getSize()<1 )
            throw new IllegalArgumentException( "fragment.size("+fragment.getSize()+")<1" );
        if( fragment.getSize()>Integer.MAX_VALUE )
            throw new IllegalArgumentException( "fragment.size("+fragment.getSize()+")>Integer.MAX" );

        this.fragemnt = fragment;

        this.request = request;
        this.response = response;
    }
    
    private final ContentFragment fragemnt;
    
    /**
     * Вовзаращет запрашиваемый фрагмент
     * @return фрагмент
     */
    public ContentFragment getFragment(){ return fragemnt; }
    
    //<editor-fold defaultstate="collapsed" desc="request : HttpRequest - Запрос к серверу">
    protected final HttpRequest request;
    
    /**
     * Запрос к серверу
     * @return запрос
     */
    public HttpRequest getRequest(){
        return request;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="response : HttpResponse - Ответ от сервера">
    protected final HttpResponse response;
    
    /**
     * Ответ от сервера
     * @return ответ от сервера
     */
    public HttpResponse getResponse(){
        return response;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="downloadedSize : long - Кол-во полученных данных">
    /**
     * Кол-во полученных данных
     * @return Кол-во полученных данных
     */
    public long getDownloadedSize(){
        return response.getDownloadedSize();
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="finished : boolean - Запрос завершен">
    /**
     * Запрос завершен
     * @return Запрос завершен
     */
    public boolean isFinished(){
        if( response==null )
            throw new IllegalStateException("response==null");
        return response.isFinished();
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="addResponseListener()">
    /**
     * Добавление подписч. на обработку запроса
     * @param listener подписчик
     * @param weak weak ссылка в  списке подписчиков
     * @param closeableReciver полчение интерфейса закрытия/отписки подписчика
     * @return этот же объект
     */
    public GetPart addResponseListener(
        HttpListener listener,
        boolean weak,
        Func1<Object,Closeable> closeableReciver)
    {
        if( listener!=null ){
            Closeable c = response.addListener(listener, weak);
            if( closeableReciver!=null )
                closeableReciver.apply(c);
        }
        return this;
    }
    
    /**
     * Добавление подписч. на обработку запроса
     * @param listener подписчик
     * @param weak weak ссылка в  списке подписчиков
     * @return интерфес закрытия
     */
    public Closeable addResponseListener(
        HttpListener listener,
        boolean weak)
    {
        Closeable c = response.addListener(listener, weak);
        return c;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="start()">
    /**
     * Запуск на выполнение
     * @return этот же объект
     */
    public GetPart start(){
        response.start();
        return this;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="stop()">
    /**
     * остановка скачивания
     */
    public void stop(){
        logFiner("stop {0}-{1}",fragemnt.getBegin(),fragemnt.getEnd());
        response.stop();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="pause()">
    /**
     * пауза скачивания
     */
    public void pause(){
        logFiner("pause {0}-{1}",fragemnt.getBegin(),fragemnt.getEnd());
        response.pause();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="resume()">
    /**
     * возобновление скачивания
     */
    public void resume(){
        logFiner("resume {0}-{1}",fragemnt.getBegin(),fragemnt.getEnd());
        response.resume();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="paused : boolean - пауза установлена">
    /**
     * пауза установлена
     * @return true - пауза
     */
    public boolean isPaused(){
        return HttpResponse.State.Pause.equals( response.getState() );
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="serverResponse : boolean - Возвращает - сервер вернул часть данных, включая status code и заголовки">
    /**
     * Возвращает - сервер вернул часть данных, включая status code и заголовки.
     * @return true - сервер вернул часть данных
     */
    public boolean isServerResponse(){
        return !HttpResponse.State.Prepare.equals(response.getState());
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="successResponse : boolean - Запрос выполнен успешно">
    /**
     * Запрос выполнен успешно. <br>
     * @return true - Запрос выполнен успешно
     */
    public boolean isSuccessResponse(){
        if( !isServerResponse() )throw new IllegalStateException("server not response");
        int sc=response.getStatusCode();
        return httpStatusHelper.isSuccess(sc);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="partialContent : boolean - Запрос содержит фграмент">
    /**
     * Запрос содержит фграмент. <br>
     * @return true - Запрос содержит фграмент
     */
    public boolean isPartialContent(){
        if( !isServerResponse())throw new IllegalStateException("server not response");
        int sc=response.getStatusCode();
        
        boolean statcode =  sc==HttpStatusHelper.PARTIAL_CONTENT;
        boolean cr = getContentRange()!=null;
        return cr & statcode;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contentRange : HttpHeaders.ContentRange - размер фрагмента">
    /**
     * Возвращает информацию о полученом фрагменте. <br>
     * можно вызывать только когда запрос завершен
     * @return размер фрагмента или null
     */
    public HttpHeaders.ContentRange getContentRange(){
        if( !isServerResponse() )throw new IllegalStateException("server not response");
        return response.getHttpHeaders().getContentRange();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="rangeFragmentValid : boolean - Проверяет что полученный диапазон (но не сами данные) соответствет запрошенному диапазону">
    /**
     * Проверяет что полученный диапазон (но не сами данные) соответствет запрошенному диапазону.
     * Можно вызывать когда данные запрос завершен
     * @return true - соответ.
     */
    public boolean isRangeFragmentValid(){
        if( !isFinished() )throw new IllegalStateException("response not finished");
        
        if( !isPartialContent() )return false;
        
        long lDownloaded = getDownloadedSize();
        if( lDownloaded > Integer.MAX_VALUE )throw new IllegalStateException("fetched more then Integer.max_value");
        
        int downloaded = (int)lDownloaded;
        if( downloaded!=fragemnt.getSize() )return false;
        
        HttpHeaders.ContentRange contentRange = getContentRange();
        if( contentRange==null )return false;
        if( contentRange.from != fragemnt.getBegin() )return false;
        
        return true;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="waitForFinished()">
    /**
     * ожидание завершения
     */
    public void waitForFinished(){
        while( !this.isFinished() ){
            Thread.yield();
        }
    }
//</editor-fold>
}
