/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http.download;


import java.io.Closeable;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.cbuffer.ContentBuffer;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Iterators;
import xyz.cofe.common.Hash;
import xyz.cofe.common.ListenersHelper;
import xyz.cofe.http.ContentFragment;
import xyz.cofe.http.ContentFragments;

/**
 * Проверка контента по хэшу
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class HashContentValidator 
implements ContentValidator, ContentValidatorSender
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(HashContentValidator.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="listeners">
    private final ListenersHelper<ContentValidatorListener,ContentValidatorEvent> listeners
        = new ListenersHelper( new Func2<Object, ContentValidatorListener, ContentValidatorEvent>(){
            @Override
            public Object apply(ContentValidatorListener listener, ContentValidatorEvent ev) {
                if( listener!=null ){
                    listener.contentValidatorEvent(ev);
                }
                return null;
            }
        });
    
    public boolean hasListener(ContentValidatorListener listener) {
        return listeners.hasListener(listener);
    }
    
    public Set<ContentValidatorListener> getListeners() {
        return listeners.getListeners();
    }
    
    public Closeable addListener(ContentValidatorListener listener) {
        return listeners.addListener(listener);
    }
    
    public Closeable addListener(ContentValidatorListener listener, boolean weakLink) {
        return listeners.addListener(listener, weakLink);
    }
    
    public void removeListener(ContentValidatorListener listener) {
        listeners.removeListener(listener);
    }
    
    public void fireEvent(ContentValidatorEvent event) {
        listeners.fireEvent(event);
    }
//</editor-fold>
    
    public HashContentValidator(){
    }
    
    public HashContentValidator(HashContentValidator src){
        if( src!=null ){
            ContentFragments scf = src.validFragments;
            if( scf!=null ){
                validFragments = scf.clone();
            }
            
            outside = src.outside;
        }
    }
    
    @Override
    public HashContentValidator clone(){
        return new HashContentValidator(this);
    }
    
    /**
     * Хэш - фрагменты контента
     */
    protected volatile ContentFragments validFragments;

    /**
     * Указывает Хэш - фрагменты контента
     * @return фрагменты
     */
    public ContentFragments getValidFragments() {
        return validFragments;
    }

    /**
     * Указывает Хэш - фрагменты контента
     * @param validFragments  фрагменты
     */
    public void setValidFragments(ContentFragments validFragments) {
        this.validFragments = validFragments;
    }
    
    /**
     * Включать в результат фрагменты не представленные в оригиналом списке
     */
    protected volatile boolean outside = true;

    /**
     * Включать в результат фрагменты не представленные в оригиналом списке
     * @return true - включать в результат
     */
    public boolean isOutside() {
        synchronized( this ){ return outside; }
    }

    /**
     * Включать в результат фрагменты не представленные в оригиналом списке
     * @param outside false - не включать в результат
     */
    public void setOutside(boolean outside) {
        synchronized( this ){ this.outside = outside; }
    }

    @Override
    public ContentValidation validate(ContentBuffer cbuff, ContentFragments fragments) 
    {
        if( cbuff==null )throw new IllegalArgumentException( "cbuff==null" );
        if( fragments==null )throw new IllegalArgumentException( "fragments==null" );
        
        ContentFragments valids = validFragments;
        if( valids==null ){
            throw new IllegalStateException("property 'validFragments' not set");
        }
        
        final LinkedHashSet<ContentFragment> failed = new LinkedHashSet<ContentFragment>();
        final LinkedHashSet<ContentFragment> failedAdds = new LinkedHashSet<ContentFragment>();
        final LinkedHashSet<ContentFragment> succMd5 = new LinkedHashSet<ContentFragment>();
        
        boolean incOutside = outside;
        
        // try lock fragments
        Lock lock1 = fragments.getLock();
        Lock lock2 = valids.getLock();
        if( lock1!=null && lock2!=null ){
            try{
                lock1.lock();
                try{
                    lock2.lock();
                    
                    validate(cbuff, fragments, valids, failed, failedAdds, succMd5);
                    
                }finally{
                    lock2.unlock();
                }
            }finally{
                lock1.unlock();
            }
        }else{
            validate(cbuff, fragments, valids, failed, failedAdds, succMd5);
        }
        
        if( incOutside ){
            failed.addAll(failedAdds);
        }
        
        ContentValidation cv = new ContentValidation() {
            @Override
            public Iterable<ContentFragment> getGood() {
                return succMd5;
            }

            @Override
            public Iterable<ContentFragment> getBad() {
                return failed;
            }
        };
        
        return cv;
    }
    
    /**
     * Проверяет скаченное содержимое
     * @param cbuff содержимое
     * @param fragments фрагменты содержимого
     * @param valids фрагменты содержащие хэши
     * @param failed фрагменты содержимого не прошедшие совпадение хэша
     * @param failedAdds фрагменты которые не содержаться в failed
     * @param successSrcMD5Res фрагменты у которых совпал хэш
     */
    protected void validate( 
        ContentBuffer cbuff, 
        ContentFragments fragments, 
        ContentFragments valids, 
        LinkedHashSet<ContentFragment> failed,
        LinkedHashSet<ContentFragment> failedAdds,
        LinkedHashSet<ContentFragment> successSrcMD5Res
    ){
        Hash hash = new Hash();
        
        long cbuffSize = cbuff.getSize();
        int blockSize = 1024 * 8;

        //------------------------------------------
        // из valids
        // превышение правой границы буфера (например файл еще не докачан)
        Set<ContentFragment> rightOutsideOfBuffer = new LinkedHashSet<ContentFragment>();
        // превышение левой границы буфера (<0)
        Set<ContentFragment> leftOutsideOfBuffer = new LinkedHashSet<ContentFragment>();
        // md5 значение из образца не установлено
        Set<ContentFragment> md5Null = new LinkedHashSet<ContentFragment>();
        // md5 значение из образца имеет не верный размер - длина строки не равна 32
        Set<ContentFragment> md5LengthInvalid = new LinkedHashSet<ContentFragment>();
        // перепутаны границы begin, end
        Set<ContentFragment> beginMoreThenEnd = new LinkedHashSet<ContentFragment>();
        // фрагмент нулевой длинны
        Set<ContentFragment> zeroLen = new LinkedHashSet<ContentFragment>();
        // ошибка вычисления md5
        Set<ContentFragment> failEvalMD5 = new LinkedHashSet<ContentFragment>();
        
        // md5 значение совпадает
        Set<ContentFragment> successMD5 = new LinkedHashSet<ContentFragment>();
        
        // md5 значение совпадает в наборе fragments
        Set<ContentFragment> successSrcMD5 = new LinkedHashSet<ContentFragment>();
        
        // md5 значение не совпадает
        Set<ContentFragment> failMD5 = new LinkedHashSet<ContentFragment>();
        
        //<editor-fold defaultstate="collapsed" desc="проверка md5 буфера">
        int sampleNum = 0;
        int sampleTot = valids.size();
        for( Object ovalid : valids ){
            sampleNum++;
            if( ovalid instanceof ContentFragment ){
                ContentFragment samplecf = (ContentFragment)ovalid;
                
                String sampleMd5 = samplecf.getMd5();
                long sampleEnd = samplecf.getEnd();
                long sampleBegin = samplecf.getBegin();
                if( sampleBegin > sampleEnd ){
                    beginMoreThenEnd.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                long sampleSize = sampleEnd - sampleBegin;
                
                if( sampleSize<1 ){
                    zeroLen.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                if( sampleBegin<0 ){
                    leftOutsideOfBuffer.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                if( cbuffSize<sampleEnd ){
                    rightOutsideOfBuffer.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                if( sampleMd5==null ){
                    md5Null.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                if( sampleMd5.length()!=32 ){
                    md5LengthInvalid.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                byte[] md5data = hash.md5(cbuff, samplecf, blockSize);
                if( md5data==null ){
                    failEvalMD5.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                String md5str = hash.toString(md5data);
                if( md5str==null ){
                    failEvalMD5.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                if( !md5str.equalsIgnoreCase(sampleMd5) ){
                    failMD5.add(samplecf);
                    fireEvent(
                        new ContentValidatorProgressEvent(
                            this, cbuff, valids, samplecf, sampleNum, sampleTot, false));
                    continue;
                }
                
                successMD5.add(samplecf);
                fireEvent(
                    new ContentValidatorProgressEvent(
                        this, cbuff, valids, samplecf, sampleNum, sampleTot, true));
            }
        }
        //</editor-fold>
        
        Iterable<ContentFragment> failedSamples = Iterators.sequence(
            rightOutsideOfBuffer,
            failMD5
        );
        
        for( ContentFragment failedcf : failedSamples ){
            for( Object ocf : fragments ){
                if( ocf instanceof ContentFragment ){
                    ContentFragment cf = (ContentFragment)ocf;
                    
                    if( cf.equalsRange(failedcf) ){
                        failed.add(cf);
                        continue;
                    }
                    
                    if( cf.include(failedcf) ){
                        failed.add(cf);
                        continue;
                    }
                    
                    if( cf.hasIntersection(failedcf) ){
                        failed.add(cf);
                        continue;
                    }
                }
            }
        }
        
        for( Object osrcf : fragments ){
            for( Object osuccf : successMD5 ){
                if( (osrcf instanceof ContentFragment) && (osuccf instanceof ContentFragment) ){
                    ContentFragment srccf = (ContentFragment)osrcf;
                    ContentFragment succf = (ContentFragment)osuccf;
                    if( succf.include(srccf) || succf.equalsRange(srccf) ){
                        successSrcMD5.add(srccf);
                    }
                }
            }
        }
        
        for( ContentFragment failedcf : failedSamples ){
            if( !failed.contains( failedcf ) ){
                failedAdds.add(failedcf);
            }
        }
        
        if( successSrcMD5Res!=null )successSrcMD5Res.addAll(successSrcMD5);
    }
}
