/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http.download;


import java.util.concurrent.locks.Lock;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Синхронный счетчик с ограничением. <br>
 * 
 * Ограничение наступает при превышении максимального значения, 
 * при этом максимальное значение должно быть больше или равно 0.
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class MaxCounter extends IntCounter {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(MaxCounter.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Конструкторы">
    /**
     * Конструктор по умолчанию.
     * Максимальное значение = 100
     */
    public MaxCounter(){
        this.max = 100;
    }
    
    /**
     * Конструктор
     * @param cnt начальное значение
     * @param max максимальное значение
     */
    public MaxCounter(int cnt,int max){
        super(cnt);
        this.max = max;
    }
    
    /**
     * Конструктор
     * @param lock блокировка
     */
    public MaxCounter(Lock lock){
        super(lock);
        this.max = 100;
    }
    
    /**
     * Конструктор
     * @param cnt начальное значение
     * @param max максимальное (-1 - без ограничения)
     * @param lock блокировка
     */
    public MaxCounter(int cnt,int max,Lock lock){
        super(cnt, lock);
        this.max = max;
    }
    
    /**
     * Конструктор копирования
     * @param src образец дял копирования
     */
    public MaxCounter(MaxCounter src){
        super(src);
        if( src!=null ){
            max = src.max;
        }else{
            max = 100;
        }
    }
    
    /**
     * Конструктор копирования
     * @param src образец дял копирования
     * @param lock блокировка
     */
    public MaxCounter(MaxCounter src, Lock lock){
        super(src, lock);
        if( src!=null ){
            try{
                src.lock.lock();
                max = src.max;
            }finally{
                src.lock.unlock();
            }
        }else{
            max = 100;
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="clone()">
    /**
     * Создание клона
     * @return клон
     */
    @Override
    public MaxCounter clone(){
        try{
            lock.lock();
            return new MaxCounter(this);
        }finally{
            lock.unlock();
        }
    }
    
    /**
     * Создание клона
     * @param lock блокировка
     * @return клон
     */
    @Override
    public MaxCounter clone(Lock lock){
        try {
            this.lock.lock();
            return new MaxCounter(this,lock);
        }
        finally {
            this.lock.unlock();
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="max : int">
    private int max = -1;
    
    /**
     * Возвращает максимальное значение.
     * @return максимальное значение, если меньше нуля (max&lt;0) - ограничение не используется
     */
    public int getMax(){
        try {
            lock.lock();
            return max;
        }
        finally {
            lock.unlock();
        }
    }
    
    /**
     * Устанавливает максимальное значение.
     * @param m максимальное значение, если меньше нуля (max&lt;0) - ограничение не используется
     */
    public void setMax(int m){
        try {
            lock.lock();
            max = m;
        }
        finally {
            lock.unlock();
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="increment()">
    /**
     * Увеличивает счечик и возвращает его значение
     * @return значение после увеличения (клон)
     */
    @Override
    public MaxCounter increment(){
        try {
            lock.lock();
            if( value==null )value = 0;
            value++;
            return clone();
        }
        finally {
            lock.unlock();
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="overflow()">
    /**
     Проверяет наличие переполнения счетчика.
     
     * @return true - true - если get() &gt; getMax() &amp;&amp; getMax() &gt;= 0
     */
    public boolean overflow(){
        try {
            lock.lock();
            if( max>=0 && value>max ){
                return true;
            }
            return false;
        }
        finally {
            lock.unlock();
        }
    }
    //</editor-fold>

    @Override
    public void assign(Counter<Integer> cnt) {
        try{
            lock.lock();
            if( cnt instanceof MaxCounter ){
                setMax(((MaxCounter)cnt).getMax());
            }
            super.assign(cnt);
        }finally{
            lock.unlock();
        }
    }
}
