/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http.download;


import java.net.URL;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.WeakHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.list.LockList;
import xyz.cofe.http.HttpRequest;

/**
 * Зеркала с которых качать данные
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class Mirrors {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(Mirrors.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected final Lock lock;
    
    //<editor-fold defaultstate="collapsed" desc="Конструкторы">
    public Mirrors(){
        lock = new ReentrantLock();

        Random secrnd = new SecureRandom();
        this.rnd = new Random(secrnd.nextLong());
        
        List<HttpRequest> list = new ArrayList<HttpRequest>();
        list = new LockList<HttpRequest>(list,lock);
        mirrors = list;
    }
    
    public Mirrors(Lock lock){
        this.lock = lock == null ? new ReentrantLock() : lock;
        
        Random secrnd = new SecureRandom();
        this.rnd = new Random(secrnd.nextLong());

        List<HttpRequest> list = new ArrayList<HttpRequest>();
        list = new LockList<HttpRequest>(list,lock);
        mirrors = list;
    }
    
    public Mirrors( HttpRequest ... requests ){
        lock = new ReentrantLock();
        
        Random secrnd = new SecureRandom();
        this.rnd = new Random(secrnd.nextLong());

        List<HttpRequest> list = new ArrayList<HttpRequest>();
        list = new LockList<HttpRequest>(list,lock);
        mirrors = list;
        
        if( requests!=null )
            for( HttpRequest req : requests )
                if( req!=null )
                    mirrors.add(req);
    }
    
    public Mirrors( Lock lock, HttpRequest ... requests ){
        this.lock = lock == null ? new ReentrantLock() : lock;
        
        Random secrnd = new SecureRandom();
        this.rnd = new Random(secrnd.nextLong());
        
        List<HttpRequest> list = new ArrayList<HttpRequest>();
        list = new LockList<HttpRequest>(list,lock);
        mirrors = list;
        
        if( requests!=null )
            for( HttpRequest req : requests )
                if( req!=null )
                    mirrors.add(req);
    }
    
    public Mirrors( Mirrors src ){
        lock = new ReentrantLock();

        Random secrnd = new SecureRandom();
        this.rnd = new Random(secrnd.nextLong());
        
        List<HttpRequest> list = new ArrayList<HttpRequest>();
        list = new LockList<HttpRequest>(list,lock);
        mirrors = list;
        
        if( src!=null ){
            try{
                src.lock.lock();
                mirrors.addAll(src.mirrors);
            }finally{
                src.lock.unlock();
            }
        }
    }
    
    public Mirrors( Mirrors src, Lock lock){
        this.lock = lock == null ? new ReentrantLock() : lock;
        
        Random secrnd = new SecureRandom();
        this.rnd = new Random(secrnd.nextLong());
        
        List<HttpRequest> list = new ArrayList<HttpRequest>();
        list = new LockList<HttpRequest>(list,lock);
        mirrors = list;
        
        if( src!=null ){
            try{
                src.lock.lock();
                mirrors.addAll(src.mirrors);
            }finally{
                src.lock.unlock();
            }
        }
    }
    //</editor-fold>
    
    @Override
    public Mirrors clone(){
        return new Mirrors(this);
    }
    
    public Mirrors clone(Lock lock){
        return new Mirrors(this, lock);
    }
    
    private final Random rnd;

    private final List<HttpRequest> mirrors;

    /**
     * Возвращает список зеркал
     * @return зеркала
     */
    public HttpRequest[] getMirrors(){
        return mirrors.toArray(new HttpRequest[]{});
    }

    /**
     * Возвращает идентификаторы зеркал
     * @return идентификаторы
     */
    public long[] getMirrorIds(){
        try {
            lock.lock();
            long[] ids = new long[]{};
            for( HttpRequest req : mirrors ){
                if( req==null )continue;
                ids = Arrays.copyOf(ids, ids.length+1);
                ids[ids.length-1] = req.id;
            }
            return ids;
        }
        finally {
            lock.unlock();
        }
    }
    
    /**
     * Возвращает зеркало по идентификатору
     * @param id идентификатор
     * @return зеркало или null
     */
    public HttpRequest getMirrorById( long id ){
        try {
            lock.lock();
            for( HttpRequest req : mirrors ){
                if( req==null )continue;
                if( req.id == id )return req;
            }
            return null;
        }
        finally {
            lock.unlock();
        }
    }
    
    public static class NRequest {
        private HttpRequest request;
        private long mirrorId;
        
        public NRequest( HttpRequest req, long mirrorId ){
            this.request = req;
            this.mirrorId = mirrorId;
        }

        public HttpRequest getRequest() {
            return request;
        }

        public long getMirrorId() {
            return mirrorId;
        }
    }
    
    /**
     * Получает зекрало из списка
     * @return запрос или null, если нет запросов в списке
     */
    public NRequest getNRequest(){
        return getNRequest(rnd.nextInt());
    }
    
    /**
     * Получает n-ый запрос, где n &gt;=0.
     * Если n &gt;= |mirrors|, то возвращается n % |mirrors|.
     * @param n номер
     * @return запрос или null, если нет запросов в списке
     */
    public NRequest getNRequest( int n ){
        try {
            lock.lock();
            int msize = mirrors.size();
            if( msize<0 )msize = -msize; // в принципе не возможная ситуация
            if( msize==0 ) return null;
            int mod = 0;
            if( msize==1 ) {
                mod = 0;
            }else{
                mod = n % msize;
                if( mod<0 ){
                    mod = msize+mod;
                }
                if( mod<0 ){
                    mod = msize-mod;
                }
            }
            HttpRequest req = mirrors.get(mod);
            HttpRequest redirect = redirects.get(req);
            return new NRequest( redirect!=null ? redirect : req, req.id );
        }
        finally {
            lock.unlock();
        }
    }
    
//    protected final Map<Integer,HttpRequest> modRedirect = new LinkedHashMap<Integer, HttpRequest>();
    protected final Map<HttpRequest,HttpRequest> redirects = new WeakHashMap<HttpRequest, HttpRequest>();
    
    public void redirect( long mirrorId, URL from, URL target ){
        if( from==null )throw new IllegalArgumentException( "from==null" );
        if( target==null )throw new IllegalArgumentException( "target==null" );
        HttpRequest req = getMirrorById(mirrorId);
        if( req!=null ){
            HttpRequest rrq = req.clone();
            rrq.setUrl(target);
            rrq.getHttpHeaders().setReferer(from.toString());
            redirects.put(req, rrq);
        }
    }
    
//    /**
//     * Устаналивает для n-го запроса redirect
//     * @param n n-ый запрос
//     * @param from С которого перенаправление
//     * @param redirect Перенаправление куда
//     */
//    public void setNRequestRedirect(int n,HttpResponse from, HttpRequest redirect){
//        if( from==null )throw new IllegalArgumentException( "from==null" );
//        if( redirect==null )throw new IllegalArgumentException( "redirect==null" );
//        try {
//            lock.lock();
//            int msize = mirrors.size();
//            if( msize<0 )msize = -msize; // в принципе не возможная ситуация
//            if( msize==0 ) return;
//            int mod = 0;
//            if( msize==1 ) {
//                mod = 0;
//            }else{
//                mod = n % msize;
//            }
//            modRedirect.put(mod, redirect);
//        }
//        finally {
//            lock.unlock();
//        }
//    }
}
