/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http.download;


import java.io.Closeable;
import java.lang.ref.WeakReference;
import java.util.Date;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.http.ContentFragment;
import xyz.cofe.http.HttpDownloader;
import xyz.cofe.http.HttpListenerAdapter;
import xyz.cofe.http.HttpResponse;

/**
 *
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class ProgressCounters {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(ProgressCounters.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected ProgressCounters(){
    }
    
    public ProgressCounters(HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        registerCounters(downl);
    }
    
    protected void registerCounters( final HttpDownloader downl ){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        
        HttpListenerAdapter lProgress = new HttpListenerAdapter(){
            @Override
            protected void downloaderProgress(HttpDownloader.ProgressEvent event, HttpDownloader downloader) {
                onProgress(downloader, event);
            }
        };
        
        HttpListenerAdapter lState = new HttpListenerAdapter(){
            @Override
            protected void downloaderStateChanged(
                HttpDownloader.StateChangedEvent event, 
                HttpDownloader downloader, 
                HttpDownloader.State oldState, 
                HttpDownloader.State newState) {
                onStateChanged( downloader, event, oldState, newState);
            }
        };
        
        Closeable clProgress = downl.addListener(lProgress);
        Closeable clState = downl.addListener(lState);

        registerContentLength(downl);
        registerTotalLength(downl);
        registerDownloaded(downl);
        registerDownloadedPct(downl);
        registerDownloadedSpeedBytes(downl);
        registerDownloadedSpeedKBytes(downl);
        registerDownloadedSpeedMBytes(downl);
        registerTotalFragments(downl);
        registerValidatedFragments(downl);
        registerDownloadedFragments(downl);
    }
    
    //<editor-fold defaultstate="collapsed" desc="onProgress/onStateChanged">
    protected void onProgress(
        HttpDownloader downloader,
        HttpDownloader.ProgressEvent event
    ){
        updateProgress(downloader);
    }
    protected void onStateChanged(
        HttpDownloader downloader,
        HttpDownloader.StateChangedEvent event,
        HttpDownloader.State oldState,
        HttpDownloader.State newState
    ){
        lastStateChangedDate = new Date();
        lastStateChangedValue = newState;
        updateProgress(downloader);
    }
//</editor-fold>
    
    protected long getTotal( HttpDownloader downloader ){
//        long total = downloader.getFragments().getTotalSize();
        long total = downloader.getTotalSize();
//        if( !downloader.isPartialMode() ){
//            total = downloader.getContentLength();
//        }
        return total;
    }
    protected long getDownloaded( HttpDownloader downloader ){
//        long downloaded = downloader.getFragments().getTotalDownloadedSize();
        long downloaded = downloader.getDownloadedSize();
//        if( !downloader.isPartialMode() ){
//            downloaded = downloader.getStreamDownloadedSize();
//        }
        return downloaded;
    }
    
    protected double pct = -2;
    protected long total = -1;
    protected long downloaded = -1;

    protected Date lastStateChangedDate;
    protected HttpDownloader.State lastStateChangedValue;
    
    protected Date lastProgressDate;
    protected Long lastProgressSize;
    
    protected double bytesPerSecond;
    protected double kBytesPerSecond;
    protected double mBytesPerSecond;
    
    protected WeakReference<HttpDownloader> downloader;
    
    public boolean updateProgress(){
        WeakReference<HttpDownloader> wdownloader = downloader;
        HttpDownloader d = wdownloader==null ? null : wdownloader.get();
        if( d==null )return false;
        
        updateProgress(d);
        return true;
    }
    
    protected void updateProgress( HttpDownloader downloader ){
        total = getTotal(downloader);
        downloaded = getDownloaded(downloader);
        
        pct = (double)(
            total > 0
                ? ( (double)(downloaded) / (double)(total) )
                : -1
        );
        pct = (pct * 1000000.0d);
        int ipct = (int)pct;
        pct = ipct / 10000.0d;
        
        Date now = new Date();
        if( lastProgressDate!=null && lastProgressSize!=null ){
            long tDelta = now.getTime() - lastProgressDate.getTime();
            long sDelta = downloaded - lastProgressSize;

            if( tDelta>=1000 ){
                double k = 1000d / ((double)tDelta);
                double speed = k * sDelta;

                bytesPerSecond = speed;
                kBytesPerSecond = k * ( sDelta / 1024.0 );
                mBytesPerSecond = k * ( sDelta / (1024.0 * 1024.0) );

                lastProgressDate = now;
                lastProgressSize = downloaded;
            }
        }else{
            lastProgressDate = now;
            lastProgressSize = downloaded;
        }
    }
    
    public void resetCounters(){
        bytesPerSecond = -1;
        kBytesPerSecond = -1;
        mBytesPerSecond = -1;
        lastProgressSize = null;
        lastStateChangedDate = null;
        lastStateChangedDate = null;
        lastStateChangedValue = null;
        total = -1;
        pct = -2;
        downloaded = -1;
    }

    protected LongCounter registerContentLength(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        
        String cntname = "progress.contentLength";
        LongCounter lcntr = new LongCounter(downl.getLock()){
            @Override
            public Long get() {
                return downl.getContentLength();
            }
            @Override
            public void reset() { resetCounters(); }
        };
        downl.getCounters().put(cntname, lcntr);
        return lcntr;
    }
    
    protected LongCounter registerTotalLength(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.totalLength";
        LongCounter lcntr = new LongCounter(downl.getLock()){
            @Override
            public Long get() {
                return total;
            }
            @Override
            public void reset() { resetCounters(); }
        };
        downl.getCounters().put(cntname, lcntr);
        return lcntr;
    }
    
    protected LongCounter registerDownloaded(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.downloadedBytes";
        LongCounter lcntr = new LongCounter(downl.getLock()){
            @Override
            public Long get() {
                return downloaded;
            }
            @Override
            public void reset() { resetCounters(); }
        };
        downl.getCounters().put(cntname, lcntr);
        return lcntr;
    }
    
    protected DoubleCounter registerDownloadedPct(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.downloadedPct";
        DoubleCounter cntr = new DoubleCounter(downl.getLock()){
            @Override
            public Double get() {
                return pct;
            }

            @Override
            public void reset() { resetCounters(); }
        };
        downl.getCounters().put(cntname, cntr);
        return cntr;
    }
    
    protected DoubleCounter registerDownloadedSpeedBytes(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.downloadedSpeedBytes";
        DoubleCounter cntr = new DoubleCounter(downl.getLock()){
            @Override
            public Double get() {
                return bytesPerSecond;
            }
            @Override
            public void reset() { resetCounters(); }
        };
        downl.getCounters().put(cntname, cntr);
        return cntr;
    }
    
    protected DoubleCounter registerDownloadedSpeedKBytes(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.downloadedSpeedKBytes";
        DoubleCounter cntr = new DoubleCounter(downl.getLock()){
            @Override
            public Double get() {
                return kBytesPerSecond;
            }
            @Override
            public void reset() { resetCounters(); }
        };
        downl.getCounters().put(cntname, cntr);
        return cntr;
    }
    
    protected DoubleCounter registerDownloadedSpeedMBytes(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.downloadedSpeedMBytes";
        DoubleCounter cntr = new DoubleCounter(downl.getLock()){
            @Override
            public Double get() {
                return mBytesPerSecond;
            }
            @Override
            public void reset() { resetCounters(); }
        };
        downl.getCounters().put(cntname, cntr);
        return cntr;
    }
    
    protected IntCounter registerTotalFragments(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.totalFragments";
        IntCounter cntr = new IntCounter(downl.getLock()){
            @Override
            public Integer get() {
                return downl==null ? 0 : downl.getFragments().size();
            }
        };
        downl.getCounters().put(cntname, cntr);
        return cntr;
    }
    
    protected IntCounter registerDownloadedFragments(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.downloadedFragments";
        IntCounter cntr = new IntCounter(downl.getLock()){
            @Override
            public Integer get() {
                if( downl==null )return 0;
                int c = 0;
                for( Object of : downl.getFragments() ){
                    if( of instanceof ContentFragment ){
                        ContentFragment cf = (ContentFragment)of;
                        if( cf.getDownloadedSize() == cf.getSize() ){
                            c++;
                        }
                    }
                }
                return c;
            }
        };
        downl.getCounters().put(cntname, cntr);
        return cntr;
    }

    protected IntCounter registerValidatedFragments(final HttpDownloader downl){
        if( downl==null )throw new IllegalArgumentException( "downl==null" );
        String cntname = "progress.validatedFragments";
        IntCounter cntr = new IntCounter(downl.getLock()){
            @Override
            public Integer get() {
                if( downl==null )return 0;
                Set set = downl.getValidatedFragments();
                return set==null ? 0 : set.size();
            }
        };
        downl.getCounters().put(cntname, cntr);
        return cntr;
    }
}
