/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.http.web;


import java.io.Closeable;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func4;
import xyz.cofe.collection.Pair;
import xyz.cofe.common.ListenersHelper;
import xyz.cofe.common.State;
import xyz.cofe.common.StateChangedEvent;
import xyz.cofe.http.HttpClient;
import xyz.cofe.http.HttpHeaders;
import xyz.cofe.http.HttpListenerAdapter;
import xyz.cofe.http.HttpRequest;
import xyz.cofe.http.HttpResponse;
import xyz.cofe.http.HttpStatusHelper;
import xyz.cofe.http.IsFinished;
import xyz.cofe.http.MimeTypes;

/**
 * Анализирует структуру сайта
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class SiteScanner 
implements IsFinished
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(SiteScanner.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(SiteScanner.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(SiteScanner.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(SiteScanner.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(SiteScanner.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(SiteScanner.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(SiteScanner.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected final Lock lock;

    public SiteScanner(){
        lock = new ReentrantLock();
    }
    
    //<editor-fold defaultstate="collapsed" desc="listeners / events">
    public interface Listener {
        void siteScannerEvent( Object e );
    }
    
    public static class ListenerAdapter implements Listener {
        @Override
        public void siteScannerEvent(Object e) {
            if( e instanceof CanceledRedirectEvent ){
                CanceledRedirectEvent fe = (CanceledRedirectEvent)e;
                canceledRedirect(fe, fe.siteScanner, fe.from, fe.to);
            }
            if( e instanceof RedirectEvent ){
                RedirectEvent fe = (RedirectEvent)e;
                redirect(fe, fe.siteScanner, fe.from, fe.to);
            }
            if( e instanceof FollowLinkEvent ){
                FollowLinkEvent fe = (FollowLinkEvent)e;
                follow(fe, fe.siteScanner, fe.resource, fe.url);
            }
            if( e instanceof NoFollowLinkEvent ){
                NoFollowLinkEvent fe = (NoFollowLinkEvent)e;
                noFollow(fe, fe.siteScanner, fe.resource, fe.url);
            }
            if( e instanceof ScanBeginEvent ){
                ScanBeginEvent fe = (ScanBeginEvent)e;
                beginScan(fe, fe.siteScanner, fe.resource);
            }
            if( e instanceof ScanEndEvent ){
                ScanEndEvent fe = (ScanEndEvent)e;
                endScan(fe, fe.siteScanner, fe.resource);
            }
        }
        
        protected void follow( FollowLinkEvent ev, SiteScanner si, Resource res, URL url ){
        }

        protected void noFollow( NoFollowLinkEvent ev, SiteScanner si, Resource res, URL url ){
        }

        protected void beginScan( ScanBeginEvent ev, SiteScanner si, Resource res ){
        }
        
        protected void endScan( ScanEndEvent ev, SiteScanner si, Resource res ){
        }

        protected void redirect( RedirectEvent ev, SiteScanner si, URL from, URL to ){
        }

        protected void canceledRedirect( CanceledRedirectEvent ev, SiteScanner si, URL from, URL to ){
        }
    }

    protected final xyz.cofe.common.ListenersHelper<Listener,Object>
        listeners = new ListenersHelper<Listener, Object>(new Func2<Object, Listener, Object>(){
            @Override
            public Object apply(Listener listener, Object ev) {
                listener.siteScannerEvent(ev);
                return null;
            }
        });
    
    public boolean hasListener(Listener listener) {
        return listeners.hasListener(listener);
    }
    
    public Set<Listener> getListeners() {
        return listeners.getListeners();
    }
    
    public Closeable addListener(Listener listener) {
        return listeners.addListener(listener);
    }
    
    public Closeable addListener(Listener listener, boolean weakLink) {
        return listeners.addListener(listener, weakLink);
    }
    
    public void removeListener(Listener listener) {
        listeners.removeListener(listener);
    }
    
    public void fireEvent(Object event) {
        listeners.fireEvent(event);
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="httpClient">
    protected HttpClient httpClient = null;
    public HttpClient getHttpClient(){
        try{
            lock.lock();
            if( httpClient==null )httpClient = new HttpClient();
            return httpClient;
        }
        finally{
            lock.unlock();
        }
    }
    
    public void setHttpClient( HttpClient client ){
        if( client==null )throw new IllegalArgumentException( "client==null" );
        try{
            lock.lock();
            httpClient = client;
        }
        finally{
            lock.unlock();
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="mime">
    protected MimeTypes mimeTypes = null;
    public MimeTypes getMimeTypes(){
        try{
            lock.lock();
            if( mimeTypes==null )mimeTypes = new MimeTypes();
            return mimeTypes;
        }
        finally{
            lock.unlock();
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="preview">
    protected void preview( HttpResponse response ){
        if( response==null )throw new IllegalArgumentException( "response==null" );
        
        List<Throwable> errors = response.getErrors();
        if( errors!=null && !errors.isEmpty() ){
            for( Throwable err : errors ){
                System.out.println(
//                    Text.template("error {0}", err.getMessage())
                    "error "+err.getMessage()
                );
            }
        }
        
        List<Pair<URL,URL>> redirects = response.getRedirectUrls();
        if( redirects!=null && !redirects.isEmpty() ){
            for( Pair<URL,URL> redirect : redirects ){
                System.out.println(
//                    Text.template("redirect {0} -> {1}", redirect.A(), redirect.B())
                    "redirect "+redirect.A()+" -> "+redirect.B()
                );
            }
        }
        
        int statusCode = response.getStatusCode();
        String statusMessage = response.getStatusMessage();
        System.out.println(
//            Text.template("status {0} {1}", statusCode, statusMessage)
            "status "+statusCode+" "+statusMessage
        );
        
        System.out.println("response headers:");
        System.out.println(response.getHttpHeaders());
        
        long prevBytesLenLong = response.getDownloadedSize();
        int prevBytesLenMax = 512;
        int prevBytesLen = prevBytesLenLong > prevBytesLenMax ?
            prevBytesLenMax : (int)prevBytesLenLong;
        
        byte[] prevBytes = response.getContentBuffer().get(0, prevBytesLen);
        
        Charset cs = response.getHttpHeaders().getContentTypeCharset();
        if( cs==null )cs = response.getRequest().getHttpClient().getDefaultCharset();
        
        String prevText = new String(prevBytes, cs);
        System.out.println("preview:");
        System.out.println(prevText);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="history">
    protected Set<URL> history = new LinkedHashSet<URL>();
    
    public Set<URL> getHistory() {
        try{
            lock.lock();
            LinkedHashSet<URL> copy = new LinkedHashSet<URL>();
            copy.addAll(history);
            return copy;
        }
        finally{
            lock.unlock();
        }
    }
    
    protected void addHistory( URL url ){
        try{
            lock.lock();
            history.add(url);
        }
        finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //    //<editor-fold defaultstate="collapsed" desc="url2resource">
//    protected Map<URL,Resource> url2resource = new LinkedHashMap<URL, Resource>();
//    
//    public Map<URL, Resource> getUrl2resource() {
//        try{
//            lock.lock();
//            return url2resource;
//        }
//        finally{
//            lock.unlock();
//        }
//    }
////</editor-fold>

    //    //<editor-fold defaultstate="collapsed" desc="resource2url">
//    protected Map<Resource,URL> resource2url = new LinkedHashMap<Resource, URL>();
//    
//    public Map<Resource, URL> getResource2url() {
//        try{
//            lock.lock();
//            return resource2url;
//        }
//        finally{
//            lock.unlock();
//        }
//    }
//    //</editor-fold>

    //    //<editor-fold defaultstate="collapsed" desc="root">
//    protected Resource root = null;
//    
//    public Resource getRoot() {
//        try{
//            lock.lock();
//            return root;
//        }
//        finally{
//            lock.unlock();
//        }
//    }
//    
//    protected void setRoot(Resource root) {
//        try{
//            lock.lock();
//            this.root = root;
//        }
//        finally{
//            lock.unlock();
//        }
//    }
//    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="state">
    protected State state = State.Ready;
    
    public State getState() {
        try{
            lock.lock();
            return state;
        }
        finally{
            lock.unlock();
        }
    }
    
    public void setState(State state) {
        try{
            lock.lock();
            this.state = state;
        }
        finally{
            lock.unlock();
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="async">
    protected Boolean async = null;
    /**
     * Указывает асинхронный режим
     * @return true - асинхронный; false (по умолч) - синхронный
     */
    public boolean isAsync(){
        try{
            lock.lock();
            if( async==null )async = false;
            return async;
        }finally{
            lock.unlock();
        }
    }
    /**
     * Указывает асинхронный режим
     * @param async true - асинхронный; false - синхронный
     */
    public void setAsync(boolean async){
        try{
            lock.lock();
            this.async = async;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="thread">
    protected Thread thread = null;
    public Thread getThread(){
        try{
            lock.lock();
            return thread;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="stop()">
    /**
     * Остановка скаичвания
     * @param waitStopMS время через которое повторно посылается сигнал остановки
     * @param forceStopMS время через которое принудительно закрывается thread скачивания
     * @param sleepTimeMS время засыпания потока, или 0 - для передачи Thread.yield()
     */
    public void stop(long waitStopMS, long forceStopMS, long sleepTimeMS){
        if( waitStopMS<0 )throw new IllegalArgumentException( "waitStopMS<0" );
        if( forceStopMS<0 )throw new IllegalArgumentException( "forceStopMS<0" );

        Thread t = thread;
        if( t==null )return;

        t.interrupt();

        Date d1 = new Date();
        while( t.isAlive() ){
            Date d2 = new Date();

            long ddiff = d2.getTime() - d1.getTime();
            if( ddiff>=waitStopMS ){
                t.interrupt();
            }

            if( ddiff>=forceStopMS ){
                t.stop();
            }

            if( sleepTimeMS<=0 ){
                Thread.yield();
            }else{
                try {
                    Thread.sleep(sleepTimeMS);
                    Thread.yield();
                } catch (InterruptedException ex) {
                    Logger.getLogger(HttpResponse.class.getName()).log(Level.SEVERE, null, ex);
                    break;
                }
            }
        }
    }

    /**
     * Остановка
     */
    public void stop(){
        int readtimeout = 30000;
        if( readtimeout>0 ){
            stop( 100, readtimeout * 3, 50 );
        }else{
            stop( 100, 1000 * 5, 50 );
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="waitForFinished()">
    /**
     * Ожидание завершения загрузки.
     * можно вызывать когда работает в параллельном/асинхронном режиме.
     * @see #getState() 
     */
    public void waitForFinished(){
        Thread tt = Thread.currentThread();
        Thread t = null;
        try{
            if( lock!=null )lock.lock();
            t = thread;
        }finally{
            if( lock!=null )lock.unlock();
        }

//        State s = getState();
//        switch( s ){
//            case Prepare:
//                throw new IllegalStateException("not started");
//            case
//        }

        if( t==null )throw new IllegalStateException("not async started");
        if( t.equals(tt) ){
            throw new Error("thread can be locked");
        }
        while( !isFinished() || t.isAlive() ){
            Thread.yield();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isFinished()">
    /**
     * Возвращает прзнак что объект завершил работу и находится в конечном состоянии
     * @return конечное состояние
     */
    @Override
    public boolean isFinished(){
        try{
            lock.lock();
            return State.Finished.equals(this.state);
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="start( url )">
    public void start(final URL url){
        if( url==null )throw new IllegalArgumentException("url == null");
        Object e = null;
        
        if( !isAsync() ){
            State s = getState();
            if( !(State.Ready.equals(s) || State.Finished.equals(s)) ){
                throw new IllegalStateException("state ("+s+") not ( ready | finished )");
            }
            
            setState(State.Running);
            e = new StateChangedEvent<SiteScanner>(this, s, getState());
            fireEvent(e);
            
            scan(url,null);

            State olds = getState();
            setState(State.Finished);
            fireEvent(new StateChangedEvent(this, olds, getState()));
        }else{
            try{
                lock.lock();
                
                State s = getState();
                if( !(State.Ready.equals(s) || State.Finished.equals(s)) ){
                    throw new IllegalStateException("state ("+s+") not ( ready | finished )");
                }
                
                setState(State.Running);
                e = new StateChangedEvent(this, s, getState());
                
                Runnable r = new Runnable() {
                    @Override
                    public void run() {
                        scan(url,null);

                        State s = getState();
                        setState(State.Finished);
                        fireEvent(new StateChangedEvent(SiteScanner.this, s, getState()));
                    }
                };
                thread = new Thread(r);
                thread.setDaemon(true);
                thread.setPriority(Thread.MIN_PRIORITY);
                thread.start();
            }finally{
                lock.unlock();
            }
            if( e!=null )fireEvent(e);
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="scan(), scanLinks()">
    protected void scan(URL url, Resource parent){
        if( parent==null ){
            getHistory().clear();
        }
        
        Resource r = inspectResource(url);
        if( r!=null ){
            addHistory(url);
            
            List<Pair<URL,URL>> lredirects = r.getRedirects();
            if( lredirects!=null && !lredirects.isEmpty() ){
                for( Pair<URL,URL> p : lredirects ){
                    addHistory(p.A());
                    addHistory(p.B());
                }
            }
            
            scan(url, r, parent);
        }
    }

    protected void scan(URL url, Resource resource, Resource referer) {
        fireEvent(new ScanBeginEvent(this, url, resource, referer));
        if( !Thread.interrupted() ){
            scanLinks(resource);
        }
        fireEvent(new ScanEndEvent(this, url, resource, referer));
    }
    
    protected void scanLinks( Resource r ){
        HtmlPage page = r.getHtmlPage();
        
        if( page==null ){
            return;
        }
        
        for( FollowLink fl : page.getFollowLinks() ){
            if( allowFollow(r, fl) ){
                fireEvent(new FollowLinkEvent(this, r, fl.link));
                scan(fl.link, r);
            }else{
                fireEvent(new NoFollowLinkEvent(this, r, fl.link));
            }
        }
    }
    //</editor-fold>
    
    /**
     * Проверка что указанный адрес уже числеце в посещенных
     * @param url адрес
     * @return true - этот адрес уже посещен
     */
    protected boolean isVisited( URL url ){
        return getHistory().contains(url);
    }
    
    //<editor-fold defaultstate="collapsed" desc="allowRedirect()">
    /**
     * Проверка на возможность перехода по ответу сервера redirect.  <br>
     * Переход возможен если соблюдено след: <br>
     * Ссылка введет в на этот же хост <br>
     * И Ссылка еще раньше не была обработана <br>
     * @param from откуда
     * @param to целевой адрес
     * @return true - переход разрешен
     */
    protected boolean allowRedirect( URL from, URL to ){
        if( isVisited(to) )return false;
        if( !from.getHost().equals(to.getHost()) )return false;
        return true;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="allowFollow()">
    /**
     * Проверка возможности перехода со страницы.  <br>
     * Переход возможен если соблюдено след: <br>
     * Ссылка введет в на этот же хост <br>
     * И Ссылка еще раньше не была обработана <br>
     * 
     * @param res ресурс со страницой
     * @param flink куда переход
     * @return true - переход разрешен
     */
    protected boolean allowFollow( Resource res, FollowLink flink ){
        if( flink==null )return false;
        
        if( isVisited(flink.link) )return false;
        if( !flink.sameHost )return false;
        
//        if( !flink.isChildPath )return false;
        
        return true;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="redirectValidate">
    protected Func4<Boolean,HttpHeaders,URL,URL,List<Pair<URL,URL>>> redirectValidate =
        new Func4<Boolean, HttpHeaders, URL, URL, List<Pair<URL, URL>>>() {
            @Override
            public Boolean apply(HttpHeaders hh, URL from, URL to, List<Pair<URL, URL>> arg4) {
                if( !allowRedirect(from,to) ){
                    fireEvent(new CanceledRedirectEvent(SiteScanner.this, from, to));
                    return false;
                }
                return true;
            }
        };
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="inspectResource()">
    private void catchResponse
    (
        HttpResponse res1, 
        final Resource resource,
        final URL url,
        final HttpStatusHelper httpStatusHelper,
        final AtomicBoolean stopSignal
    ) {
        int statusCode = res1.getStatusCode();
        
        resource.setStatusCode(statusCode);
        resource.setStatusMessage( res1.getStatusMessage() );
        
        URL rurl = res1.getLastRedirectUrlTo();
        
        resource.setLastRedirect( rurl );
        resource.setRedirects( res1.getRedirectUrls() );
        
        String contentType = res1.getHttpHeaders().getContentType();
        if( contentType==null ){
            // получение content исходя из имени-расширения файла
            URL ur = rurl;
            if( ur==null )ur = url;

            String ext = null;
            String file = null;
            String path = ur.getPath();
            if( path.contains("/") ){
                if( !path.endsWith("/") ){
                    int i=path.indexOf("/");
                    file = path.substring(i, path.length());
                }
            }else if( path.length()!=0 ){
                file = path;
            }

            if( file!=null && file.length()>0 && file.contains(".") && !file.endsWith(".") ){
                int i=file.indexOf(".");
                ext = file.substring(i, file.length());
            }

            if( ext!=null ){
                contentType = getMimeTypes().getMimeForExtension(ext);
            }
        }
        
        resource.setContentType( contentType );
        
        long contentLength = res1.getHttpHeaders().getContentLength();
        resource.setContentLength(contentLength);
        
        String acceptRanges = res1.getHttpHeaders().getAcceptRanges();
        if( statusCode==HttpStatusHelper.PARTIAL_CONTENT ||
            ( acceptRanges!=null && acceptRanges.equalsIgnoreCase("bytes") &&
            contentLength>=0
            )
            ){
            resource.setPartialContentSupport( true );
        }else{
            resource.setPartialContentSupport( false );
        }
        
        if (!httpStatusHelper.isSuccess(statusCode)) {
            if( !res1.isFinished() ){
                logInfo("send stop");
                stopSignal.set(true);
            }
        }
        if (contentType!=null && !(getMimeTypes().isHtml(contentType)) ) {
            if( !res1.isFinished() ){
                logInfo("send stop - contt");
                stopSignal.set(true);
            }
        }
    }
    
    /**
     * интерпретирвать не указанный content-type как html
     */
    private boolean nullContentTypeAsHtml = true;
    
    /**
     * ограничение на интерпретацию html, при условии если не указан content-type
     */
    private int maxNullContentTypeSize = 1024 * 1024 * 1;
    
    protected Resource inspectResource(final URL url){
        HttpClient client = getHttpClient();
        
        HttpRequest req = client.createRequest(url);
        if( req==null )return null;
        
        final Resource resource = new Resource();
        resource.setUrl( url );
        
        req.setAsync(true);
        req.setFollowRedirect(true);
        
        final HttpStatusHelper httpStatusHelper = new HttpStatusHelper();
        
        HttpResponse res = req.createResponse();
        res.setRedirectValidate(redirectValidate);
        
        HttpListenerAdapter redirectListener = new HttpListenerAdapter(){
            @Override
            protected void responseRedirect(HttpResponse.RedirectEvent event, HttpResponse response, URL from, URL to) {
                fireEvent(new RedirectEvent(SiteScanner.this, from, to));
            }
        };
        
        final AtomicBoolean stopSignal = new AtomicBoolean(false);
        final AtomicBoolean dlStateCatched = new AtomicBoolean(false);

        HttpListenerAdapter downloadStateListener = new HttpListenerAdapter(){
            @Override
            protected void responseStateChanged(
                HttpResponse.StateChangedEvent event, 
                HttpResponse res, 
                HttpResponse.State oldState, 
                HttpResponse.State newState
            ){
                if( newState==HttpResponse.State.Downloading ){
                    catchResponse(res, resource, url, httpStatusHelper, stopSignal);
                    dlStateCatched.set(true);
                }
            }
        };
        
        res.addListener(redirectListener);
        res.addListener(downloadStateListener);
        res.start();
        
        boolean fireInterupt = false;
        
        while( !res.isFinished() ){
            Thread.yield();
            if( stopSignal.get() ){
                res.stop();
            }
            
            if( Thread.interrupted() ){
                res.stop();
                fireInterupt = true;
            }
        }
        
        if( !dlStateCatched.get() ){
            catchResponse(res, resource, url, httpStatusHelper, stopSignal);
        }
        
        if( !res.isErrorsNotExists() ){
            preview(res);
            if( fireInterupt ){
                Thread.currentThread().interrupt();
            }
            return resource;
        }
        
        HttpHeaders hh = res.getHttpHeaders();
        long contentLength = hh==null ? -1 : hh.getContentLength();
        
        String ct = resource.getContentType();
        
        if( ct==null ){
            // content type не указан
            if( !nullContentTypeAsHtml ){
                // не интерпретировать null
                if( fireInterupt ){
                    Thread.currentThread().interrupt();
                }
                return resource;
            }
            
            if( contentLength < 0 || (contentLength > maxNullContentTypeSize) ){
                // ограничение на макс объем content-length
                if( fireInterupt ){
                    Thread.currentThread().interrupt();
                }
                return resource;
            }
        }else{
            if( !getMimeTypes().isHtml(ct) ){
                // content-type не соответ html
                if( fireInterupt ){
                    Thread.currentThread().interrupt();
                }
                return resource;
            }
        }
        
        try{
            HtmlPage htmlPage = new HtmlPage(res);
            resource.setHtmlPage( htmlPage );
        }catch( Throwable err ){
            logFine("html parse error {0} url = {1}", 
                    err.getMessage(),
                    url
            );
        }
        
        if( fireInterupt ){
            Thread.currentThread().interrupt();
        }
        return resource;
    }
//</editor-fold>
}
