/*
 * The MIT License
 *
 * Copyright 2015 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.text;


import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.text.UnionWriter;
import xyz.cofe.text.template.BasicTemplate;

/**
 * Поток вывода информации
 * @author nt.gocha@gmail.com
 */
public class Output
extends PrintWriter
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(Output.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(Output.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(Output.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(Output.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(Output.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(Output.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(Output.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    private final boolean autoFlush;
    
    private EndLineReWriter endlineRewriter;

    public Output( Writer out ) {
        super(out);
        
        endlineRewriter = createEndLineReWriter(out);
        this.out = endlineRewriter;
        
        autoFlush = false;
    }

    public Output( Writer out, boolean autoFlush ) {
        super(out, autoFlush);

        endlineRewriter = createEndLineReWriter(out);
        this.out = endlineRewriter;

        this.autoFlush = autoFlush;
    }

    public Output( OutputStream out ) {
        super(out);
        
        endlineRewriter = createEndLineReWriter( 
            new OutputStreamWriter(System.out, Charset.defaultCharset()) );
        
        this.out = endlineRewriter;
        
        autoFlush = false;
    }

    public Output( OutputStream out, boolean autoFlush ) {
        super(out, autoFlush);

        endlineRewriter = createEndLineReWriter( 
            new OutputStreamWriter(System.out, Charset.defaultCharset()) );
        
        this.out = endlineRewriter;

        this.autoFlush = autoFlush;
    }
    
    private EndLineReWriter createEndLineReWriter( Writer wr ){
        return new EndLineReWriter( wr ){
            @Override
            protected void writeEndLine() throws IOException {
                super.writeEndLine();
                endLineWrited = true;
            }
            
            private void wpref() throws IOException {
                if( endLineWrited ){
                    endLineWrited = false;
                    
                    String pref = getLinePrefix();
                    if( pref!=null ){
                        String[] prefLines = Text.splitNewLines(pref);
                        for( String prefLine : prefLines )
                            super.write(prefLine);
                    }
                }
            }

            @Override
            protected void writeChar(char c0) throws IOException {
                wpref();
                super.writeChar(c0);
            }

            @Override
            public void write(char[] cbuf, int off, int len) throws IOException {
                wpref();
                super.write(cbuf, off, len);
            }

            @Override
            public Writer append(char c) throws IOException {
                wpref();
                return super.append(c);
            }

            @Override
            public Writer append(CharSequence csq, int start, int end) throws IOException {
                wpref();
                return super.append(csq, start, end);
            }

            @Override
            public Writer append(CharSequence csq) throws IOException {
                wpref();
                return super.append(csq);
            }

            @Override
            public void write(String str, int off, int len) throws IOException {
                wpref();
                super.write(str, off, len);
            }

            @Override
            public void write(String str) throws IOException {
                wpref();
                super.write(str);
            }

            @Override
            public void write(char[] cbuf) throws IOException {
                wpref();
                super.write(cbuf);
            }

            @Override
            public void write(int c) throws IOException {
                wpref();
                super.write(c);
            }
        };
    }
    
    public String getEndl(){
        Object lo = this.lock;
        if( lo!=null ){
            synchronized(lo){
                return getEndl0();
            }
        }else{
            return getEndl0();
        }
    }
    
    private String getEndl0(){
        return endlineRewriter==null 
            ? System.getProperty("line.separator", "\n")
            : endlineRewriter.getEndl();
    }
    
    public void setEndl( String txt ){
        Object lo = this.lock;
        if( lo!=null ){
            synchronized(lo){
                setEndl0( txt );
            }
        }else{
            setEndl0( txt );
        }
    }
    
    private void setEndl0( String txt ){
        if( endlineRewriter!=null ){
            endlineRewriter.setEndl( txt==null ? EndLine.Default.get() : txt );
        }
    }

    public void reset( Writer output ){
        Object lo = this.lock;
        if( lo!=null ){
            synchronized(lo ){
                this.flush();

                endlineRewriter = createEndLineReWriter( 
                    output==null
                        ? new OutputStreamWriter(System.out, Charset.defaultCharset()) 
                        : output
                );
                
                this.out = endlineRewriter;
            }
        }else{
            this.flush();

            endlineRewriter = createEndLineReWriter( 
                output==null
                    ? new OutputStreamWriter(System.out, Charset.defaultCharset()) 
                    : output
            );
            
            this.out = endlineRewriter;
        }
    }

    public void reset(){
        reset(null);
    }

    public void appendWriter( Writer w ){
        if( w==null )throw new IllegalArgumentException( "w==null" );
        
        Object lo = this.lock;

        if( lo!=null ){
            synchronized( lo ){
                appendWriter0( w );
            }
        }else{
            appendWriter0(w);
        }
    }
    
    private void appendWriter0( Writer w ){
        if( this.out instanceof UnionWriter ){
            ((UnionWriter)this.out).getWriters().add(w);
        }else{
            UnionWriter uw = new UnionWriter( this.out, w );
            this.out = uw;
        }
    }
    
    public void removeWriter( Writer writer ){
        if( writer==null )throw new IllegalArgumentException( "writer==null" );
        
        Object lo = this.lock;

        if( lo!=null ){
            synchronized( lo ){
                if( this.out instanceof UnionWriter ){
                    UnionWriter uw = (UnionWriter)this.out;
                    uw.getWriters().remove(writer);
                }
            }
        }else{
            if( this.out instanceof UnionWriter ){
                UnionWriter uw = (UnionWriter)this.out;
                uw.getWriters().remove(writer);
            }
        }
    }

    public BasicTemplate.EasyTemplate template( String template ){
        if( template==null )throw new IllegalArgumentException( "template==null" );

        BasicTemplate.EasyTemplate tmpl = BasicTemplate.template(template);
        tmpl.output(this);
        tmpl.align();
        tmpl.outputFlushing();
//        tmpl.useJavaScript();
        return tmpl;
    }

    @Override
    public void write(char[] buf, int off, int len) {
        Object lo = this.lock;
        if( lo!=null ){
            synchronized(lo){
                write0(buf, off, len);
            }
        }else{
            write0(buf, off, len);
        }
    }
    
    private volatile boolean endLineWrited = true;

    public boolean isEndLineWrited() {
        return endLineWrited;
    }

    public void setEndLineWrited(boolean endLineWrited) {
        this.endLineWrited = endLineWrited;
    }

    private volatile Func0<String> linePrefix;
    
    public void setLinePrefixFn( Func0<String> fn ){
        linePrefix = fn;
    }
    
    public Func0<String> getLinePrefixFn(){
        return linePrefix;
    }
    
    public String getLinePrefix(){
        Func0<String> lPrefix = linePrefix;
        if( lPrefix==null )return null;
        
        try{
            String prefix = lPrefix.apply();
            return prefix;
        }catch( Throwable err ){
            logException(err);
        }
        
        return null;
    }
    
    public void setLinePrefix( final String prefix ){
        if( prefix==null ){
            linePrefix = null;
            return;
        }
        
        linePrefix = new Func0<String>() {
            @Override
            public String apply() {
                return prefix;
            }
        };
    }
    
    private void write0(char[] buf, int off, int len) {
//        Func0<String> lPrefix = linePrefix;
//        if( endLineWrited ){
//            endLineWrited = false;
//            if( lPrefix!=null ){
//                try{
//                    String prefix = lPrefix.apply();
//                    if( prefix!=null && prefix.length()>0 ){
//                        char[] cbuf = new char[prefix.length()];
//                        prefix.getChars(0, cbuf.length, cbuf, 0);
//                        super.write(cbuf, 0, cbuf.length);
//                    }
//                }catch( Throwable err ){
//                    logException(err);
//                }
//            }
//        }
        
        super.write(buf, off, len);
        
        if( autoFlush ){
            String s = new String(buf, off, len);
            if( Text.splitNewLines(s).length>1 ){
                flush();
            }
        }
    }
}
