/*
 * The MIT License
 *
 * Copyright 2015 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.text.template;


import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Func2;
import xyz.cofe.text.template.ast.Block;
import xyz.cofe.text.template.ast.Code;
import xyz.cofe.text.template.ast.Escape;
import xyz.cofe.text.template.ast.Text;

/**
 *
 * @author nt.gocha@gmail.com
 */
public class EvalVisitor<ResultType,EvalCode,EvalText> extends TemplateASTVisitor
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(EvalVisitor.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(EvalVisitor.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(EvalVisitor.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(EvalVisitor.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(EvalVisitor.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(EvalVisitor.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(EvalVisitor.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    public Func1<EvalText,String> evalText;
    public Func1<EvalCode,String> evalCode;
    public Func0<ResultType> initResult;
    public Func2<ResultType, ResultType, EvalText> appendText;
    public Func2<ResultType, ResultType, EvalCode> appendCode;

    public List<Func0> funs = new ArrayList<Func0>();
    public Map<Func0,Boolean> funAsCode = new LinkedHashMap<Func0, Boolean>();
    public boolean codeContext = false;
    public StringBuilder sbCode = new StringBuilder();

    public LinkedHashMap<String,String> escapeRewriteMap = new LinkedHashMap<String, String>();

    @Override
    public void visit(Text text) {
        if( codeContext ){
            sbCode.append(text.token.getMatchedText());
        }else{
            final String srcText = text.token.getMatchedText();
            Func0<EvalText> f = new Func0<EvalText>() {
                @Override
                public EvalText apply() {
                    return evalText.apply(srcText);
                }
            };

            funs.add(f);
            funAsCode.put(f, false);
        }
    }

    @Override
    public void visit(Escape escape) {
        if( codeContext ){
            sbCode.append(escape.token.getMatchedText());
        }else{
            String srcText = escape.token.getMatchedText();
            for( Map.Entry<String,String> en : escapeRewriteMap.entrySet() ){
                srcText = srcText.replace(en.getKey(), en.getValue());
            }
            final String rewritedText = srcText;

            Func0<EvalText> f = new Func0<EvalText>() {
                @Override
                public EvalText apply() {
                    return evalText.apply(rewritedText);
                }
            };
            funs.add(f);
            funAsCode.put(f, false);
        }
    }

    @Override
    public void enter(Block block) {
        if( codeContext ){
            sbCode.append(block.begin.getMatchedText());
        }else{
            final String srccode = block.begin.getMatchedText();
            Func0<EvalText> f = new Func0<EvalText>() {
                @Override
                public EvalText apply() {
                    return evalText.apply(srccode);
                }
            };
            funs.add(f);
            funAsCode.put(f, false);
        }
    }

    @Override
    public void exit(Block block) {
        if( codeContext ){
            sbCode.append(block.end.getMatchedText());
        }else{
            final String srccode = block.end.getMatchedText();
            Func0<EvalText> f = new Func0<EvalText>() {
                @Override
                public EvalText apply() {
                    return evalText.apply(srccode);
                }
            };
            funs.add(f);
            funAsCode.put(f, false);
        }
    }

    @Override
    public void enter(Code code) {
        codeContext = true;
        sbCode.setLength(0);
    }

    @Override
    public void exit(Code code) {
        codeContext = false;
        if( sbCode.length()>0 ){
            final String srccode = sbCode.toString();

            Func0<EvalCode> f = new Func0<EvalCode>() {
                @Override
                public EvalCode apply() {
                    return evalCode.apply(srccode);
                }
            };

            funs.add(f);
            funAsCode.put(f, true);
        }
    }
}
