/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.text.template;


import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Pointer;
import xyz.cofe.text.ast.AstNode;
import xyz.cofe.text.parser.Token;
import xyz.cofe.text.template.ast.Block;
import xyz.cofe.text.template.ast.BlockBody;
import xyz.cofe.text.template.ast.Code;
import xyz.cofe.text.template.ast.Escape;
import xyz.cofe.text.template.ast.Sequence;
import xyz.cofe.text.template.ast.Text;


/**
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class TemplateParser {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(TemplateParser.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(TemplateParser.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(TemplateParser.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(TemplateParser.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(TemplateParser.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(TemplateParser.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(TemplateParser.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    private final String[] codeBeginLexems;
    private final String[] escapeLexems;
    private final String[] blockBeginLexems;
    private final String[] blockEndLexems;

    public TemplateParser(){
        codeBeginLexems = new String[]{ "${" };
        escapeLexems = new String[]{ "\\${", "\\$", "\\\\" };
        blockBeginLexems = new String[]{ "{" };
        blockEndLexems = new String[]{ "}" };

        escapeRewriteMap.put("\\${", "${");
        escapeRewriteMap.put("\\$", "$");
        escapeRewriteMap.put("\\\\", "\\");
    }

    private LinkedHashMap<String,String> escapeRewriteMap = new LinkedHashMap<String, String>();

    //<editor-fold defaultstate="collapsed" desc="eval()">
    public <ResultType,
            EvalCode,
            EvalText
            >
            Func0<ResultType>
            eval( AstNode tree,
                final Func1<EvalText,String> evalText,
                final Func1<EvalCode,String> evalCode,
                final Func0<ResultType> initResult,
                final Func2<ResultType, ResultType, EvalText> appendText,
                final Func2<ResultType, ResultType, EvalCode> appendCode
            )
            {
                final List<Func0> funs = new ArrayList<Func0>();
                final Map<Func0,Boolean> funAsCode = new LinkedHashMap<Func0, Boolean>();
                
                EvalVisitor ev = new EvalVisitor();
                ev.appendCode = appendCode;
                ev.appendText = appendText;
                ev.initResult = initResult;
                ev.evalCode = evalCode;
                ev.evalText = evalText;
                ev.escapeRewriteMap = escapeRewriteMap;
                ev.funAsCode = funAsCode;
                ev.funs = funs;
                ev.codeContext = false;
                ev.sbCode = new StringBuilder();
                EvalVisitor.visit(tree, ev);
                
                return new Func0<ResultType>() {
                    @Override
                    public ResultType apply() {
                        ResultType res = initResult.apply();
                        for( Func0 f : funs ){
                            if( funAsCode.get(f) ){
                                res = appendCode.apply(res, (EvalCode)f.apply());
                            }else{
                                res = appendText.apply(res, (EvalText)f.apply());
                            }
                        }
                        return res;
                    }
                };
            }
            
            public Func0<String> evalAndPrint( AstNode tree, final Func1<String,String> evalCode ){
                if( tree==null )throw new IllegalArgumentException( "tree==null" );
                if( evalCode==null )throw new IllegalArgumentException( "evalCode==null" );
                
                return eval(
                    tree,
                    new Func1<String, String>() {
                        @Override
                        public String apply( String arg ) {
                            return arg;
                        }
                    },
                    evalCode,
                    new Func0<String>() {
                        @Override
                        public String apply() {
                            return "";
                        }
                    },
                    new Func2<String,String,String>(){
                        @Override
                        public String apply(String src,String append) {
                            return src + append;
                        }
                    },
                    new Func2<String,String,String>(){
                        @Override
                        public String apply(String src,String append) {
                            return src + append;
                        }
                    }
                );
            }

            public Func0<String> evalAndPrint( AstNode tree, final Func1<String,String> evalText, final Func1<String,String> evalCode ){
                if( tree==null )throw new IllegalArgumentException( "tree==null" );
                if( evalCode==null )throw new IllegalArgumentException( "evalCode==null" );
                if( evalText==null )throw new IllegalArgumentException( "evalText==null" );
                
                return eval(
                    tree,
                    evalText,
                    evalCode,
                    new Func0<String>() {
                        @Override
                        public String apply() {
                            return "";
                        }
                    },
                    new Func2<String,String,String>(){
                        @Override
                        public String apply(String src,String append) {
                            return src + append;
                        }
                    },
                    new Func2<String,String,String>(){
                        @Override
                        public String apply(String src,String append) {
                            return src + append;
                        }
                    }
                );
            }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="parse()">
    public AstNode parse( String source ){
        if( source==null )throw new IllegalArgumentException( "source==null" );

//        TemplateLexer lexer = new TemplateLexer();

        TemplateLexer lexer = new TemplateLexer(
            codeBeginLexems,
            escapeLexems,
            blockBeginLexems,
            blockEndLexems
        );

        List<Token> tokens = lexer.parse(source);
        Pointer<Token> ptr = new Pointer<Token>(tokens);

        AstNode v = start(ptr);
//        return v==null ? new AstNode() : v;
        return v;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="parse ast tree nodes">
    private static boolean in( String id, String ... arr ){
        return xyz.cofe.text.Text.in(id, arr);
    }

    private AstNode start( Pointer<Token> ptr ){
        Token t = ptr.lookup(0);
        if( t==null )return null;

        String id = t.getId();

        if( in(id, "codeBegin") ){
            AstNode c = code( ptr );
            if( c==null )return null;

            AstNode n = start(ptr);
            if( n==null )return c;

            return new Sequence(c, n);
        }

        if( in(id, "anyChar", "blockBegin", "blockEnd") ){
            Text text = new Text(t);
            ptr.move(1);

            AstNode v = start(ptr);
            if( v==null )return text;

            return new Sequence(text, v);
        }

        if( in(id, "escape" ) ){
            Escape esc = new Escape(t);
            ptr.move(1);

            AstNode v = start(ptr);
            if( v==null )return esc;

            return new Sequence(esc, v);
        }

        return null;
    }

    private AstNode code( Pointer<Token> ptr ){
        Token t = ptr.lookup(0);
        if( t==null )return null;

        String id = t.getId();
        if( !id.equals("codeBegin") )return null;

        ptr.push();
        ptr.move(1);

        BlockBody blockBody = blockBody(ptr);
        if( blockBody==null ){
            ptr.restore();
            return null;
        }

        Token tEnd = ptr.lookup(0);
        if( tEnd==null ){
            ptr.restore();
            return null;
        }

        if( !tEnd.getId().equals("blockEnd") ){
            ptr.restore();
            return null;
        }
        ptr.pop();
        ptr.move(1);

        return new Code(t,blockBody,tEnd);
    }

    private BlockBody blockBody( Pointer<Token> ptr ){
        Token t = ptr.lookup(0);
        if( t==null )return null;

        String id = t.getId();
        if( in( id, "anyChar" ) ){
            Text txt = new Text(t);
            ptr.move(1);

            AstNode v = blockBody(ptr);
            if( v==null )return new BlockBody( txt );

            return new BlockBody( new Sequence(txt, v) );
        }

        if( in( id, "escape" ) ){
            Escape esc = new Escape(t);
            ptr.move(1);

            AstNode v = blockBody(ptr);
            if( v==null )return new BlockBody( esc );

            return new BlockBody( new Sequence(esc, v) );
        }

        if( in( id, "blockBegin" ) ){
            AstNode v = block(ptr);
            if( v==null )return null;

            BlockBody bb = blockBody(ptr);
            if( bb==null )return new BlockBody( v );

            return new BlockBody( new Sequence(v, bb) );
        }

        return null;
    }

    private Block block( Pointer<Token> ptr ){
        Token t0 = ptr.lookup(0);
        if( t0==null )return null;

        String id0 = t0.getId();
        if( !in( id0, "blockBegin" ) )return null;
        ptr.push();
        ptr.move(1);

        AstNode body = blockBody(ptr);
        if( body==null ){
            ptr.restore();
            return null;
        }

        Token t1 = ptr.lookup(0);
        String id1 = t1.getId();
        if( !in(id1,"blockEnd") ){
            ptr.restore();
            return null;
        }

        ptr.pop();
        ptr.move(1);

        return new Block(t0, body, t1);
    }
//</editor-fold>
}
