package com.hmh.hamyeonham.hus.usagestats

import android.app.usage.UsageEvents
import android.app.usage.UsageStatsManager
import android.content.Context
import android.os.Build
import android.util.Log
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

internal class HMHUsageStatsDataSource(
    private val context: Context
) : HMHUsageStats {
    private val usageStatsManager: UsageStatsManager by lazy {
        context.getSystemService(Context.USAGE_STATS_SERVICE) as UsageStatsManager
    }

    @Throws(HMHUsageStatsException::class)
    override suspend fun getUsageStats(startTime: Long, endTime: Long): List<AppUsageInfo> {
        return withContext(Dispatchers.IO) {
            if (!hasUsageStatsPermission()) {
                Log.e("HMHUsageStatsDataSource", "UsageStats permission is not granted")
                return@withContext emptyList()
            }
            getUsageStatistics(startTime, endTime)
        }
    }

    override fun getForegroundAppPackageName(): String? {
        val endTime = System.currentTimeMillis()
        val startTime = endTime - 10000
        val usageEvents = usageStatsManager.queryEvents(startTime, endTime)
        var lastEvent: UsageEvents.Event? = null

        usageEvents?.let {
            while (it.hasNextEvent()) {
                val event = UsageEvents.Event()
                it.getNextEvent(event)
                if (event.eventType == UsageEvents.Event.ACTIVITY_RESUMED) {
                    lastEvent = event
                }
            }
        }

        return lastEvent?.packageName
    }

    private fun getUsageStatistics(startTime: Long, endTime: Long): List<AppUsageInfo> {
        val usageEvents = queryUsageEvents(usageStatsManager, startTime, endTime)
        val sameEvents = collectSameEvents(usageEvents)
        val usageMap = calculateUsageTime(sameEvents, endTime)

        return usageMap.values.toList()
    }

    private fun queryUsageEvents(
        usageStatsManager: UsageStatsManager,
        startTime: Long,
        endTime: Long
    ): List<UsageEvents.Event> {
        val usageEvents = mutableListOf<UsageEvents.Event>()
        val events = usageStatsManager.queryEvents(startTime, endTime)

        while (events.hasNextEvent()) {
            val currentEvent = UsageEvents.Event()
            events.getNextEvent(currentEvent)
            if (currentEvent.eventType == UsageEvents.Event.ACTIVITY_RESUMED ||
                currentEvent.eventType == UsageEvents.Event.ACTIVITY_PAUSED ||
                currentEvent.eventType == UsageEvents.Event.ACTIVITY_STOPPED
            ) {
                usageEvents.add(currentEvent)
            }
        }

        return usageEvents.sortedBy { it.timeStamp }
    }

    private fun collectSameEvents(
        usageEvents: List<UsageEvents.Event>
    ): Map<String, List<UsageEvents.Event>> {
        val sameEvents = mutableMapOf<String, MutableList<UsageEvents.Event>>()

        usageEvents.forEach { event ->
            val key = event.packageName
            sameEvents.getOrPut(key) { mutableListOf() }.add(event)
        }

        return sameEvents
    }

    private fun calculateUsageTime(
        sameEvents: Map<String, List<UsageEvents.Event>>,
        endTime: Long
    ): Map<String, AppUsageInfo> {
        val usageMap = mutableMapOf<String, AppUsageInfo>()

        sameEvents.forEach { (packageName, events) ->
            val appUsageInfo = usageMap.getOrPut(packageName) { AppUsageInfo(packageName) }
            var lastResumeTime = -1L
            var lastEventType = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                UsageEvents.Event.ACTIVITY_PAUSED
            } else {
                UsageEvents.Event.MOVE_TO_BACKGROUND
            }

            events.forEach { event ->
                when (event.eventType) {
                    UsageEvents.Event.ACTIVITY_RESUMED -> {
                        lastResumeTime = event.timeStamp
                        lastEventType = event.eventType
                    }

                    UsageEvents.Event.ACTIVITY_PAUSED, UsageEvents.Event.ACTIVITY_STOPPED -> {
                        appUsageInfo.timeInForeground += event.timeStamp - lastResumeTime
                        lastResumeTime = event.timeStamp
                        lastEventType = event.eventType
                    }
                }
            }

            if (lastEventType == UsageEvents.Event.ACTIVITY_RESUMED) {
                appUsageInfo.timeInForeground += endTime - lastResumeTime
            }
        }

        return usageMap
    }

    private fun hasUsageStatsPermission(): Boolean {
        val appOpsManager =
            context.getSystemService(Context.APP_OPS_SERVICE) as android.app.AppOpsManager
        val mode = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            appOpsManager.unsafeCheckOpNoThrow(
                android.app.AppOpsManager.OPSTR_GET_USAGE_STATS,
                android.os.Process.myUid(), context.packageName
            )
        } else {
            appOpsManager.checkOpNoThrow(
                android.app.AppOpsManager.OPSTR_GET_USAGE_STATS,
                android.os.Process.myUid(), context.packageName
            )
        }
        return mode == android.app.AppOpsManager.MODE_ALLOWED
    }
}