package xyz.ronella.trivial.handy.impl;

import xyz.ronella.trivial.functional.NoOperation;
import xyz.ronella.trivial.handy.AbstractBooleanSupplierKeyedMapLogic;

import java.util.Map;
import java.util.Optional;
import java.util.function.BooleanSupplier;
import java.util.function.Supplier;

/**
 * A convenience class for creating a map as factory.
 *
 * @param <TYPE_OUTPUT> The type of the output that the factory will create.
 *
 * @author Ron Webb
 *
 * @since 2.4.0
 */
public class BooleanSupplierKeyedMapGenerator<TYPE_OUTPUT> extends AbstractBooleanSupplierKeyedMapLogic<Supplier<TYPE_OUTPUT>, TYPE_OUTPUT> {

    /**
     * Creates an instance of BooleanSupplierKeyedMapFactory
     *
     * @param map          An external map that will be used as storage of factory.
     * @param defaultLogic The default create logic if the key used was not in the map.
     * @param logics       An arrays of create logic mapped to key.
     */
    @SafeVarargs
    public BooleanSupplierKeyedMapGenerator(final Map<BooleanSupplier, Supplier<TYPE_OUTPUT>> map,
                                            final Supplier<TYPE_OUTPUT> defaultLogic,
                                            final Map.Entry<BooleanSupplier, Supplier<TYPE_OUTPUT>> ... logics) {
        super(map, defaultLogic, logics);
    }

    /**
     * Creates an instance of BooleanSupplierKeyedMapFactory
     *
     * @param defaultLogic The default create logic if the key used was not in the map.
     * @param logics       An arrays of create logic mapped to key.
     */
    @SafeVarargs
    public BooleanSupplierKeyedMapGenerator(final Supplier<TYPE_OUTPUT> defaultLogic,
                                            final Map.Entry<BooleanSupplier, Supplier<TYPE_OUTPUT>>... logics) {
        this(null, defaultLogic, logics);
    }

    /**
     * Creates an instance of BooleanSupplierKeyedMapFactory
     *
     * @param logics An arrays of create logic mapped to key.
     */
    @SafeVarargs
    public BooleanSupplierKeyedMapGenerator(final Map.Entry<BooleanSupplier, Supplier<TYPE_OUTPUT>>... logics) {
        this(null, logics);
    }

    /**
     * Creates an instance of BooleanSupplierKeyedMapFactory
     *
     * @param map An external map that already contains logics.
     */
    public BooleanSupplierKeyedMapGenerator(final Map<BooleanSupplier, Supplier<TYPE_OUTPUT>> map) {
        this(map, null);
    }

    @Override
    protected Supplier<TYPE_OUTPUT> handleDefaultLogicConstructorArgument(final Supplier<TYPE_OUTPUT> defaultLogic) {
        return Optional.ofNullable(defaultLogic).orElse(NoOperation.supplier());
    }

    /**
     * Executes the logic of the first truth key.
     *
     * @return The typed object generated by the logic of the first truth key.
     */
    @Override
    public Optional<TYPE_OUTPUT> get() {

        var logic = getDefaultLogic();
        for (final var key : internalMap.keySet()) {
            if (key.getAsBoolean()) {
                logic = internalMap.get(key);
                break;
            }
        }

        return Optional.ofNullable(logic.get());
    }
}