package xyz.ronella.trivial.handy.impl;

import xyz.ronella.trivial.functional.NoOperation;
import xyz.ronella.trivial.handy.AbstractStringKeyedMapLogic;

import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * A convenience class for creating a map as factory.
 *
 * @param <TYPE_OUTPUT> The type of the output that the factory will create.
 *
 * @author Ron Webb
 *
 * @since 2.4.0
 */
public class StringKeyedMapGenerator<TYPE_OUTPUT> extends AbstractStringKeyedMapLogic<Supplier<TYPE_OUTPUT>, TYPE_OUTPUT> {

    /**
     * Creates an instance of StringKeyedMapFactory
     *
     * @param map An external map that will be used as storage of factory.
     * @param defaultLogic The default create logic if the key used was not in the map.
     * @param logics An arrays of create logic mapped to key.
     */
    @SafeVarargs
    public StringKeyedMapGenerator(final Map<String, Supplier<TYPE_OUTPUT>> map,
                                   final Supplier<TYPE_OUTPUT> defaultLogic,
                                   final Map.Entry<String, Supplier<TYPE_OUTPUT>> ... logics) {
        super(map, defaultLogic, logics);
    }

    /**
     * Creates an instance of StringKeyedMapFactory
     *
     * @param defaultLogic The default create logic if the key used was not in the map.
     * @param logics An arrays of create logic mapped to key.
     */
    @SafeVarargs
    public StringKeyedMapGenerator(final Supplier<TYPE_OUTPUT> defaultLogic,
                                   final Map.Entry<String, Supplier<TYPE_OUTPUT>> ... logics) {
        this(null, defaultLogic, logics);
    }

    /**
     * Creates an instance of StringKeyedMapFactory
     *
     * @param logics An arrays of create logic mapped to key.
     */
    @SafeVarargs
    public StringKeyedMapGenerator(final Map.Entry<String, Supplier<TYPE_OUTPUT>> ... logics) {
        this(null, logics);
    }

    /**
     * Creates an instance of StringKeyedMapFactory
     *
     * @param map An external map that already stores factories.
     */
    public StringKeyedMapGenerator(final Map<String, Supplier<TYPE_OUTPUT>> map) {
        this(map, null);
    }

    @Override
    protected Supplier<TYPE_OUTPUT> handleDefaultLogicConstructorArgument(final Supplier<TYPE_OUTPUT> defaultLogic) {
        return Optional.ofNullable(defaultLogic).orElse(NoOperation.supplier());
    }

    /**
     * Executes the logic of particular key.
     *
     * @param key The key of the create logic otherwise it will use the default create logic.
     *
     * @return The typed object generated by the create logic of the particular key.
     */
    @Override
    public Optional<TYPE_OUTPUT> get(final String key) {
        return Optional.ofNullable(internalMap.getOrDefault(key, defaultLogic).get());
    }

}
