/*
 * Copyright (c) 2021 xiamang.xyz, Inc. All Rights Reserved.
 */

package xyz.xiamang.holiday.service;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import xyz.xiamang.holiday.config.JsonConfigLoader;
import xyz.xiamang.holiday.dto.DayDetail;
import xyz.xiamang.holiday.dto.DayType;
import xyz.xiamang.holiday.exception.UnSupportException;
import xyz.xiamang.holiday.util.ResourceUtils;
import xyz.xiamang.holiday.util.StringUtils;

/**
 * 节假日判断API
 * @author ysq
 */
public class HolidayService {

    /**
     * 所有日期详情 key:yyyyMMdd value:{@link DayDetail}
     */
    private static final Map<String, DayDetail> DAY_DETAIL_CACHE = new LinkedHashMap<>(4096);
    /**
     * 节假日配置项缓存 key:yyyyMMdd value:{@link DayDetail}
     */
    private static final Map<String, DayDetail> HOLIDAY_CONFIG = new HashMap<>(512);

    private static final String[] WEEK_NAMES = { "周日", "周一", "周二", "周三", "周四", "周五", "周六" };

    private static final Integer[] WEEK_INDEX = { 7, 1, 2, 3, 4, 5, 6 };

    private static String MAX_YEAR = "2021";
    private static String MAX_DAY = "2024-12-31";

    private static String CONFIG_DEFAULT_DIR = "_CONFIG_HOlIDAY";
    private static String CONFIG_CUSTOM_DIR = "";

    private static final Logger logger = LoggerFactory.getLogger(HolidayService.class);

    public HolidayService() {
        super();
        init();
    }

    /**
     * TODO: 加载仍有问题，暂不支持
     * @param customConfigPath 自定义配置文件目录
     * 目录请放在项目的资源文件夹中，Maven 项目请放置在 src/main/resources
     */
    public HolidayService(String customConfigPath) {
        super();
        CONFIG_CUSTOM_DIR = customConfigPath;
        init();
    }

    private void init() {
        loadFromJsonConfig();
    }

    private List<File> loadConfig() {
        try {
            //            ResourceUtils.getResourcesFile2(this.getClass(), "holiday");
            //            ResourceUtils.getResourcesFile2(this.getClass(), "./holiday");
            //            ResourceUtils.getResourcesFile2(this.getClass(), "holiday/");
            //            ResourceUtils.getResourcesFile2(this.getClass(), "./holiday/");
            //ResourceUtils.getResourcesFile2(this.getClass(), "");
            System.out.println("#######" + CONFIG_CUSTOM_DIR);
            String configPath = StringUtils.isBlank(CONFIG_CUSTOM_DIR) ? CONFIG_DEFAULT_DIR : CONFIG_CUSTOM_DIR;
            List<File> fs = null;
            if (StringUtils.isBlank(CONFIG_CUSTOM_DIR)) {
                fs = ResourceUtils.readResourceDir(this.getClass(), CONFIG_DEFAULT_DIR);
            } else {
                fs = ResourceUtils.readCustomResourcesFile(this.getClass().getClassLoader(), CONFIG_CUSTOM_DIR);
            }

            //按文件名升序排列
            Collections.sort(fs, new Comparator<File>() {
                @Override
                public int compare(File f1, File f2) {
                    return f1.getName().compareTo(f2.getName());
                }
            });
            return fs;
        } catch (Exception e) {
            logger.error("获取配置文件失败! {}", e);
            e.printStackTrace();
            throw new UnSupportException("获取配置文件失败! load holiday config failed!");
        }
    }

    private void loadFromJsonConfig() {
        long start = System.currentTimeMillis();

        List<File> fs = loadConfig();
        List<String> fileNames = new ArrayList<>();
        for (File f : fs) {
            logger.debug("{}", f.getName());
            if (!f.getName().contains("json")) {
                logger.warn("{} 不包含节假日配置忽略", f.getName());
                continue;
            }
            fileNames.add(f.getName());
            MAX_YEAR = f.getName().substring(0, 4);
            @SuppressWarnings("rawtypes")
            Map map = JsonConfigLoader.load(f, Map.class);
            //logger.debug("{}", map);
            parseConfig(map);
        }
        long e1 = System.currentTimeMillis();
        logger.debug("cost1 {}", e1 - start);
        initAllDay();
        long end = System.currentTimeMillis();
        System.out.println("##### holiday config files " + fileNames);

        logger.info("#### 节假日配置初始化完成 耗时：{}ms，最大日期：{}，共{}个日期，配置文件：{}",
                end - start, MAX_DAY, DAY_DETAIL_CACHE.size(), fileNames);
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    private void parseConfig(Map map) {
        Map<String, Map> m2 = (Map) map.get("holiday");
        m2.values().forEach(m -> {
            DayDetail day = new DayDetail();
            String d = (String) m.get("date");
            day.setDay(d);
            day.setDayAlias(d.replace("-", ""));
            day.setName((String) m.get("name"));
            day.setWage((Integer) m.get("wage"));
            Boolean f = (Boolean) m.get("holiday");
            day.setHoliday(f);
            day.setDayType(f ? DayType.HOLIDAY : DayType.TRADEDAY);
            day.setHolidayName((String) m.get("target"));
            HOLIDAY_CONFIG.put(day.getDayAlias(), day);
        });
    }

    private void initAllDay() {
        Calendar calendar = Calendar.getInstance();
        int maxYear = Integer.valueOf(MAX_YEAR); //calendar.get(Calendar.YEAR) + 1;//明年
        calendar.set(2013, 0, 1);//from 2013年1月1日
        int y = 0;
        DayDetail maxDay = null;
        while (y <= maxYear) {
            maxDay = queryDayDetailAndPutCache(calendar);
            calendar.add(Calendar.DAY_OF_MONTH, 1);
            y = calendar.get(Calendar.YEAR);
        }
        MAX_DAY = maxDay.getDayAlias();
    }

    /**
     * 是否是节假日，默认为当天
     * @return boolean
     */
    public boolean isHoliday() {
        return isHoliday(new Date());
    }

    /**
     * 是否是节假日 注：放假日期都算，包含周六日及春节等法定假日，但由于节假日的调休补班不算
     * @param day 日期
     * @return 是否节假日
     */
    public boolean isHoliday(Date day) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
        String key = sdf.format(day);
        return isHoliday(key);
    }

    /**
     * 是否是节假日, 注 放假日期都算，包含周六日及春节等法定假日，但由于节假日的调休补班不算
     * @param day 日期 格式 yyyy-mm-dd或yyyymmdd 示例2021-09-09或20210909
     * @return 是否节假日
     */
    public boolean isHoliday(String day) {
        if (StringUtils.isBlank(day)
                || !day.startsWith("2")
                || (day.length() != 8 && day.length() != 10)) {
            String msg = "日期格式不合法，支持的格式：yyyy-mm-dd或yyyymmdd，示例2021-09-09或20210909，实际参数:" + day;
            throw new UnSupportException(msg);
        }

        DayDetail detail = queryDayDetail(day);
        if (detail == null) {
            String msg = "仅支持 2013-01-01至" + MAX_DAY + " 期间节假日查询，请更新至最新版本 " +
                    "https://mvnrepository.com/artifact/xyz.xiamang/holiday ，" +
                    "或在src/main/resources/holiday/文件夹中进行配置";
            throw new UnSupportException(msg);
        }
        return detail.isHoliday();
    }

    /**
     * 查询日期详情 是否节假日、补班、周几等
     * @param day 日期 格式 yyyy-mm-dd或yyyymmdd 示例2021-09-18或20210918
     * @return 日期详情
     */
    public DayDetail queryDayDetail(String day) {
        String key = dayKey(day);
        return DAY_DETAIL_CACHE.get(key);
    }

    /**
     * 根据日期类型批量查询
     * @param type 日期类型
     * @param startDay 开始日期（包含）格式yyyyMMdd或yyyy-MM-dd
     * @param endDay 结束日期（包含）格式yyyyMMdd或yyyy-MM-dd
     * @return 日期详情
     */
    public List<DayDetail> queryDayByType(DayType type, String startDay, String endDay) {
        String startKey = dayKey(startDay);
        String endKey = dayKey(endDay);
        List<DayDetail> result = DAY_DETAIL_CACHE.entrySet()
                .stream().filter(e -> {
                    Boolean f = false;
                    String key = e.getKey();
                    DayDetail value = e.getValue();
                    if (type.equals(DayType.ALL_WORKDAY)) {
                        f = DayType.WORKDAY.equals(value.getDayType())
                                || DayType.TRADEDAY.equals(value.getDayType());
                    } else if (type.equals(DayType.ALL_NO_WORKDAY)) {
                        f = DayType.HOLIDAY.equals(value.getDayType()) ||
                                DayType.WEEKEND.equals(value.getDayType());
                    } else {
                        f = type.equals(value.getDayType());
                    }
                    return f && key.compareTo(startKey) >= 0 && key.compareTo(endKey) <= 0;
                }).map(Entry::getValue)
                .collect(Collectors.toList());
        return result;
    }

    /**
     * 按类型查询最近一个日期
     * @param type 日期类型
     * @param startDay 开始日期（包含） 格式yyyyMMdd 或 yyyy-MM-dd
     * @return 日期详情
     */
    public DayDetail nextDayByType(DayType type, String startDay) {
        String startKey = dayKey(startDay);
        for (Entry<String, DayDetail> entry : DAY_DETAIL_CACHE.entrySet()) {
            String key = entry.getKey();
            if (key.compareTo(startKey) > 0) {
                Boolean f = false;
                DayDetail value = entry.getValue();
                if (type.equals(DayType.ALL_WORKDAY)) {
                    f = DayType.WORKDAY.equals(value.getDayType())
                            || DayType.TRADEDAY.equals(value.getDayType());
                } else if (type.equals(DayType.ALL_NO_WORKDAY)) {
                    f = DayType.HOLIDAY.equals(value.getDayType()) ||
                            DayType.WEEKEND.equals(value.getDayType());
                } else {
                    f = type.equals(value.getDayType());
                }
                if (f) {
                    return value;
                }
            }
        }
        return null;
    }

    public DayDetail queryDayDetailAndPutCache(Calendar calendar) {
        String key = dayKey(calendar);
        int w = calendar.get(Calendar.DAY_OF_WEEK);
        DayDetail detail = HOLIDAY_CONFIG.get(key);
        if (detail == null) {
            detail = new DayDetail();
            detail.setWage(1);
            detail.setDay(key.substring(0, 4) + "-" + key.substring(4, 6) + "-" + key.substring(6));
            detail.setDayAlias(key);
            //英文中一周从周日开始 周日=1 周六=7
            if (1 == w || 7 == w) {
                detail.setDayType(DayType.WEEKEND);
                detail.setHoliday(true);
            } else {
                detail.setDayType(DayType.WORKDAY);
                detail.setHoliday(false);
            }
            detail.setName(key);
            detail.setName(WEEK_NAMES[w - 1]);
        }
        detail.setDayOfWeek(WEEK_INDEX[w - 1]);
        DAY_DETAIL_CACHE.put(key, detail);
        return detail;
    }

    private String dayKey(String day) {
        return day.replace("-", "");
    }

    private String dayKey(Calendar calendar) {
        int y = calendar.get(Calendar.YEAR);
        int m = calendar.get(Calendar.MONTH) + 1;
        int d = calendar.get(Calendar.DAY_OF_MONTH);
        return y + (m < 10 ? "0" + m : "" + m)
                + (d < 10 ? "0" + d : "" + d);
    }

    //    public static void main(String[] args) {
    //        HolidayService holidayService = new HolidayService();
    //        //holidayService.init();
    //    }

}
